/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef _SO_CHILD_LIST_
#define _SO_CHILD_LIST_

#include <Inventor/SoLists.h>
#include <Inventor/threads/SbThreadMutex.h>
#include <Inventor/sensors/SoDataSensor.h>


//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoChildList
//
//  This class maintains a list of child nodes for any node. It allows
//  children to be added, removed, replaced, and accessed. The SoGroup
//  class has an instance of this class and provides public methods
//  for dealing with children. Other classes can also have an instance
//  of this class and may choose to publicize whatever methods they
//  wish for dealing with them. The SoNode::getChildren() method
//  returns the child list for any node class.
//
//  SoChildList automatically maintains the auditor list for parent
//  nodes. That is, when a child is added to the list, the parent node
//  is automatically added as a parent auditor of the child. To make
//  this possible, the parent node must be passed in to the
//  constructors of the SoChildList class.
//
//////////////////////////////////////////////////////////////////////////////

class SoAction;
class AuditorsContainer;
/**
 * [OIV-WRAPPER-CLASS NO_WRAP]
 */
SoEXTENDER class INVENTOR_API SoChildList : public SoNodeList
{
 public:
  // Constructors and destructor.
  SoChildList(SoNode *parentNode);
  SoChildList(SoNode *parentNode, int size);
  SoChildList(SoNode *parentNode, const SoChildList &l);
  virtual ~SoChildList();

  // All of these override the standard SoBaseList methods so they
  // can maintain auditors correctly.
  void append(SoNode * child);
  void insert(SoNode *child, int addBefore);
  void remove(int which);
  void truncate(int start);
  void copy(const SoChildList &l);
  void set(int i, SoNode *child);

  // Traverses all children to apply action. Stops if action's
  // termination condition is reached
  void traverse(SoAction *action);

  // Traverses just one child
  void traverse(SoAction *action, int childIndex)
  { traverse(action, childIndex, childIndex); }

  // Traverses all children between two indices, inclusive. Stops if
  // action's termination condition is reached.
  void traverse(SoAction *action, int firstChild, int lastChild);

  // Traverses just one node
  void traverse(SoAction *action, SoNode *node);

  // multi-instance traversals
  // Traverses all children to apply action. Stops if action's
  // termination condition is reached for one instance
  void traverseMI(SoAction *action, int instanceIndex);

  // Traverses just one child with multi instancing
  void traverseMI( SoAction *action, int childIndex, int numInstances )
  { traverseMI(action, childIndex, childIndex, numInstances); }
  
  // Traverses all children between two indices, inclusive. Stops if
  // action's termination condition is reached.
  void traverseMI(SoAction *action, int firstChild, int lastChild, int instanceIndex );

  // Traverses just one node
  void traverseMI(SoAction *action, SoNode *node, int instanceIndex );

 SoINTERNAL public:
  // SoPath calls these to be notified of changes in scene graph
  // topology:
  // moved to the .cxx
  void addPathAuditor(SoPath *p);
  void removePathAuditor(SoPath *p);
  void removeAllPathAuditor();
  static void exitClass();
 
private:

  // Inner clas that describes current notification info details.
  class INVENTOR_API SoChildNotificationInfo 
  {
  public:
    // Constructor.
    SoChildNotificationInfo()
    {
      reset();
    }
    // reset current notification details.
    inline void reset()
    {
      m_changedChild = NULL;
      m_changedIndex = -1;
      m_changeType = SoDataSensor::UNSPECIFIED;
    }
    // set current notification infos.
    inline void set(SoNode* changedChild, const int changedIndex, const SoDataSensor::ChangeType changeType)
    {
      m_changedChild = changedChild;
      m_changedIndex = changedIndex;
      m_changeType = changeType;
    }
    // Basic accessors
    inline SoNode* getChangedChild() const 
    { return m_changedChild; }
    int getChangedIndex() const
    { return m_changedIndex; }
    SoDataSensor::ChangeType getChangeType() const
    { return m_changeType;}

  private:
    // pointer on the changed child
    SoNode* m_changedChild;
    // index of the changed child
    int m_changedIndex;
    // current type of changed
    SoDataSensor::ChangeType m_changeType;
  };

  SoNode* parent;
  // This is a container of SoPath* and not a PathList because PathList ref()s the
  // paths it contains, and that screws up Path reference counting.
  AuditorsContainer *m_auditors;

  // Returns details of change (only valid during notification)
  const SoChildNotificationInfo& getChildNotificationInfo() const
  { return m_childNotificationInfo; }

  // keep track of last change during notification process
  SoChildNotificationInfo m_childNotificationInfo;

//! @cond Doxygen_Suppress
  friend class SoNode;
  friend class SoDataSensor;
//! @endcond
};

#endif /* _SO_CHILD_LIST_ */


