/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Nick Thompson (MMM yyyy)
** Modified by : David Mott (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/

#ifndef SO_PATH_LIST_H
#define SO_PATH_LIST_H

#include <Inventor/lists/SoBaseList.h>

class SoPath;

#include <Inventor/lists/SoCompactPathList.h>

#ifdef _WIN32
#pragma warning(push)
#pragma warning(disable:4251)
#endif

/**
 * Maintains a list of pointers to paths.
 *
 * @ingroup General
 *
 * @DESCRIPTION
 *   This subclass of SoBaseList holds lists of pointers to SoPaths. It updates
 *   reference counts to paths in the list whenever adding or removing pointers.
 *
 * @SEE_ALSO
 *    SoPath
 *
 *
 * [OIV-WRAPPER-CUSTOM-CODE]
 */
class INVENTOR_API SoPathList : public SoBaseList
{
 public:
  /**
   * Constructor.
   */
  SoPathList() : SoBaseList(), m_isSorted(false) {}

  /**
   * Constructor that pre-allocates storage for @B size @b pointers.
   */
  SoPathList(int size) : SoBaseList(size), m_isSorted(false) {}

  /**
   * Constructor that copies the contents of another list.
   */
  SoPathList(const SoPathList &l) : SoBaseList(l.getLength()),
    m_compactPathList(l.m_compactPathList),
    m_isSorted(l.m_isSorted)
    { copy(l); }

  /**
   * Destructor.
   */
  virtual ~SoPathList() {}

  /**
  * Remove a path from the list and return TRUE if removed
  */
  virtual SbBool removePath( const SoPath &path );

  /**
   * Adds a path to the end of the list.
   */
  virtual void append(SoPath* ptr)
  { SoBaseList::append((SoBase*)ptr); }

  /** @copydoc SoBaseList::operator[]() */
  SoPath* operator [](int i) const
  { return ((SoPath *) ((*(const SoBaseList *) this) [i])); }

  /**
   * Copies a list, keeping all reference counts correct.
   */
  SoPathList & operator =(const SoPathList &l)
  { SoBaseList::copy(l) ; return *this; }

  /**
   * @brief Equal operator.
   * @param l The list to compare with.
   * @return true if the lists are equal, false otherwise.
   */
  bool operator==(const SoPathList &l) const;

  /**
   * Returns the index of the matching path in the list, or -1 if not found.
   */
  virtual int findPath(const SoPath &path);

  /**
   * Sorts list in place based on (1) increasing address of head node, then (2)
   * increasing indices of children.
   */
  virtual void sort();

  /**
   * Given a sorted list, removes any path that (1) is a duplicate, or (2) goes
   * through a node that is the tail of another path.
   */
  virtual void uniquify();

  // Comparison method for path list sorting.
  static int comparePaths(const void *p1Ptr, const void *p2Ptr);

 SoINTERNAL public:
  // An optional compact representation of the path list.
  SoRef<SoCompactPathList> m_compactPathList;

  bool isSorted() { return m_isSorted; }
  void setSorted(bool isSorted) { m_isSorted = isSorted; }
 private:
  // sorted status
  bool m_isSorted;
};

/**
 * Writes the list to the specified output stream.
 */
std::ostream& operator << (std::ostream& os, const SoPathList& pathList);

#ifdef _WIN32
#pragma warning(pop)
#endif

#endif
