/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (SEP 2012)
**=======================================================================*/

#ifndef  _SO_LICENSES_INFO_
#define  _SO_LICENSES_INFO_

#include <Inventor/SbPImpl.h>
#include <Inventor/sys/port.h>
#include <Inventor/SbBase.h>
#include <Inventor/SbString.h>
#include <Inventor/lock/SoLicensedProduct.h>
#include <Inventor/STL/map>
#include <Inventor/STL/vector>

/** @memberof SoLicensesInfo */
typedef void SoLicensesCB(SoLicensedProduct::SbProduct product);

SO_PIMPL_BASE_PUBLIC_DECLARATION(SoLicensesInfo)

/**
 * @VSGEXT Class managing every licensed product.
 *
 * @ingroup lock
 *
 * @DESCRIPTION
 *   SoLicensesInfo manages licensed products.
 *   SoLicensesInfo::setCallback(SoLicensesCB*) deactivates the default message box to call the 
 *   specified callback.
 *
 * @B Version Number: @b
 *   You can query the Open Inventor version number as a string using the #getVersion() method or as a
 *   floating point value using the #getLicensingVersionNumber() method.  \if_cpp You can find other
 *   integer and string constants in SoVersion.h.  For example @B SO_INVENTOR_VERSION @b can be
 *   used with preprocessor directives to isolate version specific code.  It encodes the
 *   major, minor and patch numbers as an integer. For example "9400" for version 9.4. \endif
 *
 * [OIV-WRAPPER-CLASS STANDALONE,AUTO_PROPERTY]
 */
class INVENTORBASE_API SoLicensesInfo
{
  SO_PIMPL_BASE_PUBLIC_HEADER(SoLicensesInfo)

public:

  /**
   * Returns SoLicensesInfo singleton.
   */
  static SoLicensesInfo& getInstance();

  /**
   *  Gets number of licensed products.
   */
  unsigned int getProductCount() const;
  
  /**
   * Gets the licensed product corresponding to the given index.
   * If the given index is greater than the value returned
   * by #getProductCount(), the method returns NULL.
   */
  const SoLicensedProduct* getLicensedProductByIndex( unsigned int index ) const;

  /**
   * Gets the licensed product corresponding to the given SoLicensedProduct::SbProduct.
   * If the given product is invalid, the method returns NULL.
   */
  const SoLicensedProduct* getLicensedProduct( SoLicensedProduct::SbProduct prod ) const;

  /**
   *  Sets the callback to call when a license is expired. 
   * If a callback is registered, the default message box will not be displayed.
   */
  void registerExpirationCallback( SoLicensesCB* cb );
  
  /**
   *  Sets the callback to call when a license date of expiration is lower than nbDayBeforeExpiration. 
   *  If a callback is registered, the default message box will not be displayed.
   */
  void registerWarningCallback( int nbDaysBeforeExpiration, SoLicensesCB* cb );

  /**
   * Returns a character string identifying the version of the Open Inventor library
   * in use.
   * For example 2023.2.7 for the second version of OIV in 2023 and the 7th patch version.
   *
   * For a floating point version number see #getLicensingVersionNumber.
   * \if_cpp For integer and string constants, see SoVersion.h. \endif
   */
  static const char* getVersion();

  /**
   * Returns the current version number as a float for license checking.
   * The integer part of the float is the year of the published version
   * and the floating part is 1 or 2 for the version first or second version of the year.
   * For example 2023.2.7 will be returned as 2023.2, which corresponds to the 2nd version of OpenInventor in 2023.
   * See method #check for usage of this number.
   *
   * If you need a descriptive string see #getVersion.
   * \if_cpp If you just need the version number as an integer or string, see the macros in SoVersion.h. \endif
   */
  static float getLicensingVersionNumber();

  /**
   * Query if Open Inventor or an extension module is licensed on this machine.
   *
   * @return < 0 If product or version is NOT licensed on this machine. @BR
   *         >= 0 If product and version are licensed and can be used.
   *
   * Note: If productName is "ScaleViz" and ScaleViz is licensed on this machine.
   *       Then the return value is the number of render nodes allowed.
   *
   * @param productName : Product to be licensed (no blanks), e.g. OpenInventor or VolumeVizLDM.
   * @param version     : Version to be licensed (see #getLicensingVersionNumber).
   * @param licenseString : (optional) Embedded application password string, if any.
   * @param printWarning : If true, error and warning messages (if any) will be displayed to user.
   * @param errReport    : If not NULL, will be set to point to the text of the error
   *                       or warning messages (if any). @BR Note! This address is an internal static
   *                       buffer and must not be freed by the application.
   *
   * FlexNet Notes:
   * - Before checking for a FlexNet license token, SoLicensesInfo::check will automatically append
   *   "SDK" to the productName when you need an SDK license to run on the current machine.
   * [OIVNET-WRAPPER-ARG IN,IN,IN,IN,OUT]
   */
  static int check( const char* productName, float version, const char* licenseString = NULL,
                    SbBool printWarning = TRUE, char** errReport = NULL );
private:
  SoLicensesInfo();
  virtual ~SoLicensesInfo();
};

#endif /* _SO_LICENSES_INFO */
