/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : R. ALBOU (Feb 1997)
**=======================================================================*/

                                                             /* include files */
#ifndef  _SO_CLIP_PLANE_MANIP_
#define  _SO_CLIP_PLANE_MANIP_

#include <Inventor/draggers/SoDragger.h>
#include <Inventor/nodes/SoClipPlane.h>
#include <Inventor/sensors/SoFieldSensor.h>
#include <Inventor/fields/SoSFVec3f.h>

/*----------------------------------------------------------------------------*/

/**
 * @VSGEXT Manipulator to transform a clipping plane.
 * 
 * @ingroup manips
 * 
 * @DESCRIPTION
 *   This class defines a manipulator to transform a clipping plane. An SoJackDragger
 *   allows the user to rotate and translate the clipping plane.
 *
 *   This node clips all subsequent shapes in the scene graph to the half-space defined
 *   by the plane field. The half-space that is rendered is on the side of the plane in
 *   the direction of the plane normal. The half-space that is clipped away is on the 
 *   side of the plane opposite the direction of the plane normal. For example, if the 
 *   plane normal is 0,0,1 (pointing in the positive Z direction), then everything in 
 *   the negative Z direction will be clipped away.
 * 
 * @FILE_FORMAT_DEFAULT
 *    ClipPlaneManip {
 *    @TABLE_FILE_FORMAT
 *       @TR plane             @TD 1 0 0 0
 *       @TR on                @TD TRUE
 *       @TR draggerPosition   @TD 0 0 0
 *    @TABLE_END
 *    }
 * 
 * @SEE_ALSO
 *    SoClipPlane,
 *    SoDragger
 * 
 * 
 */
class INVENTOR_API SoClipPlaneManip : public SoClipPlane {
  
  SO_NODE_HEADER(SoClipPlaneManip) ;

 public:
  /**
   * Constructor.
   */
  SoClipPlaneManip() ;

  // Fields
  /**
   * Dragger position.
   * 
   */
  SoSFVec3f draggerPosition ;

  /**
   * Returns the dragger node being employed by this manip.
   */
  SoDragger *getDragger() ;

  virtual SoNode *copy(SbBool copyConnections = FALSE) const;

  /**
   * Convenience method to set the clip plane and the dragger position, orientation and size.
   *
   * The clip plane will be the plane defined by @B planeNormal @b and the
   * point at the center of @B box @b. 
   *
   * The dragger will be positioned at the center of @B box @b. The size of the
   * dragger geometry will be the default size multiplied by a scale factor. This 
   * scale factor is the largest dimension of @B box @b multiplied by 
   * @B draggerScaleFactor @b. The default extent of the main part of SoJackDragger
   * is 2, because it extends from -1 to 1 (note that the "arms" of this dragger 
   * extend further). Therefore to make the dragger fit inside the specified box,
   * the draggerScaleFactor parameter should be set to 0.5.
   */
  void setValue(const SbBox3f &box, const SbVec3f &planeNormal, float draggerScaleFactor) ;

  /**
   * Replaces a regular SoClipPlane node with this manipulator.
   */
  SbBool replaceNode(SoPath *p) ;

  /**
   * Replaces this manipulator with a regular SoClipPlane node.
   */
  SbBool replaceManip(SoPath *p, SoClipPlane *newOne) const;

 SoEXTENDER public:

  // These functions implement all actions for this manip.
  // They first traverse the children, then use the 
  // SoClipPlane version of the actions. They traverse first 
  // so that the SoClipPlane will affect objects which 
  // follow it in the tree, but not the dragger-child.
  /** @copydoc SoNode::doAction */
  virtual void doAction(SoAction *action);
  /** @copydoc SoNode::callback */
  virtual void callback(SoCallbackAction *action);
  /** @copydoc SoNode::GLRender */
  virtual void GLRender(SoGLRenderAction *action);
  /** @copydoc SoNode::getBoundingBox */
  virtual void getBoundingBox(SoGetBoundingBoxAction *action);
  /** @copydoc SoNode::getMatrix */
  virtual void getMatrix(SoGetMatrixAction *action);
  /** @copydoc SoNode::handleEvent */
  virtual void handleEvent(SoHandleEventAction *action);
  /** @copydoc SoNode::pick */
  virtual void pick(SoPickAction *action);
  /** @copydoc SoNode::search */
  virtual void search(SoSearchAction *action);
  /** @copydoc SoNode::write */
  virtual void write( SoWriteAction *action );

 SoINTERNAL public:

  // call this after SoInteraction::init();
  static void initClass();
  static void exitClass();

  virtual SoChildList *getChildren() const;
  virtual bool isGroup() { return true; }

  // Internal child management following inheritance of classes.
  virtual void internalRemoveChild( int index );
  virtual void internalRemoveChild( SoNode *child) { internalRemoveChild(internalFindChild(child)); }
  virtual void internalRemoveAllChildren();
  virtual void internalAddChild( SoNode *child );
  virtual int internalFindChild( const SoNode *child ) const;
  virtual void internalInsertChild( SoNode *child, int newChildIndex );
  virtual SoNode *internalGetChild( int index) const { return (*children)[index]; }
  virtual void internalReplaceChild( int index, SoNode *newChild);
  virtual void internalReplaceChild( SoNode *oldChild, SoNode *newChild) { internalReplaceChild(internalFindChild(oldChild),newChild); }

 protected:
  // When the dragger moves, this interprets the fields modifications 
  // of the dragger and sets the fields of this SoClip plane accordingly
  static void valueChangedCB(void *,SoDragger *);
  
  // When one of the field of this node changes, moves the
  // child-dragger to a new location, if necessary.
  SoFieldSensor *planeFieldSensor ;
  SoFieldSensor *onFieldSensor ;
  SoFieldSensor *draggerPosFieldSensor ;
  static void fieldSensorCB(void *, SoSensor *);
  
  // Establishes the given dragger as the new child-dragger
  void setDragger(SoDragger *newDragger);
  
  // The hidden children.
  SoChildList *children ;
  virtual SbBool readInstance(SoInput *in, unsigned short flags);
  virtual SbBool readChildren(SoInput *in);

  // Destructor
  ~SoClipPlaneManip();

 private:
  SbVec3f projectPointPlane() const ;

  int getNumChildren() const { return (children->getLength()); } 
} ;
/*----------------------------------------------------------------------------*/

#endif /* _SO_CLIP_PLANE_MANIP_  */

