/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : David Mott (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_BYTE_STREAM_
#define  _SO_BYTE_STREAM_

#include <Inventor/SbBasic.h>

class SoNode;
class SoPath;
class SoPathList;

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoByteStream
//
//  This class creates a byte stream representation of a scene graph,
//  using the SoWriteAction to write to an in memory buffer. Byte streams
//  are commonly used to transfer data, for instance during copy and paste.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * Converts scene graph objects to character byte streams.
 * 
 * @ingroup General
 * 
 * @DESCRIPTION
 *   This class creates a byte stream representation of a scene graph, using an
 *   SoWriteAction to write path lists to an in-memory buffer. Byte streams are
 *   commonly used to transfer data in copy and paste operations. (The SoWinClipboard
 *   class passes SoByteStream data during copy and paste.)
 * 
 * @SEE_ALSO
 *    SoWinClipboard
 * 
 * 
 */
class INVENTOR_API SoByteStream {
 public:
  /**
   * Constructor.
   */
  SoByteStream();
  /**
   * Destructor.
   */
  ~SoByteStream();

  /**
   * Converts the passed scene graph object(s) into a byte stream. 
   * The object(s) are passed by node. The caller
   * may specify whether the byte stream is written in binary (TRUE) or ASCII (FALSE)
   * format.
   * The converted data can be accessed through getData() and getNumBytes().
   */
  void                convert(SoNode *node, SbBool binaryFormat = TRUE);
    
  /**
   * Converts the passed scene graph object(s) into a byte stream. 
   * The object(s) are passed by path. The caller
   * may specify whether the byte stream is written in binary (TRUE) or ASCII (FALSE)
   * format.
   * The converted data can be accessed through getData() and getNumBytes().
   */
  void                convert(SoPath *path, SbBool binaryFormat = TRUE);
    
  /**
   * Converts the passed scene graph object(s) into a byte stream.
   * The object(s) are passed by pathList. The caller
   * may specify whether the byte stream is written in binary (TRUE) or ASCII (FALSE)
   * format.
   * The converted data can be accessed through getData() and getNumBytes().
   */
  void                convert(SoPathList *pathList, SbBool binaryFormat = TRUE);

  /**
   * Returns the data from the last convert()
   * operation. This byte stream format is well suited to data transfers, like copy
   * and paste.
   * [OIV-WRAPPER-RETURN-TYPE ARRAY{GetNumBytes()}]
   */
  void *              getData()       { return data; }
  /**
   * Returns the number of bytes from the last convert()
   * operation. This byte stream format is well suited to data transfers, like copy
   * and paste.
   * [OIV-WRAPPER-HIDDEN-FROM-DOC]
   */
  size_t/*uint32_t*/                  getNumBytes()   { return numBytes; }
    
  // Unconvert a byte stream back to a path list.
  // This static routine performs an SoDB::read on the data,
  // and returns a path list of the paths read in.
    
  /**
   * Takes byte stream data and unconverts it back to scene graph objects. The
   * objects are returned in a path list.
   */
  static SoPathList *unconvert(SoByteStream *byteStream);
  /**
   * Takes byte stream data and unconverts it back to scene graph objects. The
   * objects are returned in a path list.
   * [OIV-WRAPPER-CUSTOM-CODE]
   */
  static SoPathList *unconvert(void *data, size_t numBytes);
  
 SoEXTENDER public:
  // This allows apps to store raw data here without converting 
  // an Inventor node, path, or path list. This sets isRaw to TRUE,
  // and that data cannot be unconverted.
  void                copy(void *d, size_t len);
  SbBool              isRawData() const { return isRaw; }
     
 private:
  void                *data;
  size_t numBytes;
  SbBool              isRaw;

};

#endif // _SO_BYTE_STREAM_

