/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2018 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/

#ifndef  _SO_STATE_
#define  _SO_STATE_

#include <Inventor/SoLists.h>

class SoAction;
class SoElement;

/**
*  Traversal state
*
* @ingroup General
*
* @DESCRIPTION
*  An SoState collects and holds state while traversing a scene
*  graph. A state is composed of a variety of elements, each of which
*  holds some specific information, such as coordinates or diffuse
*  color of the surface material.
*
*  Each element is stored in its own stack so that save and restore
*  can be implemented as push and pop. These stack operations are
*  performed lazily, so that pushing of a value occurs only when the
*  value would be overwritten, for efficiency.
*
**/
SoEXTENDER_Documented class INVENTOR_API SoState {

public:
  /** Constructor.
  * Takes pointer to action instance this state is
  * part of and a list of type-ids of elements that are enabled.
  */
  SoState(SoAction *action, const SoTypeList &enabledElements);

  /** Destructor */
  ~SoState();

  /** Returns the action instance the state is part of */
  inline SoAction* getAction() const
  { return m_action; }

  /**
   * Returns a writable instance of the element on the top of the
   * stack with the given type.
   */
  template<typename T>
  inline T* getElement()
  {
    return static_cast<T*>(getElement(T::getClassTypeId(), T::getClassStackIndex()));
  }

  /**
   * Returns a writable instance of the element on the top of the
   * stack with the given index and type.
   * \if_cpp @BR We recommend using the templated type safe version. \endif
   */
  SoElement* getElement(const SoType& type, const int stackIndex);

  /**
   * Returns the top (read-only) instance of the given element.
   * @warning This method doesn't capture dependency on current cache
   */
  template<typename T>
  inline const T* getConstElement() const
  {
    return static_cast<const T*>(getConstElement(T::getClassTypeId(), T::getClassStackIndex(), true));
  }

  /** 
   * Returns the top (read-only) instance of the given element stack.
   * \if_cpp @BR We recommend using the templated type safe version. \endif
   */
  inline const SoElement* getConstElement(const SoType& type, const int stackIndex) const
  {
    return getConstElement(type, stackIndex, false);
  }

  /** Pushes (saves) the current state until a pop() restores it. The
  * push is done lazily: this just increments the depth in the
  * state. When an element is accessed with getElement() and its
  * depth is less than the current depth, it is then pushed
  * individually.
  */
  void push();

  /** Pops the state, restoring the state to just before the last push(). */
  void pop();

  /** Prints state to file (for debugging) */
  void print(FILE *fp);

#if SoDEPRECATED_BEGIN(9000)

  /**
   * Returns a writable instance of the element on the top of the
   * stack with the given stack index.
   */
  SoDEPRECATED_METHOD(9000,"Use the templated version instead.")
  SoElement* getElement(const int stackIndex);
  /**
   * Returns the top (read-only) instance of the given element stack.
   */
  SoDEPRECATED_METHOD(9000,"Use the type safe templated version instead.")
  const SoElement* getConstElement(const int stackIndex) const;

#endif /** @DEPRECATED_END */

 SoINTERNAL public:

  /** Push SoState in constructor and pop it in destructor */
  class PushPop
  {
  public:
    PushPop(SoState* state) : m_state(state) { m_state->push(); }
    ~PushPop() { m_state->pop(); }
  private:
    SoState* m_state;
  };


  /** Returns TRUE if element with given stack index is enabled in state */
  inline SbBool isElementEnabled(const SoType& type, const int stackIndex, SbBool checkType = TRUE) const
  {
    if ( !s_checkStateElementsType )
      return isElementEnabled(stackIndex);
    return isElementEnabledChecked(type,stackIndex,checkType);
  }

  // Returns TRUE if element with given stack index is enabled in state
  inline SbBool isElementEnabled(const int stackIndex) const
  {
    if (m_stack && (stackIndex < m_numStacks))
      return (m_stack[stackIndex] != NULL);
    else
      return FALSE;
  }

  // Returns current depth of state
  inline int getDepth() const
  { return m_depth; }

  // Sets/returns flag that indicates whether a cache is open. This
  // flag lets us optimize element capturing; we don't need to try
  // to capture elements if the flag is FALSE.
  inline void setCacheOpen(const SbBool flag)
  { m_cacheOpen = flag; }

  inline SbBool isCacheOpen() const
  { return m_cacheOpen; }

  /** 
   * When set to false, get() on elements won't add dependencies on the current 
   * CacheElement. Return the previous flag's value.
   */
  inline bool trackDependencies(bool flag)
  {
    bool oldValue = m_trackElementsDependencies;
    m_trackElementsDependencies = flag;
    return oldValue;
  }

  /** Return true if elements tracking is enabled (default) */
  bool isTrackingDependencies() const;
  
  /**
   * Internal-only, dangerous method that returns a writeable
   * element without checking for state depth and doing a push.
   * Be very careful and consider the caching implications before
   * using this method!
   */
  template<typename T>
  inline T* getElementNoPush() const
  {
    return static_cast<T*>(getElementNoPush(T::getClassTypeId(), T::getClassStackIndex()));
  }

  /**
   * Internal-only, dangerous method that returns a writeable
   * element without checking for state depth and doing a push.
   * Be very careful and consider the caching implications before
   * using this method!
   */
  inline SoElement* getElementNoPush(SoType type, const int stackIndex) const
  {
    return const_cast<SoElement*>(getConstElement(type, stackIndex, true));
  }

  // Internal-only.
  // Pops from the an element corresponding to classStackIndex.
  // Only the side effect of popping is done (ie. the element is not
  // removed from the stack).
  void popElement(const int classStackIndex) ;

  /**
   * Returns the top (read-only) instance of the given element stack
   * If checkType is true, check if a dervied element has not already been enabled
   */
  inline const SoElement* getConstElement(const SoType& type, const int stackIndex, bool checkType) const
  {
    //This method must be fast, it is called very often so avoid
    //testing for m_stask, it can't be NULL (set at contructor and deleted 
    //in destructor only)
    const SoElement* elt = stackIndex < m_numStacks ? m_stack[stackIndex] : NULL;
    if ( elt == NULL )
    {
      enableMissingElement(type, stackIndex, checkType);
      return m_stack[stackIndex];
    }

    return elt;
  }

  inline const SoElement* getTopElement() const { return m_topElement; }

  /**
   * Directly insert the passed element in the state at the current depth.
   */
  void insertElement( SoElement* );

#if SoDEPRECATED_BEGIN(9000)

  SoDEPRECATED_METHOD(9000,"Use the templated version instead.")
  SoElement* getElementNoPush(const int stackIndex) const;

#endif /** @DEPRECATED_END */

SoEXTENDER public:

  /** 
   * Enables or disables sendToGL in SoElements
   * affected by the GLRenderAction.
   * [OIVJAVA-WRAPPER VISIBILITY{Public}]
   */
  void enableSendToGL(bool enable);

  /**
   * Gets sendToGL activation flag.
   * @see enableSendToGL().
   * [OIVJAVA-WRAPPER VISIBILITY{Public}]
   */
  bool isSendToGLEnabled() const;

private:

  /** isElementEnabledChecked debug version */
  SbBool isElementEnabledChecked(const SoType& type, const int stackIndex, SbBool checkType = TRUE) const;

  /** Return the type which should be instancied for the current action */
  SoType getBestElementType(const SoType& type) const;

  /** Create the depth 0 element with given stackIndex */
  void createDepth0Element(int stackIndex);

  /** Show stacked elements */
  void displayStack(const char *funcStr);

  /**Check that elements are sorted by depth */
  void checkStateConsistency(const char *funcStr);

  /** Enable stackindex element for the current action */
  void enableMissingElement(const SoType& type, int stackIndex, bool checkType) const;

  SoAction            *m_action;        // Action instance state is in
  int                 m_depth;          // Current depth
  SoElement           **m_stack;        // Array of element stacks
  int                 m_numStacks;      // Number of stacks in array
  SoElement *         m_topElement;     // First element in threaded stack

  SbBool              m_cacheOpen;      // TRUE if a cache is open
  bool                m_enableSendToGL; // send to OpenGL activation flag
  bool m_trackElementsDependencies;
  
  /** True if the associated action need dependencies tracking (ie:is a render action) */
  bool m_actionNeedDependencies;

  /** True if state has been initialized with a SoGLRenderAction or derived action */
  bool m_usingGLAction;

  /** Debug modes */
  static int s_checkStateConsistency;
  static int s_displayStateElements;
  static int s_checkStateElementsType;
  static bool s_printStateDepth;
  static FILE* s_stateDepthFile;
};

#endif  /* _SO_STATE_ */


