/*=================================================================================
*** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        ***
***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                 ***
***                                                                            ***
***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS   ***
***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR   ***
***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                       ***
***                                                                            ***
***                        RESTRICTED RIGHTS LEGEND                            ***
***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS  ***
***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN  ***
***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT  ***
***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN  ***
***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         ***
***                                                                            ***
***                   COPYRIGHT (C) 1996-2019 BY FEI S.A.S,                    ***
***                              BORDEAUX, FRANCE                              ***
***                            ALL RIGHTS RESERVED                             ***
=================================================================================*/

#pragma once

#include <Inventor/SbLinear.h>
#include <Inventor/elements/SoTangentBindingElement.h>

/**
 * Surface tangent generator.
 *
 * @ingroup General
 *
 * @DESCRIPTION
 *  This class can be used by lines shape classes to generate
 *  tangents when they do not have valid ones specified. To
 *  generate tangents, create an instance of this class, then specify
 *  each line strip in the shape, then call generate(). After generate()
 *  is called, the tangents can be accessed from the instance. There
 *  will be one tangent generated for each vertex of each strip, in
 *  the order passed in.
 *
 *  For efficiency, a constructor is provided that takes an
 *  approximate number of vertices that will be specified. Use this
 *  constructor if you know roughly how many vertices will be sent;
 *  this will cut down on allocation overhead.
 *
 *  @B NOTE @b The SoTangentGenerator destructor DOES NOT delete the
 *  array of generated tangents. The caller is responsible for doing
 *  so. This allows the tangents to be cached elsewhere without having
 *  to copy them.
 *
 * [OIV-WRAPPER-CLASS NO_WRAP]
 */
SoEXTENDER class INVENTOR_API SoTangentGenerator
{
public:
  SoTangentGenerator( bool useLines, int approxNumVertices = 16 );
  ~SoTangentGenerator();

  /**
   * Resets the approximative number of vertices that
   * we have to generate tangents for. These correspond to the same parameters in
   * the constructor, but allows setting them after creation. @BR
   * Note: Must be called before first call to polygonVertex or triangle calls..
   */
  void setupApproxShapeSize( int32_t approxNumVertices );

  /**
   * Send a line strip's worth of vertices. Begin a strip, send as
   * many vertices as you want, and then end the strip.
   */
  void beginLineStrip();
  void lineStripVertex( const SbVec3f& point );
  void endLineStrip();

  /**
   * Specifies a pointer to the array of points that will be used
   * in subsequent calls to polygonVertex(index) and triangle(index1,index2,index3). @BR
   *
   * Using this mode allows reusing vertices, avoiding data copy and increasing performance.
   * IMPORTANT NOTE:
   * - This function should be called before any call to polygonVertex/triangle functions.
   * - The data should remain valid until generate() has ended.
   * - Once setupIndexedVertexArray has been called, only "by index" functions are valid.
   */
  void setupIndexedVertexArray( const SbVec3f* pointArray );

  /**
   * Send a line strip's worth of vertices using indexed vertices. @BR
   * NOTE: must be called after setupIndexedVertexArray().
   */
  void lineStripVertex( const int32_t pointIndex );

  /** Calculate the tangents once all vertices have been sent */
  void generate();

  /**
   * Returns number of tangents generated. @BR
   * This will be equal to the number of points sent. @BR
   * NOTE: must be called after generate().
   */
  int getNumTangents() const
  {
    return m_numVertTangents;
  }

  /**
   * Returns a pointer to the array of tangents. @BR
   * NOTE: must be called after generate().
   */
  const SbVec3f* getTangents() const
  {
    return m_vertTangents;
  }

  /**
   * Returns the i'th tangent in the array. @BR
   * NOTE: must be called after generate().
   */
  const SbVec3f& getTangent( int32_t i ) const
  {
    return m_vertTangents[i];
  }

  SoTangentBindingElement::Binding getBinding() const
  {
    return m_binding;
  }

  /**
   * Add or modify a tangent vector. @BR
   * Some shapes may need to adjust or reorder tangents after
   * generation. This method can be used to change a tangent. It will
   * insert room in the array if necessary to accommodate the new
   * tangent.
   * NOTE: must be called after generate().
   */
  void setTangent( int32_t index, const SbVec3f& newTangent );

  SoINTERNAL public:
    /**
     * Some shapes may need to adjust or reorder all tangents after
     * generation. This method can be used to change all tangent in one
     * shot without any copy.
     */
    void setTangentBuffer( int32_t numTangents, const SbVec3f* tangents );

private:
  SbVec3f computeTangent( const SbVec3f& previous, const SbVec3f& current, const SbVec3f& next ) const;

  std::vector<SbVec3f> m_tangents;
  std::vector<SbVec3f> m_polylinePoints;

  const SbVec3f* m_points;
  const SbVec3f* m_vertTangents;

  int32_t m_numVertTangents;
  SoTangentBindingElement::Binding m_binding;
  bool m_useLines;
  static int s_generationMode;
};
