#ifndef SO_ALGEBRAIC_CONE_H
#define SO_ALGEBRAIC_CONE_H

#include <Inventor/nodes/SoAlgebraicShape.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFBool.h>

/**
 * @VSGEXT Algebraic cone shape node.
 *
 * @ingroup ShapeNodes
 * 
 * @DESCRIPTION
 *   This node represents a simple cone whose central axis is aligned with the
 *   y-axis. By default, the cone is centered at (0,0,0) and has a size of -1 to +1
 *   in all three directions. The cone has a radius of 1 at the bottom and a height
 *   of 2, with its apex at 1.
 *   
 *   The cone is transformed by the current cumulative transformation and is drawn
 *   with the current lighting model and material.
 *
 *   This node can generally be used in place of the SoCone geometric shape node.
 *   Unlike the geometric shape nodes, which create triangle geometry to represent their
 *   shape, the algebraic shape nodes compute and render their shape on the GPU.
 *   Algebraic shapes can also be used with SoMultipleInstance to efficiently render
 *   a very large number of shapes.
 *
 *   Please see SoAlgebraicShape for important notes and limitations for algebraic shapes.
 *
 * @FILE_FORMAT_DEFAULT
 *    AlgebraicCone {
 *    @TABLE_FILE_FORMAT
 *       @TR radius   @TD 1
 *       @TR height   @TD 2
 *       @TR capping  @TD TRUE
 *    @TABLE_END
 *    }
 * 
 * @SEE_ALSO
 *    SoAlgebraicShape,
 *    SoMultipleInstance,
 *    SoCone
 */
class INVENTOR_API SoAlgebraicCone : public SoAlgebraicShape
{
  SO_NODE_HEADER(SoAlgebraicCone);

public:

  /**
   * Specifies the cone's height. Value must be greater than 0.0. Default is 2.
   */
  SoSFFloat height;

  /**
   * Specifies the radius of the base circle. Value must be greater than 0.0. Default is 1.
   */
  SoSFFloat radius;

  /**
   * Specifies if cone is capped or not. Default is true.
   */
  SoSFBool capped;

  /**
   * Default constructor.
   */
  SoAlgebraicCone ();

protected:

  /**
   * Destructor.
   */
  virtual ~SoAlgebraicCone () {}

  /**
   * Compute the bounding box of the shape
   * @param box the bounding box
   * @param center the center of the bbox
   */
  virtual void computeBBox ( SbBox3f &box, SbVec3f &center );

SoEXTENDER public:

  /**
   * Specialized ray pick action.
   */
  virtual void rayPick ( SoRayPickAction *action );

  /**
   * Alternate representation (i.e. SoCone) for unhandled actions
   */
  virtual SoNode* getAlternateRep ( SoAction* action );

SoINTERNAL public:

  //@{
  /**
   * Registration.
   */
  static void initClass ();
  static void exitClass ();
  //@}
};

#endif // SO_ALGEBRAIC_CONE_H
