#ifndef SO_ALGEBRAIC_CYLINDER_H
#define SO_ALGEBRAIC_CYLINDER_H

#include <Inventor/nodes/SoAlgebraicShape.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFBool.h>

/**
 * @VSGEXT Algebraic cylinder shape node
 *
 * @ingroup ShapeNodes
 * 
 * @DESCRIPTION
 *   This node represents a simple capped cylinder centered around the y-axis. By
 *   default, the cylinder is centered at (0,0,0) and has a default size of -1 to +1
 *   in all three dimensions. You can use the #radius and #height fields
 *   to create a cylinder with a different size.
 *
 *   The cylinder is transformed by the current cumulative transformation and is drawn
 *   with the current lighting model and material.
 *
 *   This node can generally be used in place of the SoCylinder geometric shape node.
 *   Unlike the geometric shape nodes, which create triangle geometry to represent their
 *   shape, the algebraic shape nodes compute and render their shape on the GPU.
 *   Algebraic shapes can also be used with SoMultipleInstance to efficiently render
 *   a very large number of shapes.
 *
 *   Please see SoAlgebraicShape for important notes and limitations for algebraic shapes.
 *
 * @FILE_FORMAT_DEFAULT
 *    AlgebraicCylinder {
 *    @TABLE_FILE_FORMAT
 *       @TR radius   @TD 1
 *       @TR height   @TD 2
 *       @TR capping  @TD TRUE
 *    @TABLE_END
 *    }
 * 
 * @SEE_ALSO
 *    SoAlgebraicShape,
 *    SoMultipleInstance,
 *    SoCylinder
 */
class INVENTOR_API SoAlgebraicCylinder : public SoAlgebraicShape
{
  SO_NODE_HEADER(SoAlgebraicCylinder);

public:

  /**
   * Specifies the cylinder's height. Value must be greater than 0.0. Default is 2.
   */
  SoSFFloat height;

  /**
   * Specifies the cylinder's radius. Value must be greater than 0.0. Default is 1.
   */
  SoSFFloat radius;

  /**
   * Specifies if cylinder is capped or not. Default is true.
   */
  SoSFBool capped;

  /**
   * Default constructor.
   */
  SoAlgebraicCylinder ();

protected:

  /**
   * Destructor.
   */
  virtual ~SoAlgebraicCylinder () {}

  /**
   * Compute the bounding box of the shape
   * @param box the bounding box
   * @param center the center of the bbox
   */
  virtual void computeBBox ( SbBox3f &box, SbVec3f &center );

SoEXTENDER public:

  /**
   * Specialized ray pick action.
   */
  virtual void rayPick ( SoRayPickAction *action );

  /**
   * Alternate representation (i.e. SoCylinder) for unhandled actions
   */
  virtual SoNode* getAlternateRep ( SoAction* action );

SoINTERNAL public:

  //@{
  /**
   * Registration.
   */
  static void initClass ();
  static void exitClass ();
  //@}
};

#endif // SO_ALGEBRAIC_CYLINDER_H
