#pragma once

#include <Inventor/nodes/SoNode.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFFloat.h>

/**
 * @VSGEXT Alpha (transparency) policy node
 *
 * @ingroup nodes
 *
 * @DESCRIPTION
 *  This node sets the current alpha (transparency) comparison policy for the renderer.
 *  It affects the blending operation to control how (and if) a transparent object
 *  should be rendered.
 *
 *  Current available policies include :
 *  - As is :   Use the alpha value of the fragment to control color blending
 *  - Opaque :  If the fragment to render has an alpha value greater than 0.0,
 *              the object is rendered as opaque (no transparency). If the alpha value is
 *              0.0 the object is considered fully transparent.
 *  - Discard : If the fragment alpha value is different from 1.0, the object is
 *              considered fully transparent. Otherwise it's rendered opaque.
 *
 * @SEE_ALSO SoBlendElement
 */
class INVENTOR_API SoAlphaPolicy : public SoNode
{
  SO_NODE_HEADER(SoAlphaPolicy);

public:

  /**
   * Policy to use 
   */
  enum Policy
  {
    /**
     * Use the fragment's alpha value as is.
     */
    POLICY_AS_IS = 0,

    /**
     * All semi-transparent fragment (alpha > 0) are considered as opaque.
     */
    POLICY_OPAQUE = 1,

    /**
     * All semi-transparent fragment (alpha < 1) are considered as fully transparent.
     */
    POLICY_DISCARD = 2
  };

  /**
   * Get default alpha policy
   */
  static Policy getDefaultPolicy();

  /**
   * Policy used
   * Please refer to enum Policy. Default value is POLICY_AS_IS
   */
  SoSFEnum policy;

  /**
   * Default Constructor
   */
  SoAlphaPolicy();

SoEXTENDER public:
  virtual void GLRender(SoGLRenderAction *action);

SoINTERNAL public:
  static void initClass();
  static void exitClass();

protected:
  virtual ~SoAlphaPolicy();
};
