#if !defined(_SO_BASE_EXTRUSION_)
#define  _SO_BASE_EXTRUSION_

/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoMFVec3f.h>
#include <Inventor/fields/SoSFVec3f.h>
#include <Inventor/nodes/SoShape.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/SbPImpl.h>

SO_PIMPL_PUBLIC_DECLARATION(SoExtrusion)

SO_PIMPL_PUBLIC_DECLARATION(SoBaseExtrusion)

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoBaseExtrusion
//
//  Node that defines the extrusion parameters 
//
//////////////////////////////////////////////////////////////////////////////

/**
* Abstract base class for all extrusion-based shape nodes.
* 
* @ingroup ShapeNodes
* 
* @DESCRIPTION
*   This node is the abstract base class for all extrusion-based shape (geometry)
*   nodes. It is used as a repository for convenience functions for subclasses and
*   to provide a type identifier to make it easy to determine whether a shape is
*   extrusion-based.
*   
*   All extrusion-based shapes are created by extruding a 2D shape along a 3D #spine.
*   The user can determine which caps are created using the fields #beginCap and #endCap.
*
*   The #scaleMode field is used to select the points that will be scaled by the scaleFactor
*   in the current transformation (for example SoTransform), if any.  Translation and rotation
*   are applied in all cases.
*
*   The #extrusionMode field is used to select the extrusion algorithm. 
* 
* @FILE_FORMAT_DEFAULT
*   This is an abstract class. See the reference page of a derived class for the
*   format and default values.
* 
* @SEE_ALSO
*    SoExtrusion,
*    SoCircularExtrusion
* 
* 
*/
class INVENTOR_API SoBaseExtrusion : public SoShape
{
  SO_NODE_ABSTRACT_HEADER(SoBaseExtrusion);
  SO_PIMPL_PUBLIC_HEADER(SoBaseExtrusion)

 public:

   enum ScaleMode
   {
     /** Default scaling. Every point in the generated geometry is scaled. */
     DEFAULT,
     /** Only the spine points are scaled.
      *  The cross section points are not scaled, so the cross 
      *  section shape is undistorted even is scaling is non-uniform.*/
     SPINE_ONLY,
     /** Only the cross section points are scaled. @I Not currently implemented. @i */
     SECTION_ONLY,
     /** Ignores the scaling state. @I Not currently implemented. @i */
     NONE
   };

   enum ExtrusionMode
   {
     /** Default extrusion algorithm.
      *  The algorithm uses constant section parts cut at spine points 
      *  in a way similar to what a framer would do to build a frame.
      *  This provides a better continuity of the shape and avoids cracks.
      * 
      *  This algorithm determines the piecewise extrusion directions as a vector
      *  between consecutive spine points. This means that zero or really 
      *  small distances between consecutive points can produce odd extrusion
      *  directions and result in wrong geometry. It is up to the application to filter
      *  spine points to avoid these cases. The algorithm can produce odd
      *  geometries when spines points are overlapping or zigzags.
      */
     SMOOTH = 0,

     /** Old extrusion algorithm.
      *  It does not guarantee that extrusion portions have
      *  constant section and shapes could be more twisted. */
     BASIC = 1,

     /** Smooth but avoids self intersections caused by large curvature radius of the spine.
      When the spine has curves with a large curvature radius compared to the radius of the extrusion section,
      SMOOTH generation might produce self intersections. Use this mode to avoid self intersections 
      at the cost of some localized violation of the extrusion smoothness. 
      (Section parts at spine joints may not be constant where self intersections were present.)
      This mode is useful to avoid artifacts in boolean results when SoBaseExtrusion derived 
      objects are used as operands for CSG operations. */
     SMOOTH_WITHOUT_SELF_INTERSECTIONS = 2,
   };

  // Fields
  /**
   * If TRUE, begin planar cap surfaces generated. Default is TRUE.
   */
  SoSFBool            beginCap;
  
  /**
   * If TRUE, end planar cap surfaces generated. Default is TRUE.
   */
  SoSFBool            endCap;

  /**
   *  Spine points. Default is [0 0 0, 0 1 0].
   */
  SoMFVec3f           spine;

  /**
  *  Scaling behavior of the extrusion.
  *  @useenum{ScaleMode}. Default is DEFAULT.
  */
  SoSFEnum            scaleMode;

  /**
  *  Construction behavior of the extrusion.
  *  @useenum{ExtrusionMode}.  Default is SMOOTH.
  */
  SoSFEnum            extrusionMode;

 protected:
  /** Constructor. */
  SoBaseExtrusion();

  /** Destructor. */
  virtual ~SoBaseExtrusion();

 private:

  void commonConstructor();

 SoEXTENDER public:

  /** Creates extrusion vertices in 8.5 fully compatible mode */
  static SbVec3f   *generateExtrusionVertices( SoAction *action, const SbVec3f * spine,
                                               int numSpinePts, const SbVec2f *crossSection,
                                               int numCrossSectionPts, const SbRotation *orientPts,
                                               int numOrientPts, const SbVec2f *scalePts,
                                               int numScalePts );

  static SbVec3f   findZaxis( const SbVec3f * spine, const int numspine,
                              const int i, const SbBool closed );
  static SbVec3f   findYaxis( const SbVec3f * spine, const int numspine,
                              const int i, const SbBool closed );

 SoINTERNAL public:

  static void initClass();
  static void exitClass();

  static void updateTols();

 protected:

  static float    s_dot_epsilon;
  static float    s_epsilon;

  friend class inventor::impl::SoExtrusionImpl;
 };


#endif /* _SO_BASE_EXTRUSION_ */

