/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Nick Thompson (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_CALLBACK_
#define  _SO_CALLBACK_

#include <Inventor/SbLinear.h>
#include <Inventor/nodes/SoNode.h>

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoCallback
//
//  The SoCallback node is a general mechanism that allows
//  applications to insert callback functions into a scene graph. The
//  callback function registered with the node is called each time
//  that node is traversed to perform any scene graph action. When the
//  callback function is called, it is passed a pointer to the action
//  being traversed and a user-data pointer that is registered with
//  the callback function. A NULL callback function pointer means that
//  no function will be called.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * Callback functions registered with this node should be of this type.
 *
 * @memberof SoCallback
 *
 * [OIV-WRAPPER NAME{CallbackCB}]
 */
typedef void SoCallbackCB(void *userData, SoAction *action);

/**
* Provides custom behavior during actions.
* 
* @ingroup nodes
* 
* @DESCRIPTION
* This node provides a general mechanism for executing some code during an
* Open Inventor scene graph traversal, i.e. when an action is applied to the scene graph.
*
* In some cases, implementing a new node class (a so-called custom node) derived
* from one of the existing classes is the best and cleanest solution. But for simple
* cases, and for prototyping, an SoCallback node may solve the problem.
*
* You can use this node to get values from the traversal state list using the
* SoElement classes and, conversely, to modify values in the traversal state list.
* For example, SoLineWidthElement contains the current line width and is normally
* set by an SoDrawStyle node.
*
* This node @I cannot@i be used to make OpenGL calls. OpenGL calls can only be made
* using an SoGLCallback node. (This is a change starting with Open Inventor 10.0)
*
* This node cannot be used to modify the structure of the scene graph. This may
* cause a crash. This node should not be used to modify fields in other nodes.
* The same effect can usually be obtained by setting the corresponding 'element'.
* See the Cautions section below for more details.
*
* @B Create the node:@b
* \if_cpp
* \par
* \code
*   SoCallback* callbackNode = new SoCallback();
*   callbackNode->setCallback( myCallback );
*   root->addChild( callbackNode );
* \endcode
* \endif
* \if_dotnet
* \par
* \code
*   SoCallback callbackNode = new SoCallback();
*   callbackNode.CallbackHandler = new SoCallback.CallbackCB( myCallback );
*   root.AddChild( callbackNode );
* \endcode
* \endif
* \if_java
* \par
* \code
*   SoCallback callbackNode = new SoCallback();
*   callbackNode.setCallback( new myCallback() );
*   root.addChild( callbackNode );
*   \endcode
* \endif
*
* Create the \if_dotnet @B Delegate:@b \else @B Callback: @b \endif
*
* The \if_dotnet delegate \else callback \endif function registered with the node will be
* executed during traversal of the scene graph by various Open Inventor actions
* (not only the render action).  Actions commonly applied to the scene graph
* include SoGLRenderAction, SoHandleEventAction, SoRayPickAction,
* SoGetBoundingBoxAction, SoSearchAction and SoCallbackAction.  You should consider
* which actions the function should, or should not, be executed for. In the \if_dotnet delegate
* \else callback \endif you can check the action type, for example:
*
* \if_cpp
* \par
* \code
*   void myCallback( void* data, SoAction* action )
*   {
*     // Only handle specific actions
*     if (action->isOfType(SoGLRenderAction::getClassTypeId())) {
*       // Call to OpenInventor functions
*     }
*   }
* \endcode
* \endif
* \if_dotnet
* \par
* \code
*   void myCallback( SoAction action )
*   {
*      // Only handle specific actions
*      if (action.GetType() == typeof(SoGLRenderAction)) {
*       // Call to OpenInventor functions
*      }
*   }
* \endcode
* \endif
* \if_java
* \par
* \code
*   class myCallback implements SoCallback.CB
*   {
*     @Override
*     public void invoke( SoAction action )
*     {
*       // Only handle specific actions
*       if (action instanceof SoGLRenderAction) {
*       // Call to OpenInventor functions
*       }
*     }
*   }
* \endcode
* Or create node and callback in-line:
* \code
*   SoCallback callbackNode = new SoCallback();
*   callbackNode.setCallback(
*       new SoCallback.CB() 
*       {
*           @Override
*           public void invoke( SoAction action )
*           {
*               // Only handle specific actions
*               if (action instanceof SoGLRenderAction) {
*                  // Call to OpenInventor functions
*               }
*           }
*       }
*   );
* \endcode
* \endif
*
* Callbacks that modify the Open Inventor traversal state, e.g. setting
* SoLineWidthElement, generally should do that for all actions.
*
* Getting a value from the traversal state list requires access to the SoState
* object for the current traversal.  This information (and more) is available from
* the SoAction object passed to the callback.  Modifying a value in the traversal
* state list requires the SoState object and, in most cases, a \if_cpp pointer
* \else reference \endif to the node setting the element.  This information is also
* available from the SoAction object as the "tail" node of the action's current
* path, which is updated for each node visited as the action traverses the scene
* graph.  Getting and modifying the traversal state is shown in the next example.
* The goal is to highlight some line geometry by making the lines 3X wider than
* whatever the current line width is.  We get the current line width from the
* traversal state, multiply by three, then set the new line width in the state.
*
* \if_cpp
*   \par
*   \code
*     void myCallback( void* data, SoAction* action )
*     {
*       // Traversal state
*       SoState* state = action->getState();
*
*       // Get current line width from traversal state
*       float lineWidth = SoLineWidthElement::get( state );
*
*       // Scale up line width (may be zero)
*       lineWidth = (lineWidth > 0) ? (3 * lineWidth) : 3;
*     
*       // Set new value in state
*       SoLineWidthElement::set( state, action->getCurPath()->getTail(), lineWidth );
*     }
*   \endcode
* \endif
* \if_dotnet
*   \par
*   \code
*     void myCallback( SoAction action )
*     {
*       // Traversal state
*       SoState state = action.GetState();
*
*       // Get line width in traversal state
*       float lineWidth = SoLineWidthElement.Get( state );
* 
*       // Scale up line width (may be zero)
*       lineWidth = (lineWidth > 0) ? (3 * lineWidth) : 3;
* 
*       // Set new value in state
*       SoLineWidthElement.Set( state, action.GetCurPath().GetTail(), lineWidth );
*     }
*   \endcode
* \endif
* \if_java
*   \par
*   \code
*     class myCallback implements SoCallback.CB
*     {
*       @Override
*       public void invoke( SoAction action )
*       {
*         // Traversal state
*         SoState state = action.getState();
*
*         // Get line width from traversal state
*         float lineWidth = SoLineWidthElement.get( state );
* 
*         // Scale up line width (may be zero)
*         lineWidth = (lineWidth > 0) ? (3 * lineWidth) : 3;
* 
*         // Set new value in state
*         SoLineWidthElement.set( state, action.getCurPath().getTail(), lineWidth );
*       }
*     }
*   \endcode
* \endif
*
* @B Cautions:@b
*
* Most actions do a straightforward 'depth first' traversal of the scene graph.
* In other words, nodes are visited exactly as organized in the scene graph.
* But SoGLRenderAction's traversal of the scene graph for rendering is much more complex.
* The 'behavior' implied by the structure of the scene graph is always respected.
* For example nodes inherit traversal state from above/left and SoSeparator nodes save
* and restore traversal state.  However you cannot assume that there is a single
* traversal that exactly follows the structure of the scene graph.  For example, some
* parts of the scene graph may be traversed 'out of order' to simulate transparency
* correctly and the scene graph may be traversed multiple times to implement rendering
* effects like shadows.  Because of this, there are some limitations on what can be done
* in the callback function.
*
* - Do NOT change the "structure" of the scene graph (insert or remove nodes). @BR
*   This may cause Open Inventor to crash.  If such a change is required, schedule an
*   SoOneShotSensor and do the work in the sensor's callback function.  This function
*   will be called at a safe time when no traversal is being done and the change will
*   take effect on the next traversal.
*
* - Avoid changing fields of other nodes. @BR
*   This causes notification, which is inefficient during traversal and may cause
*   sensors to trigger with undesired results. Also, some fields, if changed, invalidate
*   information that is cached during traversal and may result in incorrect rendering.
*   @BR @BR
*   Think about whether it is really necessary to make this change during the render
*   traversal.  If your callback is responding to a change in the traversal state, then
*   it may be possible (and generally is safer) to use the "Observer" pattern and attach
*   an SoNodeSensor or SoFieldSensor to the node or field of interest.  For example,
*   attach an SoFieldSensor to the position field of the camera (see the viewer's
*   getCamera() method) to make decisions based on the distance from the camera to some geometry.
*   @BR @BR
*   Whenever possible, make changes to attributes and properties by setting elements in the
*   traversal state list instead of changing field values.  For example, line width can (and
*   should) be changed by setting the SoLineWidthElement.  You can even control which child
*   of an SoSwitch node is traversed, using the SoSwitchElement.  See the "Action Behavior"
*   section of each property node's documentation to see which elements it uses.
*   @BR @BR
*   If it is necessary to change a field based on information obtained during traversal, the
*   best solution is to use an SoOneShotSensor as described in the previous point.  The next
*   best solution is to temporarily disable notification on the node that contains the field
*   (see SoNode::enableNotify()).
*
* - In general you should not use an SoCallback node to count the number of "frames"
*   rendered.  You're really counting the number of render traversals and the scene
*   graph may be traversed multiple times to render a single frame.  During a render
*   traversal you can get the actual frame count by calling the SoGLRenderAction
*   method \if_dotnet GetFrameCounter(). \else getFrameCounter(). \endif  See the example
*   code for how to determine if the current action is a render action.
*
* @B Render caching:@b
*
*  One of the most important and most effective optimizations in Open Inventor is automatic
*  building of 'render caches' (mainly at SoSeparator nodes) for 'static' parts of the scene
*  graph.  A render cache encapsulates geometry and state (attributes and properties) that
*  can be rendered without scene graph traversal.  One advantage is that geometry in a render
*  cache can be optimized for best performance.  Another important advantage is that when an
*  SoSeparator has a valid render cache, Open Inventor does not need to traverse its children,
*  avoiding the CPU time to traverse those nodes.  In general you should try to ensure that
*  your SoCallback node can be render cached.  In most cases SoCallback nodes can be cached
*  even if they access/set traversal state.
*
*  If the callback modifies the Open Inventor traversal state based on information from other
*  nodes in the scene graph, then you may be able to get that information from the SoElement
*  object(s) set by those nodes.  See the line width example above.  In that case the SoCallback
*  node can be part of a render cache, because Open Inventor remembers that the render cache has
*  a dependency on the specific value of the element(s) whose 'get' method was called.  On
*  subsequent traversals, if the element's value matches the saved value, then the render cache
*  is still valid and be used.  If the element's value has changed, then the render cache is
*  automatically discarded and the SoSeparator's children are traversed, so the SoCallback node's
*  callback will be called again.  On later traversals, if the value of the element is no longer
*  changing, Open Inventor will eventually rebuild the render cache.  The line width example
*  above can be safely render cached.
*
*  In a few cases, the callback node may not be cacheable, depending on what it does.  If a
*  callback node relies on any information outside of Inventor that may change (such as a global
*  application variable), it should not be cached, because the callback function will not be
*  executed when the render cache is valid (SoCallback will not be traversed).
*  We can imagine other cases where the callback function must be executed on every traversal.
*  For example a temporary callback that prints a debug message on every traversal.
*  To prevent Inventor from automatically creating a cache, call the SoCacheElement's invalidate()
*  method in the callback function.  But keep in mind that when you do this, it prevents
*  caching by the callback node's parent and that node's parent and every SoSeparator up to
*  the top of the scene graph.  To minimize the performance hit, try to only put non-cacheable
*  nodes near the top of the scene graph.  Here is a non-cacheable callback:
*
*  \if_cpp
*  \code
*    void myCallback( void* data, SoAction* action )
*    {
*      // Only handle specific actions
*      if (action->isOfType(SoGLRenderAction::getClassTypeId())) {
*
*        // Traversal state
*        SoState* state = action->getState();
*
*        // This node must not be render cached (must be executed).
*        SoCacheElement::invalidate( state );
*
*        std::cout << "Render traversal: " << m_counter++ << std::endl;
*      }
*    }
*  \endcode
*  \endif
*  \if_dotnet
*  \code
*    void myCallback( SoAction action )
*    {
*       // Only handle specific actions
*       if (action.GetType() == typeof(SoGLRenderAction)) {
*
*            // Traversal state
*            SoState state = action.GetState();
* 
*            // This node must not be render cached (must be executed).
*            SoCacheElement.Invalidate(state);
*
*            Console.WriteLine( "Render traversal: {0}", m_counter++ };
*        }
*    }
*  \endcode
*  \endif
*  \if_java
*  \code
*    class myCallback implements SoCallback.CB
*    {
*      @Override
*      public void invoke( SoAction action )
*      {
*        // Only handle specific actions
*        if (action instanceof SoGLRenderAction) {
*
*            // Traversal state
*            SoState state = action.getState();
* 
*            // This node must not be render cached (must be executed).
*            SoCacheElement.invalidate( state );
*
*            System.out.printf( "Render traversal: %d%n", m_counter++ );
*        }
*      }
*    }
*  \endcode
*  \endif
* 
* @FILE_FORMAT_DEFAULT
*    Callback {
*    @TABLE_FILE_FORMAT
*    @TABLE_END
*    }
* 
* @ACTION_BEHAVIOR
*    SoGLRenderAction,  SoGetBoundingBoxAction,  SoPickAction @BR
*        Calls the specified \if_dotnet delegate \else callback \endif for all actions.
* 
* @SEE_ALSO
*    SoAction,
*    SoCallbackAction,
*    SoEventCallback,
*    SoGLCallback
* 
*/
class INVENTOR_API SoCallback : public SoNode {

  SO_NODE_HEADER(SoCallback);

 public:
  /**
   * Creates a callback node with default settings.
   */
  SoCallback();

  /**
   * Sets pointer to callback function and user data. By default, the function
   * pointer in the node is NULL and does nothing.
   * [OIV-WRAPPER EVENT_NAME{CallbackHandler}]
   */
  void setCallback(SoCallbackCB *func, void *localUserData = NULL)
    { callbackFunc = func; callbackData = localUserData; }

 SoEXTENDER public:

  // Traversal methods for all the actions:
  virtual void doAction(SoAction *action);
  virtual void callback(SoCallbackAction *action);
  virtual void GLRender(SoGLRenderAction *action);
  virtual void getBoundingBox(SoGetBoundingBoxAction *action);
  virtual void getMatrix(SoGetMatrixAction *action);
  virtual void handleEvent(SoHandleEventAction *action);
  virtual void pick(SoPickAction *action);
  virtual void search(SoSearchAction *action);
  virtual void write(SoWriteAction *action);
  virtual void getPrimitiveCount(SoGetPrimitiveCountAction *action);
  
 SoINTERNAL public:

  static void initClass();
  static void exitClass();

  // Convenient Constructor
  SoCallback( SoCallbackCB* func, void* localUserData = nullptr );


  //Gets pointer to callback function. By default, the function pointer in the node is nullptr and does nothing.
  SoCallbackCB* getCallback() const;

  // Copies the contents of the given node into this instance
  virtual void copyContents(const SoFieldContainer *fromFC,
                            SbBool copyConnections);

 protected:

  // destructor
  virtual ~SoCallback();
  
  // Function to call
  SoCallbackCB *callbackFunc;

  // User data to pass it
  void *callbackData;
};

#endif /* _SO_CALLBACK_ */

