/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_CAMERA_
#define  _SO_CAMERA_

#include <Inventor/SbBox.h>
#include <Inventor/SbViewportRegion.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFRotation.h>
#include <Inventor/fields/SoSFVec3f.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/SbPImpl.h>

#ifdef _WIN32
#pragma warning(push)
#pragma warning(disable:4251)
#endif

class SbViewportRegion;
class SoState;


// Callback function prototypes

class SoCamera ;
typedef void SoCameraCB( SoCamera *camera, SoGLRenderAction* action );

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoCamera
//
//  Abstract base class Camera node.
//
//////////////////////////////////////////////////////////////////////////////

SO_PIMPL_PUBLIC_DECLARATION(SoCamera)

/**
 * Abstract base class for camera nodes.
 *
 * @ingroup CameraNodes
 *
 * @DESCRIPTION
 *   This is the abstract base class for all camera nodes. It defines the common
 *   methods and fields that all cameras have. Cameras are used to view a scene. When
 *   a camera is encountered during rendering, it sets the projection and viewing
 *   matrices and viewport appropriately; it does not draw geometry. It also controls
 *   additional camera parameters such as exposure compensation and exposure mode.
 *   You should place cameras before any shape nodes or light nodes in a scene graph;
 *   otherwise, those shapes or lights will not render correctly.
 *
 *   The current transformation affects cameras, this allows you to position a camera by placing
 *   a transformation node before it in the scene graph. The default position and orientation
 *   of a camera is at (0,0,1), looking along the negative z-axis.
 *
 *   You can also use a node kit to create a camera; see the reference page for
 *   SoCameraKit.
 *
 *   The SoCameraInteractor class provides useful algorithms for manipulating a camera.
 *
 *   Note: Exposure parameters impact SoGradientBackground in an undesirable way.
     Exposure parameters do not impact the SoAnnotation scene graph.
 *
 *  @EXAMPLE
 *  \if_cpp
 *  Compute the current view vector or up vector. \par
 *    \code
 *    SoCamera* camera . . .
 *    const SbRotation& orientation = camera->orientation.getValue();
 *    SbVec3f upVec;
 *    orientation.multVec( SbVec3f(0,1,0), upVec );
 *    SbVec3f vwVec;
 *    orientation.multVec( SbVec3f(0,0,-1), vwVec );
 *    \endcode
 *
 *  Shortcut to get the current view vector or up vector. \par
 *    \code
 *    SbMatrix mx;
 *    mx = camera->orientation.getValue();
 *    SbVec3f upVec(  mx[1][0],  mx[1][1],  mx[1][2] );
 *    SbVec3f vwVec( -mx[2][0], -mx[2][1], -mx[2][2] );
 *    \endcode
 *
 *  Compute the current focal point. \par
 *    \code
 *    SbMatrix mx;
 *    mx = camera->orientation.getValue();
 *    SbVec3f vwVec( -mx[2][0], -mx[2][1], -mx[2][2] );
 *    SbVec3f focalPt = camera->position.getValue()
 *                    + camera->focalDistance.getValue() * vwVec;
 *    \endcode
 *  \endif
 *  \if_dotnet
 *  Compute the current view vector or up vector. \par
 *    \code
 *    SoCamera camera . . .
 *    SbRotation orientation = camera.orientation.Value;
 *    SbVec3f upVec;
 *    orientation.MultVec( new SbVec3f(0,1,0), out upVec );
 *    SbVec3f vwVec;
 *    orientation.MultVec( new SbVec3f(0,0,-1), out vwVec );
 *    \endcode
 *
 *  Shortcut to get the current view vector or up vector. \par
 *    \code
 *    SbMatrix mx;
 *    mx.SetRotate(camera.orientation.Value);
 *    SbVec3f upVec = new SbVec3f( mx[1,0],  mx[1,1],  mx[1,2] );
 *    SbVec3f vwVec = new SbVec3f(-mx[2,0], -mx[2,1], -mx[2,2]);
 *
 *  Compute the current focal point. \par
 *    \code
 *    SbMatrix mx;
 *    mx.SetRotate(camera.orientation.Value);
 *    SbVec3f vwVec = new SbVec3f(-mx[2,0], -mx[2,1], -mx[2,2]);
 *    SbVec3f focalPt = camera.position.Value
 *                    + camera.focalDistance.Value * vwVec;
 *    \endcode
 *  \endif
 *  \if_java
 *  Compute the current view vector or up vector. \par
 *    \code
 *    SoCamera camera . . .
 *    SbRotation orientation = camera.orientation.getValue();
 *    SbVec3f upVec = orientation.multVec( new SbVec3f(0,1,0) );
 *    SbVec3f vwVec = orientation.multVec( new SbVec3f(0,0,-1) );
 *    \endcode
 *
 *  Compute the current focal point. \par
 *    \code
 *    SbRotation orientation = camera.orientation.getValue();
 *    SbVec3f offset = orientation.multVec( new SbVec3f(0,0,-1) );
 *    offset.multiply( camera.focalDistance.getValue() );
 *    SbVec3f focalPt = camera.position.getValue().plus( offset );
 *    \endcode
 *  \endif
 *
 * @SEE_ALSO
 *    SoOrthographicCamera,
 *    SoPerspectiveCamera,
 *    SoCameraKit,
 *    SoCameraInteractor
 *
 */
class INVENTOR_API SoCamera : public SoNode {

  SO_NODE_ABSTRACT_HEADER(SoCamera);
  SO_PIMPL_PUBLIC_HEADER(SoCamera);

 public:
   /** Viewport mapping */
  enum ViewportMapping {
    /**
     *  Crops the viewport within the current window, so that the aspect ratio matches
     * that of the camera. As the window size changes, the aspect ratio remains
     * unchanged. The cropped region is drawn as a filled gray area.
     */
    CROP_VIEWPORT_FILL_FRAME = 0,
    /**
     *  Crops the viewport, but draws a thin frame around the viewport
     */
    CROP_VIEWPORT_LINE_FRAME = 1,
    /**
     *  Crops the viewport, but gives no visual feedback as to the viewport dimensions
     * within the window
     */
    CROP_VIEWPORT_NO_FRAME   = 2,

    /**
     *  Adjusts the camera aspect ratio and height to make it fit within the given
     * window. (The camera's fields are not affected, just the values sent to the
     * graphics library.)
     */
    ADJUST_CAMERA            = 3,
    /**
     *  Do nothing. Camera image may become stretched out of proportion
     */
    LEAVE_ALONE              = 4
  };

  // NOTE: These fields are here so that all camera subclasses do
  // not have to define them separately. However, all subclasses
  // DO have to set up field data with these fields in them to make
  // things work properly.

  // Fields
  /**
   * Defines how to map the rendered image into the current viewport, when the aspect
   * ratio of the camera differs from that of the viewport.
   * @useenum{ViewportMapping}. Default is ADJUST_CAMERA.
   */
  SoSFEnum              viewportMapping;
  /**
   * The location of the camera viewpoint.
   *
   */
  SoSFVec3f             position;
  /**
   * The orientation of the camera viewpoint, defined as a rotation of the viewing
   * direction from its default (0,0,-1) vector.
   *
   */
  SoSFRotation  orientation;
  /**
   * The ratio of camera viewing width to height. This value must be greater than
   * 0.0. There are several standard camera aspect ratios defined in SoCamera.h.
   *
   */
  SoSFFloat             aspectRatio;
  /**
   * The distance from the camera viewpoint to the near clipping plane.
   *
   */
  SoSFFloat             nearDistance;
  /**
   * The distance from the camera viewpoint to the far clipping plane.
   *
   */
  SoSFFloat             farDistance;
  /**
   * The distance from the viewpoint to the point of focus. This is typically ignored
   * during rendering, but may be used by some viewers to define a point of
   * interest.
   *
   */
  SoSFFloat       focalDistance;

  /**
   * Amount of blur to apply to out of focus areas.
   *
   * This value virtually corresponds to the aperture of the camera, and is
   * computed as the inverse of a physical f-stop value of the camera.
   *
   * For instance, a value of 0.0 corresponds to an infinite f-stop, which means
   * that all of the scene is in focus. Conversely, a value of 0.5
   * corresponds to an f-stop value of 2, and only a small area of the
   * scene in focus.
   *
   * The #focalDistance field defines the distance to the plane of focus.
   *
   * The focal length of the camera is defined as a fraction of
   * (#farDistance - #nearDistance).
   *
   * Default is 0.0
   *
   * @FIELD_SINCE_OIV 2025.1
   */
  SoSFFloat blur;

  /**
   *
   * Modes that define how to map the color values in the framebuffer
   * to fir them in the [0 - 1] range.
   */
  enum ExposureMode
  {
    /**
     * Keep colors as is in the framebuffer.
     * Set RGB components greater than 1.0 as equal to 1.0.
     */
    LDR_LINEAR,

    /**
     * Apply a fixed curve function to color values to fit them in
     * the [0 - 1] range.
     *
     * The function is smooth so as to eliminate the artifacts that one obtains
     * with clamping.
     */
    HDR_NEUTRAL,
  };

  /**
   *
   * Defines how to map the color values in the framebuffer to fit them in
   * the [0 - 1] range.
   *
   * See #ExposureMode for a list of different modes that you can use with this
   * field.
   *
   * Notes:
   * - the exposure mapping is an operation performed directly on the values of
   *   the current color buffer.
   * - if this camera node is part of a SoRenderToTextureProperty scene graph,
   *   the mapping operation is performed on the color texture target of the
   *   SoRenderToTextureProperty. Otherwise, it is performed on the default
   *   color buffer of OpenInventor.
   * - the current viewport region is taken into account, so that only the
   *   pixels inside the region are impacted by the exposure mapping.
   *
   * @useenum{ExposureMode}
   *
   * Default is LDR_LINEAR
   *
   * @FIELD_SINCE_OIV 2025.1
   */
  SoSFEnum exposureMode;

  /**
   *
   * Adjusts the exposure of the rendering.
   *
   * Increasing this value by 1 doubles the exposure, while decreasing by 1
   * halves the exposure.
   *
   * Default is 0.0
   *
   * @FIELD_SINCE_OIV 2025.1
   */
  SoSFFloat exposureCompensation;

  /**
   * Sets the orientation of the camera so that it points toward the given target
   * point while keeping the "up" direction of the camera parallel to the positive
   * y-axis. If this is not possible, it uses the positive z-axis as "up."
   */
  void          pointAt(const SbVec3f &targetPoint);

  /**
   * Scales the height of the camera. Perspective cameras scale their
   * @B heightAngle @b fields, and orthographic cameras scale their @B height @b
   * fields.
   */
  virtual void  scaleHeight(float scaleFactor) = 0;

  /**
   * Returns a view volume object, based on the camera's viewing parameters. @BR
   * This object can be used, for example, to get the view and projection matrices,
   * to project 2D screen coordinates into 3D space and to project 3D coordinates
   * into screen space.
   *
   * If the useAspectRatio parameter is 0.0 (the default), the camera uses the
   * current value of the #aspectRatio field to compute the view volume.
   *
   * NOTE: In ADJUST_CAMERA mode (the default), the view volume returned when
   * useAspectRatio = 0, is not (in general) the actual view volume used for
   * rendering.  Using this view volume to project points will not (in general)
   * produce the correct results.
   *
   * This is because, in ADJUST_CAMERA mode, Inventor automatically modifies the
   * view volume to match the aspect ratio of the current viewport.  This avoids 
   * the distortion that would be caused by "stretching" the view volume when it
   * is mapped into the viewport. However the view volume values are not changed,
   * only the values passed to OpenGL. In order to get the modified values (i.e., 
   * the actual view volume used for rendering) you must pass the actual viewport
   * aspect ratio to getViewVolume. You can get the current viewport from the
   * renderArea or viewer object that contains the Open Inventor window.
   *
   * Also note that in ADJUST_CAMERA mode, when the viewport aspect ratio is
   * less than 1, Open Inventor automatically scales the actual rendering view 
   * volume by the inverse of the aspect ratio (i.e. 1/aspect). The getViewVolume 
   * method does not automatically apply this adjustment. So a correct query of
   * the actual rendering view volume can be done like this:
   *
   * \par
   *   \if_cpp
   *     \code
   *       // Given a viewer object, get the actual rendering view volume
   *       float     aspect = viewer->getViewportRegion().getViewportAspectRatio();
   *       SoCamera* camera = viewer->getCamera();
   *       SbViewVolume viewVol = camera->getViewVolume( aspect );
   *       if (aspect < 1)
   *         viewVol.scale( 1 / aspect );
   *     \endcode
   *   \endif
   *   \if_dotnet
   *     \code
   *       // Given a viewer object, get the actual rendering view volume
   *       float    aspect = viewer.GetViewportRegion().ViewportAspectRatio;
   *       SoCamera camera = viewer.GetCamera();
   *       SbViewVolume viewVol = camera.GetViewVolume( aspect );
   *       if (aspect < 1)
   *         viewVol.Scale( 1 / aspect );
   *     \endcode
   *   \endif
   *   \if_java
   *     \code
   *       // Given a viewer object, get the actual rendering view volume
   *       SoCamera camera = viewer.getArea().getCamera();
   *       float aspect    = viewer.getArea().getViewportRegion().getViewportAspectRatio();
   *       SbViewVolume viewVol = camera.getViewVolume( aspect );
   *       if (aspect < 1)
   *         viewVol.scale( 1 / aspect );
   *     \endcode
   *   \endif
   */
  virtual SbViewVolume getViewVolume(float useAspectRatio = 0.0) const = 0;

  /** Computes a view volume from the given parameters. */
  virtual SbViewVolume getViewVolume(float aspectRatio,
                                     float nearDist,
                                     float farDist,
                                     SbVec3f position,
                                     SbRotation rotation) const = 0;

  /**
   * Sets the camera to view the scene rooted by the given node.
   * The near and far clipping planes will be positioned slack bounding
   * sphere radii away from the bounding box's center. A value of 1.0 will make the
   * near and far clipping planes the tightest around the bounding sphere.
   *
   * The node applies an SoGetBoundingBoxAction to the scene graph to get the
   * bounding box of the entire scene. The bounding box will only include shapes 
   * that are actually traversed. For example the bounding box will not include 
   * shapes under an SoSwitch with whichChild set to SO_SWITCH_NONE. The action 
   * does not consider the visibility of shapes that are traversed. In other words
   * the bounding box will include shapes that are invisible (SoDrawStyle), shapes
   * that are clipped (SoClipPlane), etc.  Use an SoBBox node to exclude shapes 
   * from the bounding box computation.  Bounding boxes are automatically cached at
   * SoSeparator nodes, so getting the bounding box is very fast when the scene
   * graph has not been changed.
   *
   * \if_cpp
   * Warning: @BR
   * The SoGetBoundingBoxAction will call ref() and unref() on the specified node.
   * If the node's reference count before calling viewAll() is zero (the default),
   * the call to unref() will cause the node to be destroyed.
   * \endif
   */
  void          viewAll(SoNode *sceneRoot,
                        const SbViewportRegion &vpRegion,
                        float slack = 1.0);
  /**
   * Sets the camera to view the scene defined by the
   * given path. The near and far clipping planes will be positioned slack bounding
   * sphere radii away from the bounding box's center. A value of 1.0 will make the
   * near and far clipping planes the tightest around the bounding sphere.
   *
   * See note about bounding boxes in the sceneRoot version of this method.
   */
  void          viewAll(SoPath *path,
                        const SbViewportRegion &vpRegion,
                        float slack = 1.0);

  /**
   * Sets the camera to view the region defined by the given bounding box.
   * The near and far clipping planes will be positioned the radius of the bounding
   * sphere away from the bounding box's center.
   *
   * See note about bounding boxes in the sceneRoot version of this method.
   */
  void          viewAll(const SbBox3f& bbox, 
                        const SbViewportRegion &vpRegion);

  /**
   * Returns the viewport region this camera would use to render into the given
   * viewport region, accounting for cropping.
   */
  SbViewportRegion      getViewportBounds(const SbViewportRegion &region) const;

#if SoDEPRECATED_BEGIN(9700)

  /**
   * Sets the stereo offset (the distance of each eye from the camera position).
   * The right eye is moved plus offset and the left eye is moved minus offset.
   * Default is 0.7. The default can be set using OIV_STEREO_OFFSET environment variable.
   */
  SoDEPRECATED_METHOD_NOWARN( 9700, "Use SoStereoCamera and offset field instead." )
  virtual void  setStereoAdjustment(float adjustment);

  /**
   * Queries the stereo offset.
   */
  SoDEPRECATED_METHOD_NOWARN( 9700, "Use SoStereoCamera and offset field instead." )
  float getStereoAdjustment() const;


  /**
  * Specifies if stereo adjustments are absolute. FALSE by default.
  *
  * The default non-absolute mode allows the stereo settings to be valid over a range
  * of different view volume settings. If you chose absolute mode, you are responsible
  * for modifying the stereo settings (if necessary) when the view volume changes.
  *
  * When absolute mode is TRUE, stereo offset and balance are used as shown in the
  * following pseudo-code for the right eye view:
  * \code
  * StereoCameraOffset = getStereoAdjustment();
  * FrustumAsymmetry   = getBalanceAdjustment();
  *
  * glTranslated (-StereoCameraOffset, 0, 0);
  * glFrustum (FrustumLeft + FrustumAsymmetry, FrustumRight + FrustumAsymmetry,
  *            FrustumBottom, FrustumTop, NearClipDistance, FarClipDistance);
  * \endcode
  * The left eye view is symmetric.
  *
  * When absolute mode is FALSE, stereo offset and balance are used as shown in
  * the following pseudo-code for the right eye view:
  *
  * Xrange is right minus left (i.e., first two arguments of glFrustum) and
  * multiply that difference by the ratio of the distance to the desired plane
  * of zero parallax to the near clipping plane distance.
  * \code
  * StereoCameraOffset   = Xrange * 0.035 * getStereoAdjustment();
  * FrustumAsymmetry     = -StereoCameraOffset * getBalanceAdjustment();
  * ZeroParallaxDistance = (NearClipDistance + FarClipDistance)/0.5;
  *
  * FrustumAsymmetry *= NearClipDistance / ZeroParallaxDistance;
  *
  * glTranslated (-StereoCameraOffset, 0, 0);
  * glFrustum (FrustumLeft + FrustumAsymmetry, FrustumRight + FrustumAsymmetry,
  *            FrustumBottom, FrustumTop, NearClipDistance, FarClipDistance);
  * \endcode
  * The left eye view is symmetric.
  *
  * Not virtual pure for compatiblity reasons.
  */
  SoDEPRECATED_METHOD_NOWARN( 9700, "Use SoStereoCamera and absoluteAdjustments field instead." )
  virtual void setStereoAbsoluteAdjustments( SbBool absolute );

  /**
   * Queries the stereo absolute adjustment state.
   */
  SoDEPRECATED_METHOD_NOWARN( 9700, "Use SoStereoCamera and absoluteAdjustments field instead." )
  SbBool getStereoAbsoluteAdjustment() const;


  /**
   * Sets the stereo balance (the position of the zero parallax plane) and specifies whether
   * the balance value is defined as a fraction of the camera near distance.
   *
   * Note: Since the projection matrix always depends on the camera's near plane, in
   * some cases it may be necessary to detect changes to the camera near plane and
   * adjust by setting a new stereo balance value.  Open Inventor will make these
   * adjustments automatically if the @B nearFrac @b parameter is set to TRUE.
   * In this case the stereo balance value is defined as a fraction of the camera
   * near distance.
   *
   * Default balance is 1.0. The default can be set using the OIV_STEREO_BALANCE environment variable.
   * Default nearFrac is FALSE. The default can be set using the OIV_STEREO_BALANCE_NEAR_FRAC environment variable.
   */
  SoDEPRECATED_METHOD_NOWARN( 9700, "Use SoStereoCamera and balance / balanceNearFrac fields instead." )
  virtual void  setBalanceAdjustment(float adjustment, SbBool nearFrac = false);

  /**
   * Queries the parallax balance.
   */
  SoDEPRECATED_METHOD_NOWARN( 9700, "Use SoStereoCamera and balance field instead." )
  float getBalanceAdjustment() const;


  /**
   * Returns TRUE if the stereo balance adjustement is defined as a fraction
   * of the camera near distance.
   */
  SoDEPRECATED_METHOD_NOWARN( 9700, "Use SoStereoCamera and balanceNearFrac field instead." )
  SbBool isBalanceAdjustmentNearFrac() const;


  /**
   * Allows the camera to render in stereo. Default value is TRUE.
   */
  SoDEPRECATED_METHOD_NOWARN( 9700, "Use SoStereoCamera instead." )
  virtual void allowStereo(SbBool);

#endif /** @DEPRECATED_END */

  /** Stereo mode */
  enum StereoMode {
    /**
     *  Monoscopic (i.e., non-stereo) viewing
     */
    MONOSCOPIC,
    /**
     *  Left eye view
     */
    LEFT_VIEW,
    /**
     *  Right eye view
     */
    RIGHT_VIEW
  };

  /**
   * Sets the stereo mode. If the previous mode was MONOSCOPIC, the current
   * #position is saved. If the parameter mode is MONOSCOPIC, the saved value
   * is restored in #position. When rendering in stereo mode, set left view,
   * render, set right view, render and then restore to monoscopic mode.
   */
  void  setStereoMode(StereoMode mode);

  /**
   * Queries the stereo mode.
   */
  StereoMode    getStereoMode() const;

SoEXTENDER public:
  virtual void  doAction(SoAction *action);
  virtual void  callback(SoCallbackAction *action);
  virtual void  GLRender(SoGLRenderAction *action);
  virtual void  getBoundingBox(SoGetBoundingBoxAction *action);
  virtual void  handleEvent(SoHandleEventAction *action);
  virtual void  rayPick(SoRayPickAction *action);
  virtual void  getPrimitiveCount(SoGetPrimitiveCountAction *action);

 SoINTERNAL public:
  static void           initClass();
  static void           exitClass();

  // Set a callback at the end of SoCamera::GLRender().
  static void   setPostCallback( SoCameraCB *f ) ;

  static SbVec3f fromObjectSpace(const SbVec3f &vector, const SbMatrix &matrix, const SbViewportRegion &vpr);
  static SbVec3f toObjectSpace(const SbVec3f &pixel, const SbMatrix &matrix, const SbViewportRegion &vpr);

  /** Return adjusted viewvolume (ie: the one used by inventor) */
  SbViewVolume getViewVolume(SoState *state);

  /** Set to true if the camera is used for a shadowmap render */
  void setLightCamera(bool flag);

  /** True if the camera is used for a shadowmap render */
  bool isLightCamera() const;

  /** return last viewport region when traversed
   * used internally for highlight action.
   */
  const SbViewportRegion &getLastViewportRegion() const;

  /** Return focal point position of the camera */
  SbVec3f getFocalPoint() const;

 protected:
  /** Constructor. Abstract */
  SoCamera();

  /** Destructor. */
  virtual ~SoCamera();
  virtual float getEyeOffset( SoState* state = NULL );
  virtual void updateStereoAdjustments( SoState* state = NULL );


  /**
   * Subclasses must define this method, which changes the position
   * and clipping planes of a camera to view the passed bounding box
   * using the given aspect ratio, without changing the height
   * scaling.
   */
  virtual void viewBoundingBox(const SbBox3f &box, float aspect, float slack) = 0;

  /**
   * Returns amount to jitter camera in normalized device
   * coordinates (after projection) for anti-aliasing during
   * multi-pass rendering
   */
  virtual void jitter(int numPasses, int curPass, const SbViewportRegion &vpReg, SbVec3f &jitterAmount) const;

private:
  void commonConstructor();
};

//////////////////////////////////////////////////////////////////////////////
//
// Commonly used camera aspect ratios
//

#define SO_ASPECT_SQUARE        1.00         /* Square (1.000)               */
#define SO_ASPECT_VIDEO         1.333333333  /* Video (0.75)                 */
#define SO_ASPECT_35mm_ACADEMY  1.371        /* 35mm, Academy ap (.72939460) */
#define SO_ASPECT_16mm          1.369        /* 16mm cinema (.730460189)     */
#define SO_ASPECT_35mm_FULL     1.33333      /* 35mm cinema, full ap (0.75)  */
#define SO_ASPECT_70mm          2.287        /* 70 mm unsqueezed (.43725404) */
#define SO_ASPECT_CINEMASCOPE   2.35         /* Cinemascope (.425531914)     */
#define SO_ASPECT_HDTV          1.777777777  /* HDTV (16:9)                  */
#define SO_ASPECT_PANAVISION    2.361        /* Panavision (.423549343)      */
#define SO_ASPECT_35mm          1.5          /* 35mm still camera (.666666)  */
#define SO_ASPECT_VISTAVISION   2.301        /* Vistavision (.434593654)     */


#ifdef _WIN32
#pragma warning(pop)
#endif

#endif /* _SO_CAMERA_ */

