/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Nick Thompson (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2021 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_CUBE_
#define  _SO_CUBE_

#include <Inventor/SbVec.h>
#include <Inventor/SbBox.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/nodes/SoShape.h>
#include <Inventor/SbPImpl.h>

SO_PIMPL_PUBLIC_DECLARATION( SoCube );

/**
 * Cube shape node.
 * 
 * @ingroup ShapeNodes
 * 
 * @DESCRIPTION
 *   This node represents a cuboid aligned with the coordinate axes. By default, the
 *   cube is centered at (0,0,0) and measures 2 units in each dimension, from -1 to
 *   +1. The cube is transformed by the current cumulative transformation and is
 *   drawn with the current lighting model, drawing style, material, and geometric
 *   complexity.
 *   
 *   If the current material binding is PER_PART, PER_PART_INDEXED, PER_FACE, or
 *   PER_FACE_INDEXED, materials will be bound to the faces of the cube in this
 *   order: front, back, left, right, top, and bottom.
 *   
 *   Textures are applied individually to each face of the cube; the entire texture
 *   goes on each face. On the front, back, right, and left sides of the cube, the
 *   texture is applied right side up. On the top, the texture appears right side up
 *   when the top of the cube is tilted toward the camera. On the bottom, the texture
 *   appears right side up when the top of the cube is tilted away from the camera.
 *
 *   OpenGL vertex arrays are used if they are available and if the
 *   environment variable IV_NO_VERTEX_ARRAY is not set.
 *
 * @FILE_FORMAT_DEFAULT
 *    Cube {
 *    @TABLE_FILE_FORMAT
 *       @TR width    @TD 2
 *       @TR height   @TD 2
 *       @TR depth    @TD 2
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction @BR
 *        Draws cube based on the current coordinates, materials, drawing style, and so
 *        on.
 * 
 *    SoRayPickAction @BR
 *        Intersects the ray with the cube. The face of the cube that was picked is
 *        available from the SoCubeDetail.
 * 
 *    SoGetBoundingBoxAction @BR
 *        Computes the bounding box that encloses the cube.
 * 
 *    SoCallbackAction @BR
 *        If any triangle callbacks are registered with the action, they will be invoked
 *        for each successive triangle that approximates the cube.
 * 
 * 
 * @SEE_ALSO
 *    SoCone,
 *    SoCubeDetail,
 *    SoCylinder,
 *    SoFullSceneAntialiasing,
 *    SoSphere
 * 
 * 
 */
class INVENTOR_API SoCube : public SoShape
{
  SO_NODE_HEADER(SoCube);
  SO_PIMPL_PUBLIC_HEADER( SoCube );

 public:
  // Fields
  /**
   * Size in the x dimension.
   */
  SoSFFloat width;          // Size in x dimension
  /**
   * Size in the y dimension.
   */
  SoSFFloat height;         // Size in y dimension
  /**
   * Size in the z dimension.
   */
  SoSFFloat depth;

  /**
   * Creates a cube node with default settings.
   */
  SoCube();

 SoEXTENDER public:
  // Implements actions
  virtual void GLRender( SoGLRenderAction* action );
  virtual void rayPick( SoRayPickAction* action );
  virtual void getPrimitiveCount( SoGetPrimitiveCountAction* action );

  // Computes bounding box of cube
  virtual void computeBBox( SoAction* action, SbBox3f& box, SbVec3f& center );
  
 SoINTERNAL public:

  /** register node in database */
  static void initClass();
  /** unregister node from database */
  static void exitClass();

  // Renders or picks cube representing given bounding box. These
  // are used by SoShape to implement BOUNDING_BOX complexity.
  void GLRenderBoundingBox( SoGLRenderAction* action, const SbXfBox3d& bbox );
  void GLRenderBoundingBox( SoGLRenderAction* action, const SbBox3f& bbox );

  void rayPickBoundingBox( SoRayPickAction* action, const SbBox3f& bbox );

  virtual SoChildList* getChildren() const;

protected:
  // Generates triangles representing a cube
  virtual void generatePrimitives( SoAction* action );

  // Overrides standard method to create an SoCubeDetail instance
  virtual SoDetail* createTriangleDetail( SoRayPickAction* action,
                                          const SoPrimitiveVertex* v1,
                                          const SoPrimitiveVertex* v2,
                                          const SoPrimitiveVertex* v3,
                                          SoPickedPoint* pp );

  virtual ~SoCube();

  using SoShape::GLRenderBoundingBox;
};

#endif /* _SO_CUBE_ */
