/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef _SO_DEPTH_BUFFER_H_
#define _SO_DEPTH_BUFFER_H_

#include <Inventor/SbPImpl.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/elements/SoDepthBufferElement.h>

#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFVec2f.h>


class SoGLRenderAction;

SO_PIMPL_PUBLIC_DECLARATION( SoDepthBuffer )

/**
 * @VSGEXT Depth buffer parameters node.
 *
 * @ingroup PropertyNodes
 *
 * @DESCRIPTION
 *   This class specifies the depth buffer parameters used for rendering,
 * including:
 * - Depth buffer test enabled,
 * - Depth buffer writing enabled,
 * - Depth buffer range, and
 * - Depth comparison function.
 *
 * It can also be used to clear the depth buffer by setting the
 * #clearBuffer field to true.
 *
 * To control whether frame buffer color components are written or not,
 * see SoColorMask.
 *
 * @B Limitations:@b
 * - Transparency: @BR
 *   The "depth peeling" transparency types (SORTED_PIXEL) do not work correctly
 *   if this node is used to change depth buffer settings.
 *
 * - Test function @BR
 *   By default SoDepthBuffer sets the OpenGL depth buffer test function to LESS.
 *   (Meaning that only fragments with a depth value strictly less than the current
 *   depth buffer value will be rendered.) This is NOT the same as the Open Inventor default. @BR @BR
 *   By default Open Inventor sets the depth buffer test function to LEQUAL and
 *   this value is recommended for most applications.  Some Open Inventor features,
 *   for example SoGradientBackground, will not work correctly with the depth buffer
 *   test function set to LESS.
 *
 * @FILE_FORMAT_DEFAULT
 *    DepthBuffer {
 *      @TABLE_FILE_FORMAT
 *       @TR test  @TD TRUE
 *       @TR write @TD TRUE
 *       @TR function @TD LESS
 *       @TR range @TD (0.0,1.0)
 *       @TR clearBuffer @TD FALSE
 *      @TABLE_END
 *    }
 *
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction
 *
 * @SEE_ALSO
 *       SoPolygonOffset, SoDepthOffset, SoColorMask, SoGLRenderAction
 *
 *
 */
class INVENTOR_API SoDepthBuffer : public SoNode
{
  SO_NODE_HEADER(SoDepthBuffer);
  SO_PIMPL_PUBLIC_HEADER( SoDepthBuffer )

public:
  /**
   * Depth function mode as defined by OpenGL.
   */
  enum DepthWriteFunction {
    /** Never passes. */
    NEVER = SoDepthBufferElement::NEVER,
    /** Passes if the	incoming depth value is	less than the stored depth value. */
    LESS = SoDepthBufferElement::LESS ,
    /** Passes if the	incoming depth value is	equal to the stored depth value. */
    EQUAL = SoDepthBufferElement::EQUAL ,
    /** Passes if the	incoming depth value is	less than or equal	to the stored depth value. */
    LEQUAL = SoDepthBufferElement::LEQUAL,
    /** Passes if the	incoming depth value is greater than the stored depth	value. */
    GREATER = SoDepthBufferElement::GREATER,
    /** Passes if the	incoming depth value is	not equal	to the stored depth value. */
    NOTEQUAL = SoDepthBufferElement::NOTEQUAL,
    /** Passes if the	incoming depth value is greater than or equal	to the stored depth value. */
    GEQUAL = SoDepthBufferElement::GEQUAL,
    /** Always passes. */
    ALWAYS = SoDepthBufferElement::ALWAYS
  };

  /**
   * Enable depth buffer testing (glEnable). Default is TRUE.
   * 
   * Note that disabling depth testing will prevent the
   * depth buffer from being updated even if depth buffer
   * writing is set to TRUE.
   */
  SoSFBool test;

  /**
   * Enable depth buffer writing (glDepthMask). Default is TRUE.
   */
  SoSFBool write;

  /**
   * Depth comparison function to use (glDepthFunc). 
   * @useenum{DepthWriteFunction}. Default is LESS.
   */
  SoSFEnum function;

  /**
   * Value range for the depth buffer (glDepthRange). Default is [0.0-1.0].
   * The range will be clamped to [0.0-1.0] by OpenGL.
   */
  SoSFVec2f range;

  /**
   * If true, the depth buffer is cleared when the node is traversed.
   * The extent of the buffer that is actually cleared may be limited 
   * using an SoViewportClipping node.
   * Default is FALSE.
   *
   * Note: Be careful when using this feature along with a transparency mode other than SoGLRenderAction::NO_SORT
   * because the objects rendering order might be different from their ordering in the scene graph.
   *
   * @FIELD_SINCE_OIV 8.6
   */
  SoSFBool clearBuffer;

  /** Constructor */
  SoDepthBuffer();

  /**
   * @copydoc SoNode::setOverride.
   */
  inline virtual void setOverride(const SbBool state)
  { override.setValue(state); }

  /**
   * @copydoc SoNode::isOverride.
   */
  inline virtual SbBool isOverride() const
  { return override.getValue(); }

SoEXTENDER public:

  /** default actions behavior */
  virtual void doAction(SoAction *action);

  /** @copydoc SoNode::callback */
  virtual void callback(SoCallbackAction* action);

  /** @copydoc SoNode::GLRender */
  virtual void GLRender(SoGLRenderAction * action);

  /** @copydoc SoNode::pick */
  virtual void pick(SoPickAction* action);

SoINTERNAL public:
  /** @copydoc SoNode::initClass. */
  static void initClass();

  /** @copydoc SoNode::exitClass. */
  static void exitClass();

  SoSFBool override;

protected:
  /** Destructor */
  virtual ~SoDepthBuffer();

private:
  void commonConstructor();
};

#endif /* _SO_DEPTH_BUFFER_H_ */


