/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Nicolas DAGUISE (Dec 2008)
**=======================================================================*/


#ifndef SO_DEPTH_OFFSET_H
#define SO_DEPTH_OFFSET_H

#include <Inventor/nodes/SoNode.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/SbMatrix.h>

/**
 * @VSGEXT Property node that applies a depth offset.
 * 
 * @ingroup PropertyNodes
 * 
 * @DESCRIPTION
 *
 * This node can be used to offset shapes that are co-planar, in
 * order to resolve depth buffer artifacts, commonly called z-buffer "stitching".
 *
 * The effect is similar to SoPolygonOffset but this node uses a different 
 * algorithm (based on Lengyel's method from Game Programming Gems and improved by VSG) in which 
 * the projection matrix is modified to offset subsequent geometry. As a result 
 * it has advantages over SoPolygonOffset in some cases. Specifically, 
 * SoDepthOffset applies its offset value to all types of geometry, not just 
 * polygons, and does not require any additional per-vertex calculations on the GPU.
 *
 * The #offset value is accumulated in the traversal state. For example:
 *   @IMAGE depthOffsetState.png
 * Note that if the accumulation of offset values during the traversal is greater than 1,
 * the accumulated offset value is normalized so that, in this case, the offset values 
 * are not absolute but relative to each other.
 *
 * SoDepthOffset has two limitations.  First, since it modifies the projection matrix 
 * during traversal, it may prevent building a render cache for part of the scene graph. 
 * Second, the offset does not take into account the depth slope of the geometry (as
 * SoPolygonOffset does), so a larger offset may be required for geometry that is not
 * perpendicular to the view vector (facing the camera).
 *
 * The presence of a SoDepthOffset node in the scene graph causes depth buffer values
 * to be different, even if 'offset' is equal to 0.
 *
 * The render caching issue is easily handled, by adjusting your scene graph structure, 
 * just as you would for any non-cacheable node, to ensure that the actual shape nodes 
 * are cached even if the parent group node cannot cache: @BR
 *   For example, using this scene graph structure: @BR
 *   @IMAGE depthOffsetCache.png
 *
 *   instead of this one: @BR
 *   @IMAGE depthOffsetNoCache.png
 *
 *   
 * @FILE_FORMAT_DEFAULT
 *    DepthOffset {
 *    @TABLE_FILE_FORMAT
 *       @TR offset   @TD 0.001
 *       @TR on       @TD TRUE
 *    @TABLE_END
 *    }
 *
 * @ACTION_BEHAVIOR
 *        Sets: SoProjectionMatrixElement
 * 
 * @SEE_ALSO
 *    SoPolygonOffset, SoDepthBuffer, SoOverlayGroup
 *
 * 
 */
class INVENTOR_API SoDepthOffset : public SoNode
{
  SO_NODE_HEADER(SoDepthOffset);

public:
  /**
   * Constructor.
   */
  SoDepthOffset();

  /** 
   * Offset to apply to the projection matrix. @BR
   * Positive values move geometry toward the camera (the opposite of SoPolygonOffset).
   * Offset value must be in the range [-0.5 .. 0.5] (normalized depth coordinates).
   * Default is 0.001
   */
  SoSFFloat offset;

  /**
   * Enables depth offset.
   * @FIELD_SINCE_OIV 8.1
   */
  SoSFBool on;


SoINTERNAL public:
  static void initClass();
  static void exitClass();

SoEXTENDER public:
  /** @copydoc SoNode::GLRender */
  virtual void GLRender(SoGLRenderAction* action);
  /** @copydoc SoNode::search */
  virtual void search( SoSearchAction* action );

  /** Use offset of DepthOffsetElement to setup :
   * - normalizationFactor
   * - NDCadjustment
   */
  static void setupGlobalParameters(SoState *state);
  /** Compute matrix NDCadjustment by scaling input matrix on z. */
  static void computeNDCadjustment(SoState *state, SbMatrix& matrix);

protected:
  /** Destructor. */
  virtual ~SoDepthOffset();

// Re-open this scope here to avoid breaking binary compatibility
SoEXTENDER public:
  virtual void doAction( SoAction* action );
  virtual void rayPick( SoRayPickAction* action );

private:
  static SbBool s_compat80;
  static uint32_t instanceCount;
};

#endif //SO_DEPTH_OFFSET_H

/**/


