/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2021 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_ENVIRONMENT_
#define  _SO_ENVIRONMENT_

#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFColor.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFVec3f.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/nodes/SoNode.h>

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoEnvironment
//
//  Node that describes global environmental attributes such as
//  ambient lighting, light attenuation, and fog.
//
//  Ambient lighting is the amount of extra light impinging on each
//  surface point when computing Phong lighting. 
//
//  Light attenuation affects all subsequent lights in a scene. It is
//  a quadratic function of distance from a light source to a surface
//  point. The three coefficients are specified in the attenuation
//  field. Attenuation works only for light sources with a fixed
//  location, such as point and spot lights.
//
//  Fog has one of four types, each of which blends each surface point
//  with the specified fog color. Each type interprets the visibility
//  field to be the distance at which fog totally obscures objects. A
//  visibility value of 0 (the default) causes the SoEnvironment node
//  to set up fog so that the visibility is the distance to the far
//  clipping plane of the current camera.
//
//  Note that this node has effect only during rendering, and that it
//  does not inherit field values from other SoEnvironment nodes.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * Global environment node.
 * 
 * @ingroup LightNodes
 * 
 * @DESCRIPTION
 *   This node describes global environmental attributes such as ambient lighting,
 *   light attenuation, ambient occlusion and fog.
 *   
 *   Ambient light (#ambientIntensity and #ambientColor) is the amount of global light
 *   impinging on each surface point when the lighting model is Phong (see SoLightModel).
 *   The ambient light values effectively specify a minimum brightness for all shapes.
 *   Even if no light source is directly shining on a surface, the ambient component will
 *   light up the surface a little bit to prevent it from being pure black.
 *   
 *   Light attenuation (#attenuation) is the loss of light intensity over distance.
 *   The greater the distance from the light source, the lower the intensity.
 *   Attenuation affects all subsequent Point and Spot lights in a scene (see SoLight).
 *   See #attenuation field for more details.
 *   
 *   Ambient occlusion (#ambientOcclusion) is a rendering effect that simulates how much
 *   of the global ambient light can reach different parts of a shape. Parts that are less
 *   exposed to the ambient light, i.e. more occluded, will appear darker. This effect is
 *   very useful to better discern the shapes of objects. The #ambientOcclusion field
 *   controls this effect for shape nodes other than SoVolumeRendering and point sets.
 *   (although point sets can "generate" ambient occlusion). SoVolumeRendering also
 *   supports ambient occlusion, but it must be enabled using an SoVolumeRenderingQuality node.
 *   @BR @B Ambient occlusion limitations@b:
 *      - Important performance penalty can be experienced on Intel HD and UHD Graphics integrated GPU
 *
 *   @TABLE_0B
 *   @TR No Ambient Occlusion
 *   @TD Ambient Occlusion
 *   @TR @IMAGE AO_DTI_off.png
 *   @TD @IMAGE AO_DTI_on.png
 *   @TR @I Example with a lineset extracted from a <a href="https://fr.wikipedia.org/wiki/Tractographie">DTI</a> @i @TD
 *   @TABLE_END
 *
 *   Fog has one of four types, each of which blends each surface point with the
 *   specified fog color (usually set equal to the rendering window background color).
 *   Each type interprets the #fogVisibility field to be the
 *   distance at which fog totally obscures objects. A fogVisibility value of 0
 *   (the default) causes the SoEnvironment node to set up fog so that the visibility
 *   is the distance to the far clipping plane of the current camera. Using a fog of type HAZE
 *   results in a linear fog between the #fogStart value and the fogVisibility value.
 * 
 * @FILE_FORMAT_DEFAULT
 *    Environment {
 *    @TABLE_FILE_FORMAT
 *       @TR ambientIntensity           @TD 0.2
 *       @TR ambientColor               @TD 1 1 1
 *       @TR attenuation                @TD 0 0 1
 *       @TR ambientOcclusion           @TD FALSE
 *       @TR ambientOcclusionIntensity  @TD 1.0
 *       @TR ambientOcclusionRadius     @TD 4.0
 *       @TR fogType                    @TD NONE
 *       @TR fogColor                   @TD 1 1 1
 *       @TR fogStart                   @TD 0
 *       @TR fogVisibility              @TD 0
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction @BR
 *        Sets the current environment parameters to those specified with this node.
 *        Successive geometries will be rendered using this environment.
 *        Sets: SoEnvironmentElement, SoLightAttenuationElement
 * 
 * @SEE_ALSO
 *    SoLight,
 *    SoLightModel
 * 
 * 
 */
class INVENTOR_API SoEnvironment : public SoNode {

  SO_NODE_HEADER(SoEnvironment);

 public:
   /** Type of fog */
  enum FogType {
    /**
     *  No fog 
     */
    NONE,
    /**
     *  Linear increase in opacity with distance 
     */
    HAZE,
    /**
     *  Exponential increase in opacity 
     */
    FOG,
    /**
     *  Exponential squared increase in opacity 
     */
    SMOKE
  };

  // Fields

  /**
   * Intensity of ambient light (for Phong lighting). Default is 0.2.
   * 
   */
  SoSFFloat           ambientIntensity;
  /**
   * RGB color of ambient lighting (for Phong lighting). Default is 1, 1, 1.
   * 
   */
  SoSFColor           ambientColor;

  /**
   * Squared, linear, and constant light attenuation coefficients (in that order).
   * Default is 0, 0, 1. So, by default, no attenuation is applied.
   *
   * Attenuation only applies to Point and Spot lights. In the physical world attenuation
   * is proportional to 1/d^2, where 'd' is the distance from the surface to the light
   * source. However, this function causes the light to decrease very rapidly, so linear
   * and constant coefficients can also be specified. The attenuation factor applied to
   * the light is 1 / (c0*d^2 + c1*d + c2).
   */
  SoSFVec3f           attenuation;

  /**
   * If set to TRUE, ambient occlusion is activated.
   *
   * Default is FALSE
   *
   * @B Limitation@b:
   *  - Important performance penalty can be experienced on Intel HD and UHD Graphics integrated GPU
   *
   * @FIELD_SINCE_OIV 10.2
   */
  SoSFBool            ambientOcclusion;

  /**
   * Intensity of ambient occlusion: the higher the value, the darker the scene.
   *
   * Default is 1.0.
   *
   * Note: This field is "global", which means that if you have multiple instances of SoEnvironment node,
   * any change in the intensity field of one instance will automatically be applied to the intensity field of the other instances.
   * As a result, the intensity of the ambient occlusion is the same for the whole scene
   *
   * @FIELD_SINCE_OIV 10.10
   */
  SoSFFloat ambientOcclusionIntensity;

  /**
   * Specifies the radius of the "sphere" (in world space) beyond which the ambient occlusion has no more effect.
   *
   * Default is 4.0.
   *
   * Note: This field is "global", which means that if you have multiple instances of SoEnvironment node,
   * any change in the radius field of one instance will automatically be applied to the radius field of the other instances.
   * As a result, the radius of the ambient occlusion is the same for the whole scene.
   *
   * @FIELD_SINCE_OIV 10.10
   */
  SoSFFloat ambientOcclusionRadius;

  /**
   * Type of fog.
   * @useenum{FogType}. Default is NONE.
   */
  SoSFEnum            fogType;
  /**
   * Fog color. Default is 1, 1, 1.
   */
  SoSFColor           fogColor;
  /**
  * Distance at which the linear fog HAZE starts to be applied. Default is 0.
  * Between fogStart and fogVisibility, the HAZE fog is linear,
  * and beyond the fogVisibility distance, the fog totally obscures objects.
  * This field has no effect on other types of fog.
  */
  SoSFFloat           fogStart;
  /**
   * Fog visibility distance, which is the distance at
   * which fog totally obscures objects. Default is 0.
   * A value of 0 (the default) causes the SoEnvironment node to set up fog so
   * that the visibility is the distance to the far clipping plane of the current camera.
   */
  SoSFFloat           fogVisibility;

  /**
   * Creates an environment node with default settings.
   */
  SoEnvironment();

  /**
   * Sets the state of the overridden field.
   * see SoNode::setOverride doc.
   */
  inline virtual void setOverride(const SbBool state)
  { override.setValue(state); }

  /**
   * Returns the state of the overridden field.
   */
  inline virtual SbBool isOverride() const
  { return override.getValue(); }

 SoEXTENDER public:
  // Only rendering is implemented, since this node has no other effects
  virtual void GLRender(SoGLRenderAction *action);
  virtual void callback(SoCallbackAction *action);
  virtual void doAction(SoAction *action);

 SoINTERNAL public:
  static void         initClass();
  static void         exitClass();

  SoSFBool override;

 protected:
  virtual ~SoEnvironment();

};

#endif /* _SO_ENVIRONMENT_ */

