/*==============================================================================
***   THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),    ***
***             AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                ***
***                                                                          ***
*** REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS  ***
*** SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR  ***
*** WRITTEN AUTHORIZATION OF FEI S.A.S.                                      ***
***                                                                          ***
***                       RESTRICTED RIGHTS LEGEND                           ***
*** USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS ***
*** WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN ***
*** SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT ***
*** CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN ***
*** TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.        ***
***                                                                          ***
***                  COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                   ***
***                           MERIGNAC, FRANCE                               ***
***                         ALL RIGHTS RESERVED                              ***
==============================================================================*/

#pragma once

#include <Inventor/nodes/SoNode.h>
#include <Inventor/nodes/SoTextureCubeMap.h>

#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoMFFilePathString.h>
#include <Inventor/fields/SoSFImage.h>

class SoGLRenderAction;
class SoShaderParameter1i;
class SoShaderParameterMatrix3;

/**
 * Environment map node.
 *
 * @ingroup LightNodes
 *
 * @DESCRIPTION
 *   This node describes an environment map lighting attribute.
 *
 *   An environment map is defined by a texture that is used to simulate
 *   the reflection of the environment on the surface of objects.
 *   The node uses a cubemap texture to represent the environment map.
 *   The cubemap texture is a 2D texture array with 6 faces,
 *   one for each side of a cube.
 *
 * There are two ways to send the texture to Open Inventor:
 * - Read the texture from the six image files specified in the #filenames
 * field.
 * - Specify the textures in memory by setting the the image fields
 * (#imagePosX, ...) to contain the texture data.
 *
 * But in all cases, you must specify the images in this order:
 * - imagePosX (left)
 * - imagePosY (bottom)
 * - imagePosZ (back)
 * - imageNegX (right)
 * - imageNegY (top)
 * - imageNegZ (front)
 *
 * The intensity field specifies the intensity of the environment map.
 *
 * The material properties and light model also affect how the reflections are computed.
 * Enabling specular reflections and setting the shininess of the material to a high value
 * (or the roughness to a low value) produces mirror-like reflections.
 *
 * @FILE_FORMAT_DEFAULT
 *    EnvironmentMap {
 *    @TABLE_FILE_FORMAT
 *       @TR filenames                @TD ""
 *       @TR imagePosX                @TD 0 0 0
 *       @TR imageNegX                @TD 0 0 0
 *       @TR imagePosY                @TD 0 0 0
 *       @TR imageNegY                @TD 0 0 0
 *       @TR imagePosZ                @TD 0 0 0
 *       @TR imageNegZ                @TD 0 0 0
 *       @TR intensity                @TD 0.2
 *    @TABLE_END
 *    }
 *
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction @BR
 *        Sets the current environment map parameters to those specified with this node.
 *        Subsequent shapes will be rendered using this environment map.
 *
 * @SEE_ALSO
 *    SoLight,
 *    SoLightModel,
 *    SoEnvironment,
 *    SoTextureCubeMap
 *
 *
 * @NODE_SINCE_OIV 2025.1
 */
class INVENTOR_API SoEnvironmentMap : public SoNode
{
  SO_NODE_HEADER(SoEnvironmentMap);

public:

  /**
   * Names file(s) from which to read the environment map's texture image(s).
   *
   * The images are used to create a cube map, which is used to simulate the reflection
   * of the environment on the surface of an object.
   *
   * The standard image file formats are supported. See SoRasterImageRW for the list.
   * If the first file is an HDRI file, only this file is used since it has all 6 faces encoded.
   * If the filename is not an absolute path name, the list of directories maintained by
   * SoInput is searched. If the texture is not found in any of those directories,
   * then the file is searched for relative to the directory from which the
   * SoEnvironmentMap node was read. For example, if an SoEnvironmentMap node with a filenames of
   * "../tofu.rgb" is read from /usr/people/bob/models/food.iv, then
   * /usr/people/bob/tofu.rgb will be read (assuming tofu.rgb isn't found in the
   * directories maintained by SoInput).
   */
  SoMFFilePathString  filenames;

  /**
   * Contains an in-memory representation of the environment map.
   *
   * The image is set programmatically using the methods provided by SoSFImage.
   */
  SoSFImage imagePosX;

  /**
   * See #imagePosX.
   */
  SoSFImage imagePosY;

  /**
   * See #imagePosX.
   */
  SoSFImage imagePosZ;

  /**
   * See #imagePosX.
   */
  SoSFImage imageNegX;

  /**
   * See #imagePosX.
   */
  SoSFImage imageNegY;

  /**
   * See #imagePosX.
   */
  SoSFImage imageNegZ;

  /**
   * Intensity of the environment. Default is 0.2.
   */
  SoSFFloat intensity;

  SoEnvironmentMap();

  /**
   * @copydoc SoNode::setOverride
   */
  void setOverride(const SbBool state) override
  {
    override.setValue(state);
  }

  /**
   * @copydoc SoNode::isOverride
   */
  SbBool isOverride() const override
  {
    return override.getValue();
  }

SoINTERNAL public:
  SoSFBool display;
  SoSFBool override;

SoEXTENDER public:
  void GLRender(SoGLRenderAction* action) override;

SoINTERNAL public:
  static void initClass();
  static void exitClass();

  void fieldHasChanged(SoField* field) override;

  SoTextureCubeMap* getTexture() const
  {
    return m_envMapTex.ptr();
  }

  static SbMatrix3 getMatrix(SoState* state);

protected:
  virtual ~SoEnvironmentMap();

private:
  static void copySFImage(const SoSFImage& src, SoSFImage& dst);
  void setupInternalFormat();

  SoRef<SoTextureCubeMap> m_envMapTex;
  SoRef<SoNode> m_envMapDisplayScene;
  SoShaderParameter1i* m_isHDRParam;
  SoShaderParameterMatrix3* m_envMapMatrixParam;
};
