/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Fabien ARNAUD (MMM yyyy)
** Modified by : Christophe OGNIER (MMM yyyy)
** Modified by : Daniel LICHAU (update)        * (MMM yyyy)
**=======================================================================*/

#ifndef _SO_EXTENDED_SELECTION_H_
#define _SO_EXTENDED_SELECTION_H_

//------------------------------------------------------------------------------
//Include files

#include <Inventor/actions/SoCallbackAction.h>
#include <Inventor/nodes/SoSubNode.h>
#include <Inventor/nodes/SoSelection.h>
#include <Inventor/nodes/SoColorIndex.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoMatrixTransform.h>
#include <Inventor/SbViewportRegion.h>
#include <Inventor/STL/vector>
#include <Inventor/misc/SoRef.h>
#include <Inventor/fields/SoSFNode.h>
#include <Inventor/SbEventArg.h>
#include <Inventor/SbEventHandler.h>

//------------------------------------------------------------------------------
// Class declaration
class SoCamera;
class SoSeparator;
class SoCoordinate3;
class SbViewportRegion;
class SoMaterial;
class SoPath;
class LassoRenderer;
class SoOrthographicCamera;
class LassoPointsWrapper;
class MatricesEvaluator;
class SbPlanarConvexPolygon;
class SoExtSelection;

SO_PIMPL_PUBLIC_DECLARATION( SoExtSelection )

/**
 * @VSGEXT Selection node that supports lasso selection.
 *
 * @ingroup GroupNodes
 *
 * @DESCRIPTION
 *   This class extends the functionality of SoSelection to allow the user to select
 *   an object or multiple objects by drawing a lasso around the object(s) to be
 *   selected. The lasso can be represented as a rectangle between the mouse up and
 *   down positions or as a set of lines that the user draws around the objects,
 *   depending on the field #lassoType. When #lassoType is NOLASSO, or
 *   when the lasso is empty (the screen coordinates of the button down and button up
 *   event are the same), SoExtSelection behaves exactly like its parent class
 *   SoSelection.
 *
 *   See SoSelection for more information about managing the selection list
 *   and automatically highlighting the selected shape(s).
 *
 *   Objects are selected if they are surrounded by the lasso or if the lasso
 *   intersects them, depending on the field #lassoPolicy, as follows:
 *
 *   - FULL_BBOX: lasso selects objects whose bounding box is completely contained by
 *   the lasso
 *
 *   - PART_BBOX: lasso selects objects whose bounding box intersects the lasso (or is
 *   contained by the lasso)
 *
 *   - FULL: lasso selects objects whose geometry is completely contained by the
 *   lasso
 *
 *   - PART: lasso selects objects whose geometry intersects the lasso (or is
 *   contained by the lasso)
 *
 *   FULL and PART require more computation and are generally slower.
 *
 *   A lasso mode controls whether only visible shapes (front shapes) are selectable,
 *   by controlling the selection algorithm used, either geometry-based or
 *   pixel-based. A specific lasso mode may also be preferred for performance reasons
 *   depending on the scene.
 *
 *   Primitive details for selected parts of shapes can be obtained through dedicated
 *   callbacks that can be also used to filter the selection. These primitive filter
 *   callbacks can be called for each generated primitive.
 *
 *   When using SoExtSelection node, the \if_dotnet delegate \else callback \endif set by @B setPickFilterCallback @b
 *   will be called only once per selected shape. Only
 *   SoPickedPoint::getPath() makes sense for the picked point passed to the
 *   pick filter \if_dotnet delegate \else callback \endif in this case. Other SoPickedPoint methods may return
 *   undefined results.
 *
 *   Lasso selection can be also be controlled programmatically, without requiring
 *   user interaction.
 *
 *   NOTE: Using the FastEdit mode:
 *
 *   FastEdit mode provides the advantages of overlay mode without the overlay hardware
 *   support requirement. See SoSeparator for more information about the
 *   fast editing feature. FastEdit mode can allow the lasso geometry to be redrawn
 *   at interactive rates even when the underlying scene takes a long time to redraw.
 *   On Windows platforms, it also avoids using GDI to draw the lasso, which conflicts
 *   with the "Aero" user interface mode on Windows Vista.
 *
 *   When the application uses fast editing it must call the function useFastEdit(TRUE) first.
 *   If it doesn't call this function, operations which require fast edit will output a warning
 *   in debug mode.
 *
 *   FastEdit mode also provides some powerful additional features including the option to
 *   provide custom (and dynamic) lasso geometry (see the setSelectionSceneGraph()
 *   method) and the option to set callbacks on mouse events during selection (see the
 *   methods setStartSelectionCallback(), setEndSelectionCallback(),
 *   and setMovingSelectionCallback()).
 *   The geometry in the selection scene graph can be dynamically updated by the
 *   application as the user moves the cursor, using these callbacks.  This can useful to
 *   display dynamic feedback without actually doing selection.  For example to display a
 *   "rubber band" line while positioning annotations or measurements.
 *   However note that the selection scene graph is @I always@i traversed, not just when
 *   the user is selecting. It is the application's responsibility to empty the selection
 *   scene graph (or hide its contents using an SoSwitch) when appropriate.
 *
 *   Note: The geometry in the selection scene graph is not automatically used for selection.
 *   (It may not be appropriate.)  However the application can use the select() method, if
 *   desired, to do a selection using any set of 2D points defining a "lasso".
 *
 *   Finally note that since Open Inventor 8.6, SoScreenDrawer and its specialized
 *   sub-classes provide a more general mechanism for interactively drawing polylines,
 *   rectangles, ellipses, etc in screen space.
 *
 *   Clipping: @BR
 *   SoExtSelection avoids selecting shapes that are culled by active SoClipPlane(s)
 *   in both the ALL_SHAPES and VISIBLE Lasso modes. The user can
 *   disable this, and restore the legacy behavior of SoExtSelection
 *   before Open Inventor 9.4.0, by setting the environment variable
 *   OIV_SELECTION_USE_CLIPPLANES to 0 (see SoPreferences).
 *
 *   Instancing: @BR
 *   When instance nodes of a SoMultipleInstance or a SoMultipleCopy group
 *   are selected, the application can get the instance identifier using the
 *   SoPath method getInstanceIndex().
 *
 *   See SoSelection for additional information about the selection list and code example.
 *
 * @FILE_FORMAT_DEFAULT
 *    ExtSelection {
 *    @TABLE_FILE_FORMAT
 *       @TR boundingBoxCaching   @TD AUTO
 *       @TR renderCulling        @TD AUTO
 *       @TR pickCulling          @TD AUTO
 *       @TR policy               @TD SHIFT
 *       @TR lassoType            @TD NOLASSO
 *       @TR lassoPolicy          @TD FULL_BBOX
 *       @TR lassoMode            @TD ALL_SHAPES
 *       @TR fastEditing          @TD DISABLE
 *    @TABLE_END
 *    }
 *
 * @SEE_ALSO
 *    SoRayPickAction,
 *    SoSelection,
 *    SoWinRenderArea,
 *    SoBoxHighlightRenderAction,
 *    SoHaloHighlightRenderAction,
 *    SoLineHighlightRenderAction
 *
 *
 */
class INVENTOR_API SoExtSelection : public SoSelection {

  SO_NODE_HEADER(SoExtSelection);
  SO_PIMPL_PUBLIC_HEADER( SoExtSelection );

  //============================================================================
 public:

  /** Lasso type */
  enum LassoType {
    /** No lasso (default) */
    NOLASSO,
    /** Lasso */
    LASSO,
    /** Rectangle */
    RECTANGLE,
    /** Ellipse */
    ELLIPSE
  };

  /** Lasso policy */
  enum LassoPolicy {
    /** Full bounding box (default) */
    FULL_BBOX,
    /** Partial bounding box */
    PART_BBOX,
    /** Full */
    FULL,
    /** Partial */
    PART
  };

  /** Lasso mode  */
  enum LassoMode {
    /** Only visible shapes can be selected.
     * NOTE: Transparency is not considered. If a semi-transparent shape is in front of another
     * shape, only the front shape will be selected. Even if the shape in back is "visually"
     * visible, it is considered to be hidden by the front shape.
     */
    VISIBLE_SHAPES,

    /** All shapes can be selected. Even if outside of view volume. (default) */
    ALL_SHAPES
  };

  /**
   * Specifies lasso type (none, lasso, or rectangle).
   * @useenum{LassoType}. Default is NOLASSO.
   */
  SoSFEnum lassoType;

  /**
   * Specifies the lasso policy. See description above.
   * @useenum{LassoPolicy}. Default is FULL_BBOX.
   */
  SoSFEnum lassoPolicy;

  /**
   * Specifies the lasso mode (whether ALL_SHAPES or only VISIBLE_SHAPES can be
   * selected).
   * @useenum{LassoMode}.  Default is ALL_SHAPES.
   * It applies only when the lasso policy is FULL or PART. When the lasso
   * mode is VISIBLE_SHAPES, you will only select @I visible @i shapes. That is, if
   * a shape is within (or partially within) the lasso volume, but is hidden by
   * another shape, it will not be selected. VISIBLE_SHAPES mode provides faster
   * performance than ALL_SHAPES (the default).
   *
   *
   * The algorithm for the ALL_SHAPES mode is based on geometry projection while the
   * VISIBLE_SHAPES mode uses offscreen rendering.
   *
   *
   * When using VISIBLE_SHAPES, none of the primitive selection callback nodes can be
   * called. This means that you cannot retrieve selection details using this mode.
   *
   *
   * Unlike the ALL_SHAPES mode or the picking action, the VISIBLE_SHAPES mode
   * does not allow you to select multiple overlapping shapes. You cannot select a
   * shape obscured by another.
   *
   */
  SoSFEnum lassoMode;

  /**
   * Controls lasso animation. Default is FALSE.
   * @FIELD_SINCE_OIV 10.0
   */
  SoSFBool animateLasso;

  /**
   * Color of lasso. Default value is red (1,0,0).
   * @FIELD_SINCE_OIV 10.0
   */
  SoSFColor lassoColor;

  /**
   * Width of lasso.
   * The default line width is 0, meaning to use the default OpenGL value (1).
   * Line widths greater than zero are considered to be specified in printer's points,
   * where 1 inch = 72 printer's points.  However the default pixels-per-inch value (managed
   * by SbViewportRegion) is also 72.  So effectively line width is specified in pixels
   * unless the application sets a different pixels-per-inch value.
   *
   * Please see the notes in the class description for more information.
   * @FIELD_SINCE_OIV 10.0
   */
  SoSFFloat lassoWidth;

  /**
   * Stipple pattern.
   * This specifies how dashed or dotted lasso will be drawn.
   * The pattern is a 16-bit series of 0s and 1s and is repeated
   * as necessary to stipple a given line. A 1 indicates that
   * drawing occurs, and a 0 that it does not, on a
   * pixel-by-pixel basis, starting with the low-order bits of the pattern.
   * Values can range from 0 (invisible) to 0xffff (solid). Default is 0xe0e0,
   * corresponding to small dashes.
   *
   * The line pattern can be stretched using the #lassoPatternScaleFactor field.
   * @FIELD_SINCE_OIV 10.0
   */
  SoSFUShort lassoPattern;

  /**
   * Stipple pattern scale factor. It stretches the lasso pattern
   * (see #lassoPattern) by multiplying each subseries of consecutive 1s and 0s.
   * Scale factors are clamped to lie between 1 and 255. Default is 1.
   * @FIELD_SINCE_OIV 10.0
   */
  SoSFInt32 lassoPatternScaleFactor;

  /** Return type */
  enum ReturnType {
    /**
     * same as FALSE - Do not select the node, but continue with node's next
     * triangle/line/point
     */
    NOSELECT_CONTINUE,
    /**
     * same as TRUE - Select the node and stop traversing node's
     * triangle/line/point
     */
    SELECT_RETURN,
    /**
     * Select the node but continue for the remaining triangle/line/point
     */
    SELECT_CONTINUE,
    /**
     * Do not select the node and stop traversing node's triangle/line/point
     */
    NOSELECT_RETURN
  };

  /**
   * Constructor.
   */
  SoExtSelection();

  /**
   * Sets whether the lasso should be drawn using fast editing. @BR
   * When the application uses the FastEdit mode it must call this
   * function first before calling any other function in this class.
   * See also SoGLRenderAction::setFastEditSavePolicy for possible limitations.
   * Default is FALSE.
   */
  void useFastEditing(SbBool flg = TRUE);

  /**
   * Gets whether the lasso is drawn using fast editing.
   */
  SbBool isUsingFastEditing();

  /**
   * Returns the scene graph that should be passed to the render area function
   * @B setOverlaySceneGraph @b. Applies only if the lasso is drawn in the overlay
   * plane.
   */
  SoSeparator *getOverlaySceneGraph();

#if SoDEPRECATED_BEGIN(10000)
  /**
   * Sets the lasso color index. Applies only if the lasso is drawn in the
   * overlay plane.
   */
  SoDEPRECATED_METHOD(10000, "No longer supported. Calling this method does nothing.")
  void setOverlayLassoColorIndex(int index);

  /**
   * Gets the lasso color index. Applies only if the lasso is drawn in the
   * overlay plane.
   */
  SoDEPRECATED_METHOD(10000, "No longer supported. Always returns 0.")
  int getOverlayLassoColorIndex();

  /**
   * Sets the lasso color. Applies only if the lasso is not drawn in the overlay
   * plane. This function sets the color in FastEdit mode also.
   * Default is red (1,0,0).
   */
  SoDEPRECATED_METHOD(10000, "Use lassoColor field instead.")
  void setLassoColor(SbColor c);

  /**
   * Gets the lasso color. Applies only if the lasso is not drawn in the overlay
   * plane.
   */
  SoDEPRECATED_METHOD(10000, "Use lassoColor field instead.")
  SbColor getLassoColor();

  /**
   * Sets the lasso line width.
   * This function sets the width in FastEdit mode also. Default is 1.
   */
  SoDEPRECATED_METHOD(10000, "Use lassoWidth field instead.")
  void setLassoWidth(float width);

  /**
   * Gets the lasso line width.
   */
  SoDEPRECATED_METHOD(10000, "Use lassoWidth field instead.")
  float getLassoWidth();

  /**
   * Sets the lasso stipple pattern. Values can range from 0 (invisible) to
   * 0xffff (solid). This specifies how dashed or dotted lasso lines will be drawn.
   * This function actually sets the line pattern in all cases, not just overlay.
   * Default is 0xffff (solid).
   */
  SoDEPRECATED_METHOD(10000, "Use lassoPattern field instead.")
  void setOverlayLassoPattern(unsigned short pattern);

  /**
   * Gets the lasso stipple pattern. Values can range from 0 (invisible) to
   * 0xffff (solid).
   */
  SoDEPRECATED_METHOD(10000, "Use lassoPattern field instead.")
  unsigned short getOverlayLassoPattern();

  /**
   * Sets the the lasso animation flag. This controls whether the lasso is
   * animated (its line pattern changes) or not.  Default is FALSE.
   */
  SoDEPRECATED_METHOD(10000, "Use animateLasso field instead.")
  void animateOverlayLasso(SbBool flg = TRUE);

  /**
   * Gets the the lasso animation flag.
   */
  SoDEPRECATED_METHOD(10000, "Use animateLasso field instead.")
  SbBool isOverlayLassoAnimated();

#endif /** @DEPRECATED_END */

  /**
   * Simulates interactive selection starting from the root node with the given lasso
   * contour, updating the selection list and triggering any associated \if_dotnet delegate \else callback \endif\.
   * Even though the selection operation may not involve a window per se, some nodes
   * need the viewport information to determine their size and placement.
   *
   * NOTE: The root node must be the root of a scene graph that includes the camera node.
   * For example, when using an Open Inventor viewer, use getSceneManager()->getSceneGraph().
   *
   * Setting @I shiftPressed @i true simulates the effect of pressing the shift key while
   * selecting, consistent with the SoSelection::Shift policy.
   *
   * The lasso type can be set using the lassoType field, but the lasso will not be
   * drawn when using this method. The lasso coordinates are given in normalized
   * device coordinates ranging from -1 to +1 in the X and Y directions.
   *
   * Note: When the lassoType is set to LASSO the objects are selected according to the points defined by the 
   * lasso coordinates so the segments between those points are not part of the lasso!.
   * [OIVNET-WRAPPER-ARG IN,IN,ARRAY,IN,IN]
   * [OIVJAVA-WRAPPER-ARG IN,NO_WRAP{lassoCoord->Length},CONST&ARRAY,IN,IN]
   */
  void select(SoNode *root, int numCoord, SbVec2f *lassoCoord,
              const SbViewportRegion &viewport, SbBool shiftPressed);

  /**
   * [OIV-WRAPPER NAME{TriangleCB}]
   * [OIV-WRAPPER-ARG IN,IN,IN,IN,IN]
   */
  typedef SbBool SoExtSelectionTriangleCB(void *userData,
                                          SoCallbackAction *action,
                                          const SoPrimitiveVertex *v1,
                                          const SoPrimitiveVertex *v2,
                                          const SoPrimitiveVertex *v3);
  /**
   * [OIV-WRAPPER NAME{LineSegmentCB}]
   * [OIV-WRAPPER-ARG IN,IN,IN,IN]
   */
  typedef SbBool SoExtSelectionLineSegmentCB(void *userData,
                                             SoCallbackAction *action,
                                             const SoPrimitiveVertex *v1,
                                             const SoPrimitiveVertex *v2);
  /**
   * [OIV-WRAPPER NAME{PointCB}]
   * [OIV-WRAPPER-ARG IN,IN,IN]
   */
  typedef SbBool SoExtSelectionPointCB(void *userData,
                                       SoCallbackAction *action,
                                       const SoPrimitiveVertex *v1);

   /**
    * This is the declaration to use for selection event callback functions. @BR
    * See setStartSelectionCallback(), setEndSelectionCallback(),
    * and setMovingSelectionCallback(). These callbacks are useful to
    * modify the selection scene graph (see setSelectionSceneGraph) to
    * implement dynamically changing selection feedback.
    *
    * Note: These callbacks can only be used in FastEdit mode.
    *
    * @param position The position of the event in pixel coordinates (0,0 is lower left corner).
    * @param scenePosition The position of the event in normalized -1..1 3D coordinates.
    * @param selection The selection node that called the function.
    * @param userData Optional pointer specified when the callback was set.
    *
    * @return If the callback handles the event it must return TRUE, else return FALSE.
    */
   typedef SbBool SelectionEventCallback( const SbVec2s& position, const SbVec3f& scenePosition,
                                          SoExtSelection* selection, void* userData );

  /**
   * Adds a child at the end of the list of children.
   *
   * It redefines the SoSeparator::addChild function just to ensure that the selection
   * scenegraph is always the last child in the list.
   */
  virtual void addChild( SoNode *child );

  /**
  * Structure given to callback when an event is raised.
  * Please refer to #onPreFilter event.
  * [OIV-WRAPPER-CLASS AUTO_PROPERTY]
  */
  struct PreFilterEventArg : public SbEventArg {
  public:

    /** constructor */
    PreFilterEventArg(SoPath* path, SoExtSelection* source)
      :m_path(path), m_source(source), m_isSkipped(false)
    {}

    /** Destructor */
    ~PreFilterEventArg() { }

    /** Returns the ExtSelection. */
    SoExtSelection* getSource() const { return m_source; }

    /** Returns node that will be traversed just after event is called.
    * This is equivalent to calling getPath()->getTail() */
    const SoNode* getNode() const { return m_path->getTail(); }

    /** Returns path to node that will be traversed just after event is called. */
    const SoPath* getPath() const { return m_path; }

    /** Don't traverse the node returned by getNode(). */
    void skipNode()  { m_isSkipped = true; }

    /** Returns if node must be skipped for traversal */
    SbBool isNodeSkipped() const { return m_isSkipped; }

  private:

    SoPath* m_path;
    SoExtSelection* m_source;
    SbBool m_isSkipped;
  };

  /**
  * The ExtSelection will traverse each node of scene graph and test intersection
  * against each nodes.
  * The event is called before doing intersection test and can be used to skip a particular node
  * to avoid doing costly computation.
  * To skip a node, call the method #SoExtSelection::PreFilterEventArg::skipNode() inside your callback.
  */
  SbEventHandler<PreFilterEventArg&> onPreFilter;

  /**
   * Sets the scenegraph to be displayed as "lasso" in fast edit mode.
   *
   * Note: This function is available only in FastEdit mode.
   *
   * The geometry in this scene graph can be dynamically updated by the
   * application as the user moves the cursor. See the methods
   * setStartSelectionCallback(), setMovingSelectionCallback(), etc.
   * This can useful to display dynamic feedback without actually doing
   * selection.  For example to display a "rubber band" line while
   * positioning annotation or measurements.
   *
   * Note: The geometry in this scene graph is not automatically
   * used for selection!  (It may not be appropriate.)  The application
   * can use the select() method, if desired, to do a selection using
   * any set of 2D points defining a "lasso".
   */
  virtual void setSelectionSceneGraph( SoNode* sceneGraph );

  /**
   * Returns the scenegraph displayed as "lasso" in fast edit mode.
   *
   * Note: This function is available only in FastEdit mode, returns NULL otherwise.
   */
  virtual SoNode* getSelectionSceneGraph() const;

  /**
   * Sets the callback called when the mouse button is clicked in selection mode.
   *
   * Note: This function is available only in FastEdit mode.
   */
  virtual void setStartSelectionCallback( SelectionEventCallback* callback, void* userData = NULL );

  /**
   * Sets the callback called when the mouse button is released in selection mode.
   *
   * Note: This function is available only in FastEdit mode.
   */
  virtual void setEndSelectionCallback( SelectionEventCallback* callback, void* userData = NULL );

  /**
   * Sets the callback called when the mouse is moved in selection mode.
   * The callback is only called while the mouse button is down.
   *
   * Note: This function is available only in FastEdit mode.
   */
  virtual void setMovingSelectionCallback( SelectionEventCallback* callback, void* userData = NULL );

  /**
   * Sets the triangle filter \if_dotnet delegate \else callback \endif\.
   * The primitive filter callbacks can be used to retrieve information about the
   * selected primitives of traversed shape nodes - points, line segments, or
   * triangles - and to filter the shape selection based on the selected primitives.
   * These callbacks are called while traversing the scene graph shapes with an
   * SoCallbackAction started by the SoExtSelection node. Whenever a shape primitive
   * is surrounded by the lasso, the \if_dotnet delegate \else callback \endif is called and receives as argument the
   * action and the primitive. Depending on the shape type, the point, line or
   * triangle \if_dotnet delegate \else callback \endif will be called. See SoCallbackAction.
   *
   * When a \if_dotnet delegate \else callback \endif returns TRUE, the corresponding shape is selected according to
   * the selection policy. Then the selection stops processing the current shape's
   * primitives and continues with the next shape in the scene graph. When a \if_dotnet delegate \else callback \endif
   * returns FALSE, the action continues normally to process the remaining shape's
   * primitives, allowing you to filter or collect additional primitives.
   *
   * The primitive filter callbacks are called only when the #lassoPolicy is set
   * to SoExtSelection::PART and the #lassoMode is set to
   * SoExtSelection::ALL_SHAPES.
   *
   * It is the responsibility of the application to manage the primitive selection
   * policy. Exactly like SoSelection, the node selection list can only handle paths
   * to nodes and cannot store primitive detail information.
   */
  void setTriangleFilterCallback(SoExtSelectionTriangleCB *cb,
                                 void *userData = NULL);

  /**
   * Sets the line segment filter callback. See setTriangleFilterCallback()
   * for details.
   */
  void setLineSegmentFilterCallback(SoExtSelectionLineSegmentCB *cb,
                                    void *userData = NULL);
  /**
   * Sets the point filter callback. See setTriangleFilterCallback()
   * for details.
   */
  void setPointFilterCallback(SoExtSelectionPointCB *cb,
                              void *userData = NULL);

  /**
   * Returns the lasso coordinates in device coordinates.
   * [OIVJAVA-WRAPPER-ARG NO_WRAP]
   * [OIV-WRAPPER-RETURN-TYPE ARRAY{numCoords}]
   */
  const SbVec2s *getLassoCoordsDC(int &numCoords);

  /**
   * Returns the lasso coordinates in world coordinates.
   * [OIVJAVA-WRAPPER-ARG NO_WRAP]
   * [OIV-WRAPPER-RETURN-TYPE ARRAY{numCoords}]
   */
  const SbVec3f *getLassoCoordsWC(int &numCoords);

  /**
   * Returns paths list of selected objects.
   */
  const SoPathList &getSelectionPathList() const;

  // Routines to add callbacks for checked primitives
  //(triangles, line segments, and points) for all shapes.

 SoEXTENDER public:
  // Traversal routine for SoHandleEventAction - this will call
  // action->setHandled() if the event is handled
  virtual void handleEvent(SoHandleEventAction *action);
  virtual void search(SoSearchAction *action);
  virtual void GLRender(SoGLRenderAction *action);
  virtual void getBoundingBox(SoGetBoundingBoxAction *action);

 SoINTERNAL public:
  static void           initClass();
  static void           exitClass();

  virtual void notify( SoNotList* list );

  void setSelection( SbBool on );
  SbBool getSelection() const;

  //============================================================================
 protected:

  // Destructor ----------------------------------------------------------------
  // Private to keep people from trying to delete
  // nodes, rather than using the reference count mechanism.
  virtual ~SoExtSelection();

 private:
  void commonConstructor();
};

#endif  // _SO_EXTENDED_SELECTION_H_

