/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2019 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_FACE_SET_
#define  _SO_FACE_SET_

#include <Inventor/SbLinear.h>
#include <Inventor/fields/SoMFInt32.h>
#include <Inventor/nodes/SoNonIndexedShape.h>
#include <Inventor/SbPImpl.h>


//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoFaceSet
//
//  Set of faces. Each face consists of 3 or more vertices, taken in
//  order from the current coordinates. The startIndex field gives the
//  starting coordinate index for the first face. The number of
//  vertices in each face is determined by successive entries in the
//  numVertices field.
//
//////////////////////////////////////////////////////////////////////////////

// This value, when used in the numVertices field, means that the
// corresponding face should use the rest of the vertices
#define SO_FACE_SET_USE_REST_OF_VERTICES        (-1)

class SoNormalBundle;
class SoState;
class SoVolumeFaceSet;
class IShapeVrtxImpl2ShapeVrtxAPI;

SO_PIMPL_PUBLIC_DECLARATION(SoFaceSet)

/**
 * Polygonal face shape node.
 *
 * @ingroup ShapeNodes
 *
 * @DESCRIPTION
 *   This node represents a 3D shape formed by constructing faces (polygons) from
 *   vertices located at the coordinates specified in the #vertexProperty field
 *   (from SoVertexShape), or the current inherited state. For optimal performance,
 *   the #vertexProperty field is recommended.
 *
 *   Coordinates are used in order, starting with #startIndex (zero by default).
 *
 *   If the first value in the #numVertices field is -1 (the default), then all the coordinates
 *   (starting with startIndex) are rendered as a single face.
 *
 *   Otherwise each face uses the number of vertices specified by a value in the #numVertices field.
 *   For example, an SoFaceSet with a #numVertices of [3,4,4] would use the first
 *   three coordinates for the first face, the next four coordinates for the
 *   second face, and the next four coordinates for the third face.
 *   Thus the number of values in the #numVertices field indicates the number of
 *   faces in the set.
 *
 *   SoFaceSet should only be used to render faces with 3 or more vertices. Otherwise the
 *   result is undefined. For example, do not set values in the #numVertices field to 0, 1 or 2.
 *   To render lines or points, use SoLineSet or SoPointSet respectively.
 *
 *   SoFaceSet is a general purpose polygon rendering node.  It can render 
 *   triangles, quadrilaterals, and other types of polygons or any combination thereof.
 *   However:
 *   - For best performance, arrange the faces with all triangles first, then all quadrilaterals, 
 *     then general polygons.
 *
 *   - If the shape has only triangles, use SoTriangleSet instead (simpler and possibly faster).
 *   
 *   SoFaceSet supports both convex and non-convex polygons (unlike OpenGL).
 *   However polygons are assumed to be convex by default. Rendering non-convex
 *   polygons in this mode may produce incorrect results.  Enable non-convex
 *   polygons using the faceType field of an SoShapeHints node.  In this mode
 *   SoFaceSet will automatically convert non-convex polygons, if necessary,
 *   to faces that the hardware can handle.  
 *
 *   SoFaceSet can also directly handle complex polygons with "holes" (interior boundaries).
 *   See the following paragraphs.
 *
 *   If SoShapeHints is not used, or if the SoShapeHints::windingType field is equal to
 *   NO_WINDING_TYPE, SoFaceSet defines a set of faces. It uses the coordinates in order,
 *   starting with #startIndex.  Each face
 *   has a number of vertices specified by a value in the #numVertices field.
 *   For example, an SoFaceSet with #numVertices of [3,4,4] would use
 *   coordinates 1, 2, and 3 for the first face, coordinates 4, 5, 6, and 7 for the
 *   second face, and coordinates 8, 9, 10, and 11 for the third.
 *   The number of values in the #numVertices field indicates the number of
 *   faces in the set.  The default value (-1) means to use all the coordinates
 *   to define a single face.
 *
 *   If SoShapeHints is used with a SoShapeHints::windingType field is different from NO_WINDING_TYPE,
 *   SoFaceSet defines a complex (multi-contour) face.
 *   It uses the coordinates in order, starting with the first one. Each contour has
 *   a number of vertices specified by a value in the #numVertices field.
 *   For example, an SoFaceSet with #numVertices of [3,4,4] would use
 *   coordinates 1, 2, and 3 for the first contour, coordinates 4, 5, 6, and 7 for the
 *   second contour, and coordinates 8, 9, 10, and 11 for the third.
 *   The number of values in the #numVertices field indicates the number of
 *   contours in the complex face.
 *
 *   The coordinates of the face set are transformed by the current cumulative
 *   transformation. The faces are drawn with the current light model and drawing
 *   style (SoDrawStyle).  Diffuse color and opacity may be specified using the
 *   @I orderedRGBA@i field of SoVertexProperty. Diffuse color, transparency and
 *   other color parameters used in the lighting equation may be specified using
 *   SoMaterial.
 *
 *   Treatment of the current material and normal binding is as follows: The PER_PART
 *   and PER_FACE bindings specify a material or normal for each face. The _INDEXED
 *   bindings are equivalent to their non-indexed counterparts. The default material
 *   binding is OVERALL. The default normal binding is PER_VERTEX.
 *
 *   If any normals (or materials) are specified, Open Inventor assumes you provide
 *   the correct number of them, as indicated by the binding. You will see unexpected
 *   results if you specify fewer normals (or materials) than the shape requires.
 *
 *   Lighting is enabled by default (see SoLightModel).  If lighting is enabled, by
 *   default only the "front face" of polygons are lit (the back face will be black).
 *   Which side is the front face is determined by the normal vector, if specified,
 *   else by the ordering of the vertices and the @I vertexOrdering@i field of
 *   SoShapeHints. You can also enable "two sided lighting" using SoShapeHints.
 *
 *   If lighting is enabled and no normals were specified, they will be computed automatically.
 *   The normal generator computes "facet" or "smooth" (or a combination) normal vectors
 *   depending on the @I creaseAngle@i field of SoShapeHints.
 *
 *   Textures may be applied using (for example) SoTexture2. Texture coordinates may be
 *   supplied using SoVertexProperty or SoTextureCoordinate2.  If a texture is applied
 *   and no texture coordinates were specified, they will be computed automatically.
 *   By default the computed coordinates map the texture to the axis-aligned bounding box
 *   of the shape, with the texture S axis corresponding to the longest axis of the box.
 *
 *   @B Warning@b: A face set may not render or pick correctly if any of its polygons are
 *   self-intersecting or non-planar.
 *
 *   Limitations:
 *   - Due to limitations of the OpenGL VBO (vertex buffer object) rendering model, it is not possible
 *     to use VBO rendering (and performance may be lower) if either the normal binding or the material
 *     binding is set to either PER_PART(_INDEXED) or PER_FACE(_INDEXED).
 *
 * @FILE_FORMAT_DEFAULT
 *    FaceSet {
 *    @TABLE_FILE_FORMAT
 *       @TR vertexProperty   @TD NULL
 *       @TR startIndex       @TD 0
 *       @TR numVertices      @TD -1
 *    @TABLE_END
 *    }
 *
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction @BR
 *        Draws faces based on the current coordinates, normals, materials, drawing
 *        style, and so on.
 *
 *    SoRayPickAction @BR
 *        Picks faces based on the current coordinates and transformation. Details about
 *        the intersection are returned in an SoFaceDetail.
 *
 *    SoGetBoundingBoxAction @BR
 *        Computes the bounding box that encloses all vertices of the face set with the
 *        current transformation applied to them. Sets the center to the average of the
 *        coordinates of all vertices.
 *
 *    SoCallbackAction @BR
 *        If any triangle callbacks are registered with the action, they will be invoked
 *        for each successive triangle generated from each face in the set.
 *
 *
 * @SEE_ALSO
 *    SoCoordinate3,
 *    SoDrawStyle,
 *    SoFaceDetail,
 *    SoFullSceneAntialiasing,
 *    SoIndexedFaceSet,
 *    SoShapeHints,
 *    SoTriangleSet,
 *    SoVertexProperty
 *
 */
class INVENTOR_API SoFaceSet : public SoNonIndexedShape
{
  SO_NODE_HEADER(SoFaceSet);
  SO_PIMPL_PUBLIC_HEADER(SoFaceSet)

public:

  /**
   * Number of vertices per face or per contour.
   */
  SoMFInt32 numVertices;

  /**
   * Creates a face set node with default settings.
   */
  SoFaceSet();

SoEXTENDER public:
  // Implement actions
  virtual void GLRender( SoGLRenderAction* action );

  virtual void getPrimitiveCount( SoGetPrimitiveCountAction* action );

  // Generates default normals using the given normal bundle.
  // Returns TRUE.
  virtual void generateDefaultNormals( SoState* state,
                                       const SbVec3f* coords,
                                       int numCoords,
                                       SoNormalBundle* nb,
                                       SbBool storeRef = FALSE );

  /** Disable tangents generation for triangle shapes */
  virtual void shouldGenerateTangents( SoGLRenderAction* /*action*/, const SoShapeStyleElement* /*shapeStyle*/ ) {}

SoINTERNAL public:

  static void  initClass();
  static void  exitClass();

  virtual int getNumVert( SoState* state = NULL );

  virtual int getNumPrim( SoState* state = NULL );

protected:

  //shape must check shape hints in GlRender
  virtual SbBool shouldCheckShapeHints(){ return TRUE; }

  //Test function
  virtual SbBool shapeHintsTest( SoState* state = NULL );

  //Free memory
  virtual void exitRender( SoState* state,
                           int stateFlag,
                           SoTextureCoordinateBundle* tcb,
                           SoTextureCoordinate3Bundle* tc3b);

  virtual void countPrim( SoState* state = NULL );

  // Generates triangles representing faces
  virtual void generatePrimitives( SoAction* action );

  // render function
  void doRendering( SoGLRenderAction* action, const SoShapeStyleElement* shapeStyle );

  // Overrides standard method to create an SoFaceDetail instance
  virtual SoDetail* createTriangleDetail( SoRayPickAction* action,
                                          const SoPrimitiveVertex* v1,
                                          const SoPrimitiveVertex* v2,
                                          const SoPrimitiveVertex* v3,
                                          SoPickedPoint* pp );

  ~SoFaceSet();

  //When tesselating complex polygons, keep the result cached
  virtual bool hasGeneratePrimitiveCache()
  {
    return true;
  }

SoINTERNAL public:

  // set numTris/Quads/Faces to -1, checks SO_USE_REST_OF_VERTICES
  // when notified
  virtual void notify( SoNotList* list );

private:

  // This enum is used to indicate the current material or normal binding
  enum Binding {
    OVERALL, PER_FACE, PER_VERTEX
  };

  void commonConstructor();

};

#endif /* _SO_FACE_SET_ */

