/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Thad Beier (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : TGS (Dec 2004)
** Modified by : MCS (Mmm YYYY)
** Modified by : VSG (Mmm YYYY)
**=======================================================================*/




#ifndef  _SO_FONT_
#define  _SO_FONT_

#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFName.h>
#include <Inventor/fields/SoSFString.h>
#include <Inventor/nodes/SoNode.h>

/**
 * Node that defines font name, size, and type for text.
 * 
 * @ingroup PropertyNodes
 * 
 * @DESCRIPTION 
 *   This node defines the current font type and point size for all subsequent text
 *   shapes in the scene graph. Depending on the font and render style selected, text shapes may be
 *   generated as polygons, lines, or textures. 
 *
 *   Starting with Open Inventor 6.0, text computation is done using the FreeType 2 library (www.freetype.org).
 *   FreeType is a customizable and portable software font engine which is capable of producing
 *   high-quality output. FreeType allows support of more font formats than in previous Open Inventor
 *   versions: TrueType, Type 1, CID-keyed Type 1, CFF, SFNT-based bitmap, X11 PCF (including gz
 *   compressed fonts), Windows FNT, BDF, PFR, dfont, etc.
 *
 *   Starting with Open Inventor 6.0, Unicode text strings are supported. 
 *   Please note that many fonts have limited support for Unicode characters,
 *   especially the Japanese, Chinese, and Korean glyphs.
 *
 *   Some font representing large extended characters and symbols Unicode set can be found:
 *   - Free Quivira font can be found at http://www.quivira-font.com
 *   - Open Source Code2000 font can be found at http://www.fontspace.com/james-kass/code2000
 *   - GNU GPL Unifont can be found at http://www.unifoundry.com/unifont.html
 *
 *   Note: @BR
 *   SoFont will not issue an error message if the specified font is not found at runtime.
 *   It will quietly use the default font (e.g. Times New Roman on Windows) as a fallback.  
 *   Applications can use the static method isAvailableFont() to check if a specific font is available.
 *   Default font can be setup explicitly by enironment variable OIV_FONT_NAME or programmatically
 *   see SoFontNameElement::setDefault() SoFontNameElement::getDefault().
 *
 *
 *   @B Font name:@b
 *
 * @OL 
 * @LI Common Font: @BR
 *
 *      The font name can be set either of two ways:
 *      - specifying directly the font file name (e.g., arial.ttf) with an absolute or relative path, or no path.
 *      - specifying the font family name and font style name with the form "family : style".
 *
 *       Typical family names include:
 *        - Arial @BR
 *        - Courier New @BR
 *        - Times New Roman
 *       
 *       Typical style names include:
 *        - Bold @BR
 *        - Italic @BR
 *        - Bold Italic
 *
 *     Loadable fonts:@BR
 *       The names of all loadable font files available in your environment 
 *       (depending on Open Inventor environment
 *       variables) can be obtained by calling #getAvailableFonts().
 *
 *     Font file location:@BR
 *
 *       To find the font file on your system, Open Inventor uses several rules. 
 *       The following directories are searched (in this order):
 *       - 1 - Current directory
 *       - 2 - $OIV_FONT_PATH
 *       - 3 - System-default font directory
 *       - 4 - $OIVHOME/data/fonts
 *
 *       The system-default font directory depends on the platform:
 *       
 *       @TABLE_0B
 *         @TR Windows  @TD (Retrieved from the registry. Often c:/Windows/Fonts)
 *         @TR Solaris  @TD /usr/openwin/lib/X11/fonts/Type1/outline
 *         @TR AIX      @TD /usr/lib/X11/fonts/Type1/DPS
 *         @TR HP-UX    @TD /usr/lib/X11/fonts/type1.st/typefaces
 *         @TR OSF      @TD /usr/lib/X11/fonts/Type1Adobe
 *         @TR Linux    @TD /usr/share/X11/fonts;/usr/share/fonts;/usr/lib/X11/fonts
 *         @TR MacOS    @TD /Library/Fonts
 *         @TR Other    @TD /usr/lib/DPS/outline/base
 *       @TABLE_END
 *       
 *       Environment variable OIV_FONT_PATH contains a colon-separated or semicolon-separated list of directories
 *       to search for font definition files.
 *
 *  @LI Stroke Fonts: @BR
 *
 *       Stroke fonts, which are drawn with vectors rather than triangles, can provide
 *       higher performance, and also better readability at small sizes. Stroke fonts can
 *       be used with SoText3 and SoAnnoText3, but not with other text nodes, such as
 *       SoAsciiText. Stroke fonts cannot be extruded.
 *       Stroke fonts character code valid values must be in the range [32 .. 127]
 *       
 *       To use a stroke font, specify one of the following stroke font names:
 *       
 *       "OIV_Simplex_TGS" @BR
 *       "OIV_Simplex_Roman" @BR
 *       "OIV_Complex_Roman" @BR
 *       "OIV_Duplex_Roman" @BR
 *       "OIV_Triplex_Roman" @BR
 *       "OIV_Simplex_Script" @BR
 *       "OIV_Complex_Script" @BR
 *       "OIV_Simplex_Greek" @BR
 *       "OIV_Complex_Greek" @BR
 *       "OIV_Gothic_German" @BR
 *       "OIV_Gothic_Italian" @BR
 *       "OIV_Gothic_English" @BR
 *       "OIV_Complex_Cyrillic" @BR
 *       "OIV_Upper_Case_Mathematics" @BR
 *       "OIV_Lower_Case_Mathematics" @BR
 *       "OIV_Music" @BR
 *       "OIV_Meteorology" @BR
 *       "OIV_Symbols" @BR
 *       "OIV_Astrology" @BR
 *       "OIV_Complex_Italic" @BR
 *       "OIV_Triplex_Italic" 
 *       
 *       The following figure shows an assortment of characters from each stroke font.
 *       
 *       @IMAGE stroke_fonts.gif 
 *       
 *       Stroked fonts are subject to the current line attributes, line width and line
 *       pattern, which are specified with SoDrawStyle. 
 *       
 *       By default, the stroke font files are found in $OIVHOME/data/fonts. If you move
 *       them to a different directory, set the environment variable
 *       OIV_STROKE_FONT_PATH to be the path to the new directory.
 *
 *       Open Inventor uses several rules to find the font file on your system.
 *       The following directories are searched (in this order): (NB: If one of these
 *       directories exists, the rest are ignored)
 *       - 1 - $OIV_STROKE_FONT_PATH
 *       - 2 - $OIVHOME/data/fonts
 *       - 3 - ./fonts/
 *
 *   Stroke fonts - Kanji and Katakana:
 *
 *       Additional stroke font files are available for Kanji and Katakana output. To
 *       produce Kanji or Katakana stroke output, set the font to any one of the stroke
 *       font names. Then, for the text string, pass the desired multi-byte character
 *       string (MBCS) to Open Inventor, using the usual techniques. The font file name
 *       is not used except as a flag to indicate that stroke output should be produced. 
 *
 * @ol
 *   @B Font name cache:@b
 *
 *       Making a call to the getAvailableFonts() method causes Open Inventor to build 
 *       an internal font name cache. This cache allows quick access to a newly requested 
 *       font name without requiring Open Inventor to parse all found font files on your system.
 *       In the general case, when the #name field changes, Open Inventor searches for the 
 *       corresponding font file and adds its name to its cache for faster retrieval the next time.
 *       For example, a request to "Arial : Regular" font name will be internally associated with
 *       the arial.ttf font file. The next time "Arial : Regular" is requested, Open Inventor will use
 *       arial.ttf. Specifying a not-previously-accessed string as a font name will
 *       cause Open Inventor to parse this new string to find a new association. If no association is found,
 *       a default font is used for this association. For instance, "Arial : Regul" doesn't
 *       necessarily correspond to the arial.ttf font file.
 *       To reset the font name cache, call the getAvailableFonts() method and specify the
 *       clearDB parameter as TRUE
 *    
 *   @B Font rendering - caching strategy:@b
 *
 *     The maximum number of caches used during text rendering is specified by the OIV_MAX_FONT_CACHES
 *     environment variable. The default number of caches is 20.
 *     A cache corresponds to a set of parameters that are taken into account during the computation
 *     of a text string. These parameters can be, depending on the kind of text node you are using, font name,
 *     size, complexity, etc. The same cache is used
 *     for strings that have exactly the same appearance (but not
 *     necessarily the same characters). 
 *     Note that the cache depends on color only for text bitmap rendering. That is,
 *     2D text with renderStyle POLYGON.
 *
 *     It's important to specify an appropriate value for the maximum number of font caches 
 *     as application performance can be
 *     highly dependent on the font caching. In fact, if not enough caches are available, in some
 *     cases, Open Inventor will have to recompute text metrics. 
 *
 *   @B Font render style: @b
 * 
 *     The #renderStyle field allows you to control the rendering mode for the font. It's possible 
 *     to render 3D text, e.g. SoText3, using filled polygons (default), filled and outlined polygons or 
 *     textured quads. It's possible to render 2D text, e.g. SoText2, using bitmaps (default) or textured quads.
 *
 *     Text rendered in POLYGON renderStyle with high complexity (for nice appearance) may require a large number of
 *     polygons and reduce performance.  Quality is controlled by the @I value@i field of SoComplexity.
 *
 *     Text rendered in TEXTURE modes uses very few polygons and takes advantage of fast texturing hardware
 *     on modern graphics boards.  Quality is controlled by the @I textureQuality@i field of SoComplexity. @BR
 *     NOTE: When using a TEXTURE renderStyle, it is not possible to apply a texture image (e.g. SoTexture2)
 *     to the text geometry and the SoText3 geometry is not extruded even if the parts field is set to ALL.
 *
 *   @B Textured font rendering considerations: @b
 *
 *     Texture generation quality depends on the current complexity (defined by the
 *     SoComplexity::value field) and the current textureQuality (defined by the
 *     SoComplexity::textureQuality field) in collaboration with a texture resolution value range
 *     defined by the SoTextProperty::textureQualityRange field.
 *     Increasing these values will improve the quality of the output, but at the cost of
 *     decreased performance and increased memory consumption.
 *     
 *    If render style TEXTURE is selected, then a texture image is generated for each character.
 *    The image of the character is automatically antialiased, which means that some pixels
 *    in the texture image will be transparent or partially transparent. Text rendered in this
 *    mode is a "transparent" object like any other transparent geometry, and may be rendered
 *    incorrectly depending on the order in which objects are rendered. To avoid this issue,
 *    you can set the transparency type for rendering the scene graph to,
 *    for example, SORTED_PIXEL (see SoGLRenderAction). Or, set
 *    the text #renderStyle to TEXTURE_DELAYED. This allows delayed rendering of just the text
 *    objects. Note that even though setting the #renderStyle field to TEXTURE_DELAYED is a very simple
 *    solution, it can reduce performance due to less efficient render caching.
 *    Another option is to move the text node definition in the scene graph to be after
 *    all shapes.
 *
 *    For text rendering with small font sizes,
 *    the default anti-aliasing can lead to undesirable blurring effects. The anti-aliasing level 
 *    can be changed through the use of the SoTextProperty::aliasingFactor.
 *
 * @FILE_FORMAT_DEFAULT
 *    Font {
 *    @TABLE_FILE_FORMAT
 *       @TR name        @TD "defaultFont"
 *       @TR size        @TD 10
 *       @TR renderStyle @TD POLYGON
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction,  SoCallbackAction,  SoGetBoundingBoxAction,  SoRayPickAction @BR
 *        Sets the font name and size in the current traversal state.
 *        Sets: SoFontNameElement, SoFontSizeElement, 
 *              SoFontRenderStyleElement, SoTextOutlineEnabledElement
 * 
 * @SEE_ALSO
 *    SoAnnoText3,
 *    SoAsciiText,
 *    SoText2,
 *    SoText3,
 *    SoTextProperty
 * 
 *
 */
class INVENTOR_API SoFont : public SoNode {
  
  SO_NODE_HEADER(SoFont);
  
 public:
  // Fields
  /**
   * This field specifies the font name. 
   * The font name can be set either of two ways: 
   *      - specifying directly the font file name (e.g., arial.ttf) with an absolute or relative path, or no path.
   *      - specifying the font family name and font style name with the form "family : style".
   * Default is "defaultFont" (system dependent).
   */
  //SoSFName name;
  SoSFString name;

  /**
   * This field specifies the font size. The value is in printer's
   * points for 2D text (SoText2) and is in the current (object space) units
   * for 3D text (SoText3). Default is 10.
   */  
  SoSFFloat size;

  /** Render style */
  enum RenderStyle {
    /** Render font as polygonal objects. (default). */
    POLYGON,
    /** Render font as texture objects. @BR
     * @I Warning: @i Small artifacts may appear around the text because of transparency artifacts.
     * To remove them, please use the TEXTURE_DELAYED value. */
    TEXTURE,
    /** 
     * Render font as texture objects, but delay the rendering to
     * prevent transparency artifacts, similar to delaying transparent objects.
     * @ENUM_SINCE_OIV 6.1
     */
    TEXTURE_DELAYED,
    /** Render outlining font. Not available for SoText2 text node. */
    POLYGON_AND_OUTLINE
  };

  /**
   * This field specifies the rendering style of the font.
   * @useenum{RenderStyle}. Default is POLYGON.
   *
   * @FIELD_SINCE_OIV 5.0
   */
  SoSFEnum renderStyle;

   /**
   * Creates a font node with default settings.
   */
  SoFont();
  
  /**
   * Sets flag to enable strings to be interpreted using the Japanese MBCS
   * encoding. This encoding is only used when a stroke font is specified. Default is
   * TRUE.
   * 
   * Set this flag to FALSE if you want to use characters in the range 128-255 (ISO
   * Latin-1) with stroke fonts. @BR @B NOTE: @b In this case the TGS Roman Simplex font will be
   * used regardless of the stroke font specified (limitation).
   * 
   * [Win32] If the current code page is Japanese, the standard Microsoft MBCS
   * encoding is used independent of this flag.
   */
  static void enableJapaneseSF(SbBool flag = TRUE);

  /**
   * Gets flag that indicates whether strings should be interpreted using the 
   * MCS Japanese MBCS encoding. This encoding is only used when a stroke font
   * is specified. 
   */
  static SbBool isJapaneseSFEnabled() ;

  /** 
   * Sets the maximum number of caches that can be used for font rendering. Default is 20. This number
   * corresponds to the number of changes that can be applied to a font: for example,
   * to use the "Arial" and the "Times" fonts one after the other, two caches are created.
   * Subsequently changing a rendering property like SoFont::renderStyle causes the creation of new
   * caches.
   * Properties that cause a cache to be rebuilt include the following: 
   *  - SoAsciiText: font name, size, render style, text orientation
   *  - SoText2: font name, size, viewport size, render style, text orientation
   *  - SoText3: font name, size, render style, text orientation, profile definition
   *
   * Note: This list is not exhaustive.
   */
  static void setNumFontCaches(int maxNumCaches);

  /** 
   * Returns the maximum number of caches that can be used for font rendering.
   */
  static int getNumFontCaches();

  /** 
   * Returns the current number of caches used for font rendering.
   */
  static int getCurrNumFontCaches();

  /** 
   * Returns the number of loadable font files and a list of their names. Each name in the list is formatted 
   * as follows: "Name : Style : (font file)".
   * The retrieved string can be passed directly to the SoFont::name field 
   * (the "(font file)" substring will be ignored).
   * Internally, this method builds a font name cache. It scans some predefined directories and
   * registers the association "Name : Style" <=> "font file". Passing the clearDB parameter as TRUE 
   * empties the font name cache, and then regenerates it.
   * [OIV-WRAPPER-CUSTOM-CODE]
   */
  static int getAvailableFonts(SbString*& strings, SbBool clearDB = FALSE);

  /** 
   * Specifies paths for searching for font files.
   * Using this method overrides all other font path definitions, i.e.,
   * Open Inventor environment variables and default system font directories.
   * The paths must be separated by a semicolon (";")
   *
   * @UNICODE_WARNING
   */
  SoNONUNICODE static void setFontPaths(char*);

  /** 
   * Specifies paths for searching for font files.
   * Using this method overrides all other font path definitions, i.e.,
   * Open Inventor environment variables and default system font directories.
   * The paths must be separated by a semicolon (";")
   */
  static void setFontPaths( const SbString& );

  /** 
   * Specifies paths for searching for font files.
   * Use this method to add new font paths to the current list of searched paths.
   * The paths must be separated by a semicolon (";")
   *
   * @UNICODE_WARNING
   */
  SoNONUNICODE static void addFontPaths(char*);

  /** 
   * Specifies paths for searching for font files.
   * Use this method to add new font paths to the current list of searched paths.
   * The paths must be separated by a semicolon (";")
   */
  static void addFontPaths( const SbString& );

  /** 
   * Gets current font path list.
   */
  static SbString getFontPaths();
  
  /** 
   * Returns TRUE if the specified font is available for use by Open Inventor.
   * Internally, this method builds a font name cache (once only). It scans 
   * some predefined directories and
   * registers the association "Name : Style" <=> "font file". 
   *
   * @UNICODE_WARNING
   */
  SoNONUNICODE static SbBool isAvailableFont(char* );

  /** 
   * Returns TRUE if the specified font is available for use by Open Inventor.
   * Internally, this method builds a font name cache (once only). It scans 
   * some predefined directories and
   * registers the association "Name : Style" <=> "font file". 
   */
  static SbBool isAvailableFont(const SbString& );

  /** 
   * Clears the Open Inventor internal font name cache, i.e., the list of available fonts.
   */
  static void clearFontNameCache();

  /**
   * Set the state of the override field.
   * see SoNode::setOverride doc.
   */
  inline virtual void setOverride(const SbBool state)
  { override.setValue(state); }

  /**
   * Returns the state of the override field.
   */
  inline virtual SbBool isOverride() const
  { return override.getValue(); }

SoEXTENDER public:
  virtual void doAction(SoAction *action);
  virtual void GLRender(SoGLRenderAction *action);
  virtual void callback(SoCallbackAction *action);
  virtual void pick(SoPickAction *action);
  virtual void getBoundingBox(SoGetBoundingBoxAction *action);
  virtual void getPrimitiveCount(SoGetPrimitiveCountAction *action);

 SoINTERNAL public:
  static void initClass();
  static void exitClass();

  SoSFBool override;

 protected:
  virtual ~SoFont();

  // TRUE by default
  static SbBool s_isJapaneseSFEnabled;

};

#endif /* _SO_FONT_ */

