/*=================================================================================
 ***     THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),    ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                 ***
 ***                                                                            ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS   ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR   ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                       ***
 ***                                                                            ***
 ***                        RESTRICTED RIGHTS LEGEND                            ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS  ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN  ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT  ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN  ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         ***
 ***                                                                            ***
 ***                   COPYRIGHT (C) 1996-2018 BY FEI S.A.S,                    ***
 ***                              BORDEAUX, FRANCE                              ***
 ***                            ALL RIGHTS RESERVED                             ***
 =================================================================================*/

#ifndef SO_GL_CALLBACK_H
#define SO_GL_CALLBACK_H

#include <Inventor/nodes/SoCallback.h>

/**
 * Provides custom OpenGL behavior during rendering actions.
 *
 * @ingroup nodes
 *
 * @DESCRIPTION
 *   This node behaves exactly like an SoCallback @I except...@i
 *   the SoGLCallback node allows the application to make OpenGL calls during a
 *   render traversal.  It is normally used to make OpenGL rendering calls (and
 *   any state setting calls needed for rendering).
 *
 *   SoGLCallback automatically calls glPushAttrib() to save the OpenGL state before
 *   executing the callback and then calls glPopAttrib() (restore) after executing
 *   the callback, ensuring that OpenGL state cannot leak outside the node.
 *
 *   Please note that this means an SoGLCallback node @I cannot@i modify the OpenGL
 *   state used for Open Inventor rendering. (This is a change since Open Inventor 10.0)
 *   You must use standard Open Inventor nodes to modify state for Open Inventor rendering.
 *   For example, use an SoViewport node to change the OpenGL viewport, use the SoDepthBuffer
 *   node to change the OpenGL depth buffer test and so on.
 *
 *   You can use this node to get values from the Open Inventor traversal state list using
 *   the SoElement classes. For example, SoLineWidthElement contains the current line width
 *   set by an SoDrawStyle node.
 *
 *   This node should not be used to do Open Inventor rendering. Open Inventor rendering
 *   calls should only be done using an SoCallBack node.
 *
 *   There is some "overhead" for rendering using SoGLCallback instead of Open Inventor
 *   nodes. Normal Open Inventor rendering is optimized in various ways to maximize GPU
 *   performance and to keep the GPU "busy".  When an SoGLCallback node is traversed,
 *   Open Inventor must finish all current rendering, execute the application's OpenGL
 *   calls, then resume normal rendering.
 *
 * @B Create the node:@b
 * \if_cpp
 * \par
 * \code
 *   SoGLCallback* glCallbackNode = new SoGLCallback();
 *   glCallbackNode->setCallback( myGLCallback );
 *   root->addChild( glCallbackNode );
 * \endcode
 * \endif
 * \if_dotnet
 * \par
 * \code
 *   SoGLCallback glCallbackNode = new SoGLCallback();
 *   glCallbackNode.CallbackHandler = new SoGLCallback.CallbackCB( myGLCallback );
 *   root.AddChild( glCallbackNode );
 * \endcode
 * \endif
 * \if_java
 * \par
 * \code
 *   SoGLCallback glCallbackNode = new SoGLCallback();
 *   glCallbackNode.setGLCallback( new myGLCallback() );
 *   root.addChild( glCallbackNode );
 *   \endcode
 * \endif
 *
 * Create the \if_dotnet @B Delegate:@b \else @B Callback: @b \endif
 *
 * The \if_dotnet delegate \else callback \endif function registered with the node will only be
 * executed during SoGLRenderAction traversal, as calling OpenGL functions requires a current OpenGL context.
 * \if_cpp
 * \par
 * \code
 *   void myCallback( void* data, SoAction* action )
 *   {
 *     // Safe to call OpenGL functions
 *   }
 * \endcode
 * \endif
 * \if_dotnet
 * \par
 * \code
 *   void myCallback( SoAction action )
 *   {
 *      // Safe to call OpenGL functions
 *   }
 * \endcode
 * \endif
 * \if_java
 * \par
 * \code
 *   class myCallback implements SoCallback.CB
 *   {
 *     @Override
 *     public void invoke( SoAction action )
 *     {
 *       // Safe to call OpenGL functions
 *     }
 *   }
 * \endcode
 * Or create node and callback in-line:
 * \code
 *   SoCallback callbackNode = new SoCallback();
 *   callbackNode.setCallback(
 *       new SoCallback.CB()
 *       {
 *           @Override
 *           public void invoke( SoAction action )
 *           {
 *               // Safe to call OpenGL functions
 *           }
 *       }
 *   );
 * \endcode
 * \endif
 * 
 * @FILE_FORMAT_DEFAULT
 *    GLCallback {
 *    @TABLE_FILE_FORMAT
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction,  SoGetBoundingBoxAction,  SoPickAction @BR
 *        Calls the specified \if_dotnet delegate \else callback \endif for all actions.
 * 
 * @SEE_ALSO
 *    SoAction,
 *    SoCallback,
 *    SoCallbackAction,
 *    SoEventCallback
 *
 * @NODE_SINCE_OIV 10.0
 */

class INVENTOR_API SoGLCallback : public SoCallback
{
  SO_NODE_HEADER(SoGLCallback);

public:
  /** Constructor */
  SoGLCallback();

SoEXTENDER public:

  // Traversal methods for all the actions:
  virtual void doAction(SoAction *action);

SoINTERNAL public:
  static void initClass();
  static void exitClass();

protected:
  /** Destructor */
  virtual ~SoGLCallback();

};

#endif /* SO_GL_CALLBACK_H */
