/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Nick Thompson (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_GROUP_
#define  _SO_GROUP_

class SoChildList;

#include <Inventor/nodes/SoNode.h>
#include <Inventor/sensors/SoDataSensor.h>  // for ChangeType
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/SbPImpl.h>

SO_PIMPL_PUBLIC_DECLARATION(SoGroup)

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoGroup
//
//  Base group node: all children are traversed.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * Base class for all group nodes.
 * 
 * @ingroup GroupNodes
 * 
 * @DESCRIPTION
 *   This node defines the base class for all group nodes. SoGroup is a node that
 *   contains an ordered list of child nodes. The ordering of the child nodes
 *   represents the traversal order for all operations (for example, rendering,
 *   picking, and so on). This node is simply a container for the child nodes and
 *   does not alter the traversal state in any way. During traversal, state
 *   accumulated for a child is passed on to each successive child and then to the
 *   parents of the group (SoGroup does not push or pop traversal state as
 *   SoSeparator does).
 * 
 * @FILE_FORMAT_DEFAULT
 *    Group {
 *    @TABLE_FILE_FORMAT
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction,  SoCallbackAction,  SoGetBoundingBoxAction,  SoHandleEventAction,  SoRayPickAction @BR
 *        Traverses each child in order.
 * 
 *    SoGetMatrixAction @BR
 *        Does nothing unless the group is in the middle of the path chain the action is
 *        being applied to. If so, the children up to and including the next node in the
 *        chain are traversed.
 * 
 *    SoSearchAction @BR
 *        If searching for group nodes, compares with this group. Otherwise, continues to
 *        search children.
 * 
 *    SoWriteAction @BR
 *        Writes out the group node. This method also deals with any field data
 *        associated with the group node. As a result, this method is used for most
 *        subclasses of SoGroup as well.
 * 
 * 
 * @SEE_ALSO
 *    SoArray,
 *    SoLevelOfDetail,
 *    SoMultipleCopy,
 *    SoPathSwitch,
 *    SoSeparator,
 *    SoSwitch
 * 
 * 
 */
class INVENTOR_API SoGroup : public SoNode
{
  SO_NODE_HEADER(SoGroup);
  SO_PIMPL_PUBLIC_HEADER(SoGroup)

 public:
  /**
   * Whether to ignore this node during bounding box traversal. Default is FALSE.
   */
  SoSFBool boundingBoxIgnoring;

  /**
   * Creates an empty group node.
   */
  SoGroup();

  /**
  * Constructor that takes approximate number of children. Space is allocated to
  * contain the specified number of children, but the group does not contain any actual child nodes.
  */
  SoGroup(int nChildren);

  /**
   * Adds a child as last one in group.
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onAddChild(child)}]
   */
  virtual void addChild(SoNode *child);

  /**
   * Adds a child so that it becomes the one with the given index.
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onInsertChild(child, newChildIndex)}]
   * [OIVNET-WRAPPER-ARG IN,INDEX{0,GetNumChildren()}]
   */
  virtual void insertChild(SoNode *child, int newChildIndex);

  /**
   * Returns pointer the child node with the given index.
   * [OIVNET-WRAPPER-ARG INDEX{0,(GetNumChildren()-1)}]
   */
  virtual SoNode *getChild(int index) const;

  /**
   * Finds index of given child within group. Returns -1 if not found.
   */
  virtual int findChild(const SoNode *child) const;

  /**
   * Returns number of children.
   */
  virtual int getNumChildren() const;

  /**
   * Removes child with given index from group.
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onRemoveChild(index)}]
   * [OIVNET-WRAPPER-ARG INDEX{0,(GetNumChildren()-1)}]
   */
  virtual void removeChild(int index);

  /**
   * Removes first instance of given child from group.
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onRemoveChild(child)}]
   */
  virtual void removeChild(SoNode *child)
    { removeChild(findChild(child)); }

  /**
   * Removes all children from group.
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onRemoveAllChildren()}]
   */
  virtual void removeAllChildren();

  /**
   * Replaces child with given index with new child.
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onReplaceChild(index,newChild)}]
   * [OIVNET-WRAPPER-ARG INDEX{0,(GetNumChildren()-1)},IN]
   */
  virtual void replaceChild(int index, SoNode *newChild);

  /**
   * Replaces first instance of given child with new child.
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onReplaceChild(oldChild,newChild)}]
   */
  virtual void replaceChild(SoNode *oldChild, SoNode *newChild)
    { replaceChild(findChild(oldChild), newChild); }

 SoEXTENDER public:

  /** 
   * Used by BoundingBoxAction to known if bounding box computation should 
   * be ignored or not.
   * By default the returned value is given by boundingBoxIgnoring field.
   */
  inline virtual SbBool isBoundingBoxIgnoring() const
  { return boundingBoxIgnoring.getValue(); }

  // Implement actions
  virtual void doAction(SoAction *action);
  virtual void callback(SoCallbackAction *action);
  virtual void GLRender(SoGLRenderAction *action);
  virtual void getBoundingBox(SoGetBoundingBoxAction *action);
  virtual void getMatrix(SoGetMatrixAction *action);
  virtual void handleEvent(SoHandleEventAction *action);
  virtual void pick(SoPickAction *action);
  virtual void search(SoSearchAction *action);
  virtual void write(SoWriteAction *action);
  virtual void getPrimitiveCount(SoGetPrimitiveCountAction *action);

SoINTERNAL public:
  static void initClass();
  static void exitClass();

  // Returns pointer to children
  virtual SoChildList *getChildren() const;

  // Internal child management following inheritance of classes.
  virtual void internalRemoveChild( int index )                          { removeChild(index); }
  virtual void internalRemoveChild( SoNode *child  )                     { removeChild(findChild(child)); }
  virtual void internalRemoveAllChildren()                               { removeAllChildren(); }
  virtual void internalAddChild( SoNode *child )                         { addChild(child); }
  virtual int internalFindChild( const SoNode *child ) const             { return findChild(child); }
  virtual void internalInsertChild( SoNode *child, int newChildIndex )   { insertChild(child, newChildIndex); }
  virtual SoNode *internalGetChild( int index) const                     { return getChild(index); }
  virtual void internalReplaceChild( int index, SoNode *newChild)        { replaceChild(index, newChild); }
  virtual void internalReplaceChild( SoNode *oldChild, SoNode *newChild) { replaceChild(oldChild,newChild); }

  virtual void notify(SoNotList *list);

  virtual bool isGroup() { return true; }
  
  // Copies the contents of the given node into this instance
  virtual void copyContents(const SoFieldContainer *fromFC, SbBool copyConnections);

  // serialize children list
  static void writeChildren( SoWriteAction *action, SoChildList* children_ );

 protected:
  SoChildList *children;
  
  // Reads stuff into instance of SoGroup. Returns FALSE on error
  virtual SbBool readInstance(SoInput *in, unsigned short flags);
  
  // Reads just the children into instance of SoGroup. Returns FALSE on error
  virtual SbBool readChildren(SoInput *in);
  
  virtual ~SoGroup();


private:

  void commonConstructor();

};


#endif /* _SO_GROUP_ */

