/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Nick Thompson (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2019 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : David Beilloin (Mar 2010)
**=======================================================================*/


#ifndef  _SO_INDEXED_LINE_SET_
#define  _SO_INDEXED_LINE_SET_

#include <Inventor/nodes/SoIndexedShape.h>
#include <Inventor/SbPImpl.h>

class SoState;
class SoTangentBundle;

SO_PIMPL_PUBLIC_DECLARATION(SoIndexedLineSet)

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoIndexedLineSet
//
//  Indexed set of (poly)lines. Each line consists of 2 or more
//  vertices, each of which is denoted by an integer index (from the
//  coordIndex field) into the current coordinates. Depending on the
//  current material, normal and tangent binding values, the materials,
//  normals and tangents for the lines or vertices may be accessed in order or
//  indexed. If they are indexed, the materialIndex, normalIndex and tangentIndex
//  fields are used. Texture coordinates may also be indexed, using
//  the textureCoordIndex field.  
//
//////////////////////////////////////////////////////////////////////////////

// This coordinate index indicates that the current line ends and the
// next line begins
#define SO_END_LINE_INDEX       (-1)

/**
 * Indexed polyline shape node.
 * 
 * @ingroup ShapeNodes
 * 
 * @DESCRIPTION
 *   This node represents a 3D shape formed by constructing a set of polylines from vertices
 *   located at the coordinates specified in the #vertexProperty field (from
 *   SoVertexShape), or the current inherited coordinates. For optimal performance,
 *   the #vertexProperty field is recommended.
 *   
 *   SoIndexedLineSet uses the indices in the #coordIndex field (from
 *   SoIndexedShape) to specify the polylines. An index of SO_END_LINE_INDEX (-1)
 *   indicates that the current polyline has ended and the next one begins.
 *   
 *   The coordinates of the line set are transformed by the current cumulative
 *   transformation. 
 *
 *   The line width and pattern are controlled by fields in SoDrawStyle.
 *
 *   The lines are drawn with the current drawing style (SoDrawStyle), but
 *   drawing style FILLED is treated as LINES.
 *
 *   Treatment of the current material, normal and tangent binding is as follows:
 *   - PER_PART binding specifies a material, normal or tangent for each segment of the line.
 *   - PER_FACE binding specifies a material, normal or tangent for each polyline.
 *   - PER_VERTEX specifies a material, normal or tangent for each vertex.
 *   - The corresponding _INDEXED bindings are the same, but use the #materialIndex, #normalIndex or #tangentIndex
 *     indices (see SoIndexedShape).
 *   - The default material binding is OVERALL.
 *   - The default normal and tangent bindings are PER_VERTEX_INDEXED
 *
 *   The current complexity value has no effect on the rendering of indexed line sets.
 *
 *   Shape Antialiasing type is SoShape::LINES.
 *
 *   Lighting: @BR
 *   - By default, lighting is enabled (SoLightModel model = PER_VERTEX_PHONG). But... @BR
 *     Unlike polygonal shapes, lighting will @B not @b be applied to line
 *     shapes unless the application explicitly sets normal vectors (using
 *     SoVertexProperty or SoNormal) or sets tangent vectors (using SoVertexProperty).
 *     Normal vectors are not automatically computed for line shapes. @BR @BR
 *   - If lighting is enabled and @B tangent vectors @b are specified: @BR
 *     then @I normal vectors are ignored @i and lines are lighted using the Illuminated
 *     Stream Lines algorithm (Zckler, Stalling & Hege, 1996)
 *     (http://www.zib.de/visual-publications/sources/src-1996/isl.pdf). @BR @BR
 *   - If lighting is enabled and @B normal vectors @b are specified but @I not @i tangent vectors: @BR
 *     then lines are lighted using the same algorithm as polygonal shapes. @BR @BR
 *   - Note the above discussion is specific to rendering of @I line shapes @i, e.g. this node. @BR
 *     If you render a polygonal shape, e.g. SoFaceSet, using LINES mode (SoDrawStyle),
 *     the rules for polygonal shapes apply (if lighting is enabled and no normal vectors
 *     were specified, then normal vectors are automatically computed by Open Inventor). @BR @BR
 *   - NOTE: If you want to apply lighting in a shader, you must explicitly set normal
 *     or tangent vectors, otherwise lighting will be disabled for the line shape.
 *
 *   Limitations:
 *   - The VBO (vertex buffer object) rendering model only allows one set of indices for a primitive.
 *     Therefore it is not possible to use VBO rendering (and performance may be lower) if indices
 *     are set in the @I materialIndex@i, @I normalIndex@i, @I tangentIndex@i or @I textureCoordIndex@i fields.
 *
 *   - Due to limitations of the VBO (vertex buffer object) rendering model, it is not possible
 *     to use VBO rendering (and performance may be lower) if either the normal binding, the tangent binding
 *     or the material binding is set to either PER_PART(_INDEXED) or PER_FACE(_INDEXED).
 * 
 * @FILE_FORMAT_DEFAULT
 *    IndexedLineSet {
 *    @TABLE_FILE_FORMAT
 *       @TR vertexProperty      @TD NULL
 *       @TR coordIndex          @TD -1
 *       @TR materialIndex       @TD -1
 *       @TR normalIndex         @TD -1
 *       @TR tangentIndex        @TD -1
 *       @TR textureCoordIndex   @TD -1
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction @BR
 *        Draws lines based on the current coordinates, normals, tangents, materials, drawing
 *        style, and so on.
 * 
 *    SoRayPickAction @BR
 *        Picks lines based on the current coordinates and transformation. Details about
 *        the intersection are returned in an SoLineDetail.
 * 
 *    SoGetBoundingBoxAction @BR
 *        Computes the bounding box that encloses all vertices of the line set with the
 *        current transformation applied to them. Sets the center to the average of the
 *        coordinates of all vertices.
 * 
 *    SoCallbackAction @BR
 *        If any line segment callbacks are registered with the action, they will be
 *        invoked for each successive segment in the line set.
 * 
 * 
 * @SEE_ALSO
 *    SoCoordinate3,
 *    SoDrawStyle,
 *    SoFullSceneAntialiasing,
 *    SoLineDetail,
 *    SoLineSet,
 *    SoVertexProperty
 * 
 * 
 */
class INVENTOR_API SoIndexedLineSet : public SoIndexedShape
{
  SO_NODE_HEADER(SoIndexedLineSet);
  SO_PIMPL_PUBLIC_HEADER(SoIndexedLineSet)

public:
  // Inherits all fields

  /**
   * Creates an indexed line set node with default settings.
   */
  SoIndexedLineSet();

SoEXTENDER public:
  /** Implements SoGLRenderAction actions */
  virtual void GLRender( SoGLRenderAction* action );

  /** 
   * Implements getBoundingBox actions
   * to tell open caches that they contain lines
   */
  virtual void getBoundingBox( SoGetBoundingBoxAction* action );

  /** Implements SoGetPrimitiveCountAction action */
  virtual void getPrimitiveCount( SoGetPrimitiveCountAction* action );

  /**
   * Subclasses may define this method to generate tangents to use
   * when the tangent binding is DEFAULT and there aren't enough
   * tangents in the state. This should use the given SoTangentBundle
   * to generate the tangents.
   */
  virtual void generateDefaultTangents( SoState* state,
                                        const SbVec3f* coords,
                                        int numCoords,
                                        SoTangentBundle* tb,
                                        SbBool storeRef = FALSE );
SoINTERNAL public:

  /** register in database */
  static void initClass();
  /** unregister from database */
  static void exitClass();

  // return num lines for test  
  virtual int getNumPrim( SoState* state = NULL );

protected:

  // Generates line segments representing line set
  virtual void generatePrimitives( SoAction* action );

  // count the number of primitives in the shape
  virtual void countPrim( SoState* state = NULL );

  // really do the rendering part
  virtual void doRendering( SoGLRenderAction* action, const SoShapeStyleElement* shapeStyle );

  virtual void callSetupIndices( SoState* state, const SoShapeStyleElement* shapeStyle, uint32_t useTexCoordsAnyway );

  // says wether to generate normals
  virtual void shouldGenerateNormals( SoGLRenderAction* action,const SoShapeStyleElement* shapeStyle );

  // setup lazy element
  virtual void setupLazyElement(SoGLRenderAction *action, const SoShapeStyleElement *shapeStyle);

  // says that rendering is PointOrLine
  virtual SbBool isRenderingPointsOrLines( SoGLRenderAction* action );

  // Overrides standard method to create an SoLineDetail instance
  virtual SoDetail* createLineSegmentDetail( SoRayPickAction* action,
                                             const SoPrimitiveVertex* v1,
                                             const SoPrimitiveVertex* v2,
                                             SoPickedPoint* pp );
  
  /** Destructor */
  virtual ~SoIndexedLineSet();
};

inline SbBool 
SoIndexedLineSet::isRenderingPointsOrLines( SoGLRenderAction* )
{
  return TRUE;
}

#endif /* _SO_INDEXED_LINE_SET_ */
    

