/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2018 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (Aug 2013)
**=======================================================================*/


#ifndef _SO_INSTANCE_PARAMETER_H_
#define _SO_INSTANCE_PARAMETER_H_

#include <Inventor/SbBasic.h>
#include <Inventor/nodes/SoNode.h>

#include <Inventor/nodes/SoShaderParameter.h>

#include <Inventor/fields/SoSFBufferObject.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/fields/SoSFUInt32.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFEnum.h>


/**
 * @VSGEXT Per-instance parameter node storing a buffer object.
 * 
 * @ingroup ShaderNodes
 * 
 * @DESCRIPTION
 *   This node allows the application to provide a per-instance parameter of any type, using
 *   an SoBufferObject. It is used with the SoMultipleInstance node.
 *
 *   Some predefined parameter names can be used for common instance parameters like
 *   position, rotation and scale factor.  Parameter objects with these names are
 *   automatically managed and are recognized and used by the default shaders.
 *   Applications can define additional parameter objects to be used by custom shaders.
 *
 *   See the base classes SoShaderParameter, SoVertexShaderParameter and SoVertexShaderParameterBufferObject
 *   for more details.
 *
 *   Note: Unlike SoVertexShaderParameter, instances of this node must be added to the
 *   @a parameters field of an SoMultipleInstance node, not directly in the scene graph.
 *
 * @FILE_FORMAT_DEFAULT
 *    InstanceParameter {
 *      @TABLE_FILE_FORMAT
 *        @TR name                  @TD ""
 *        @TR identifier            @TD 0
 *        @TR value                 @TD NULL
 *        @TR components            @TD 4
 *        @TR type                  @TD SbDataType::FLOAT
 *        @TR stride                @TD 0
 *        @TR shouldBeNormalized    @TD FALSE
 *        @TR divisor               @TD 1
 *      @TABLE_END
 *    }
 * 
 * 
 * @SEE_ALSO
 *    SoVertexShaderParameter, SoVertexShaderParameterBufferObject, SoMultipleInstance
 *
 * @NODE_SINCE_OIV 9.5
 */
class INVENTOR_API SoInstanceParameter : public SoVertexShaderParameterBufferObject
{
  SO_NODE_HEADER(SoInstanceParameter);

public:

  /**
   * Constructor
   */
  SoInstanceParameter();

  /**
   * This field modifies the rate at which values advance during multiple instance rendering. 
   * Divisor must be non-zero and the attribute advances once per divisor instances of the set(s) of vertices being rendered.
   * Default is 1 (1 parameter per instance).
   */
  SoSFUInt32 divisor;

  /**
  * List of predefined per-instance attribute names.
  * The names corresponding to the following enum symbols can be queried 
  * by mean of the getPredefinedParameterName() method.
  * They correspond to a set of predefined parameter names that
  *  are automatically managed/interpreted:
  * - OivShapeInstanceTranslation:
  *      Handles an array of vec3f translation to apply to vertices of each instance. @BR
  *
  * - OivShapeInstanceScale:
  *      Handles an array of vec3f scale factor applied to vertices of each instance. @BR
  *
  * - OivShapeInstanceRotation:
  *      Handles an array of vec4f quaternion applied to vertices of each instance. @BR
  *
  * - OivShapeInstanceColor:
  *      Handles an array of vec3f color applied to vertices of each instance. @BR
  *
  * - OivShapeInstanceTransparency:
  *      Handles an array of float transparency applied to vertices of each instance. @BR
  *
  * - OivShapeInstanceVisibility:
  *      Handles an array of int32 visibility flag to apply to each instance.
  *
  * OivShapeInstanceTranslation, OivShapeInstanceScale and OivShapeInstanceRotation
  * are combined together to create an equivalent matrix 
  * available in GLSL shaders with the function
  * \code mat4 OivInstanceMatrix() \endcode
  * If no parameters are present then an identity matrix is returned.
  *
  * OivShapeInstanceColor and OivShapeInstanceTransparency affect, respectively, the diffuse color
  * and the transparency of each instance. Note that when one of these parameters is defined,
  * it overrides the diffuse color or the transparency value of a SoMaterial node.
  *
  * OivShapeInstanceVisibility affects the visibility of each instance (i.e. 0 means hidden and a
  * value greater than zero means visible). This attribute is available in GLSL shaders with the function
  * \code int OivInstanceVisibility() \endcode.
  */
  enum PredefinedParameters
  {
    TRANSLATION = 0,
    ROTATION,
    SCALE,
    VISIBILITY,
    COLOR,
    TRANSPARENCY,
    NUM_PREDEFINED_PARAMETERS
  };

  /**
  * Creates an instance parameter object for one of the predefined attribute names.
  *
  * paramKey = TRANSLATION: @BR
  * Sets the translation per instance parameter values using the specified SoBufferObject.
  * The SoBufferObject is expected to contain translations encoded as 3 float values, e.g. array of SbVec3f.
  * The buffer size must be at least (numInstances/divisor) * 3 * sizeof(float).
  * The divisor parameter indicates by how many instances a translation value is shared.
  *
  * Example :
  * - numInstance = 4
  * - translationValues = {(1, 0, 0), (0, 1, 0)}
  * - divisor = 2
  * => The first 2 instances will be translated by using (1, 0, 0) and the next two using (0, 1, 0).
  *
  * paramKey = ROTATION: @BR
  * Sets the rotation per instance parameter values using the specified SoBufferObject.
  * The SoBufferObject is expected to contain rotations encoded as 4 float values, e.g. array of SbRotation.
  * The buffer size must be at least (numInstances/divisor) * 4 * sizeof(float).
  * The divisor parameter indicates by how many instances a rotation value is shared.
  *
  * paramKey = SCALE: @BR
  * Sets the scaling per instance parameter values using the specified SoBufferObject.
  * The SoBufferObject is expected to contain scaling factor encoded as 3 float values, e.g. array of SbVec3f.
  * The buffer size must be at least (numInstances/divisor) * 3 * sizeof(float).
  * The divisor parameter indicates by how many instances a scale value is shared.
  *
  * paramKey = COLOR: @BR
  * Sets the color per instance parameter values using the specified SoBufferObject.
  * The SoBufferObject is expected to contain colors encoded as 3 float values, e.g. array of SbColor.
  * The buffer size must be at least (numInstances/divisor) * 3 * sizeof(float).
  * The divisor parameter indicates by how many instances a color value is shared.
  *
  * paramKey = TRANSPARENCY: @BR
  * Sets the transparency per instance parameter values using the specified SoBufferObject.
  * The SoBufferObject is expected to contain colors encoded as 1 float value, e.g. array of float.
  * The buffer size must be at least (numInstances/divisor) * sizeof(float).
  * The divisor parameter indicates by how many instances a transparency value is shared.
  *
  * paramKey = VISIBILITY: @BR
  * Sets the visibility per instance parameter values using the specified SoBufferObject.
  * The SoBufferObject is expected to contain visibility value encoded as an int32_t value.
  * The buffer size must be at least (numInstances/divisor) * sizeof(int32_t).
  * The divisor parameter indicates by how many instances a visibility value is shared.
  */
  static SoInstanceParameter* createPredefinedParameter( SoInstanceParameter::PredefinedParameters paramKey, 
                                                         const SoBufferObject* values, uint32_t divisor = 1 );

  /** Returns the predefined instance parameter name for the specified paramKey */
  static SbString getPredefinedParameterName( SoInstanceParameter::PredefinedParameters paramKey );

  /** Sets the #name field to the string corresponding to the given paramKey */
  void setPredefinedParameterName( SoInstanceParameter::PredefinedParameters paramKey );

SoINTERNAL public:
  /** @copydoc SoNode::initClass() */
  static void initClass();

  /** @copydoc SoNode::exitClass() */
  static void exitClass();

  /** checks field integrity: divisor filed is not zero */
  virtual void        notify(SoNotList *list);

  // Attrib divisor is set by a field
  virtual unsigned int getVertexAttribDivisor() { return divisor.getValue(); }

protected:
  // Destructor
  virtual ~SoInstanceParameter();

};

/*----------------------------------------------------------------------------*/

#endif /* _SO_INSTANCE_PARAMETER_H_ */

