/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author(s) : VSG (Sep 2011)
**=======================================================================*/


#ifndef  _SO_INTERACTIVE_SWITCH_
#define  _SO_INTERACTIVE_SWITCH_

#include <Inventor/nodes/SoSwitch.h>

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoSwitch
//
//  Switch group node: traverses only the child indicated by integer
//  "whichInteractiveChild" field. If this field is SO_SWITCH_NONE, no children
//  are traversed, while SO_SWITCH_INHERIT means inherit the index
//  from the current switch state, doing a modulo operation to make
//  sure the child index is within the correct bounds of the switch
//  node.
//
//////////////////////////////////////////////////////////////////////////////

/** 
 * Group node that traverses one chosen child depending on scene interaction.
 * 
 * @ingroup GroupNodes
 * 
 * @DESCRIPTION
 *   This group node has the same base behavior as the SoSwitch node for its inherited
 *   #whichChild field.
 *
 *   It has the additional feature of (optionally) traversing a
 *   different (or no) child during an "interaction".  This allows the application to
 *   provide higher performance during an interaction by switching to a less complex
 *   version of a geomtry or not rendering the geometry at all.  Interaction includes
 *   moving the camera (navigating around the scene) and interacting with a dragger
 *   (see SoDragger), for example dragging a slice through a volume data set.
 *
 *   The #whichInteractiveChild field specifies the index of the child to traverse (where
 *   the first child has index 0), when a user interaction, for example moving the camera,
 *   is in progress. This means that while the camera is moving, the traversed children
 *   will be the children defined by the #whichInteractiveChild field, and when
 *   interaction is stopped, the standard SoSwitch #whichChild field value will again be
 *   used as the index to traverse.
 *
 *   @B NOTES: @b
 *   - Remember that the default value for both @I whichChild @i and @I whichInteractiveChild @i 
 *     is SO_SWITCH_NONE, meaning that nothing is displayed by default.
 *
 *   - Remember that this node is an SoGroup, NOT an SoSeparator, and therefore does not
 *     provide render caching.  Generally if the children contain geometry, it is good 
 *     practice to make each child an SoSeparator to make render caching more effective.
 * 
 * @FILE_FORMAT_DEFAULT
 *    InteractiveSwitch {
 *    @TABLE_FILE_FORMAT
 *       @TR whichChild   @TD -1
 *       @TR whichInteractiveChild   @TD -1
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *        inherited from SoSwitch.
 *        While interacting, traverses the chosen (from #whichInteractiveChild) child or children. Sets: SoInteractiveSwitchElement.
 *        While not interacting, traverses the chosen (from #whichChild) child or children. Sets: SoSwitchElement.
 * 
 * @SEE_ALSO
 *    SoArray,
 *    SoLevelOfDetail,
 *    SoMultiPathSwitch,
 *    SoMultipleCopy,
 *    SoPathSwitch,
 *    SoSwitch
 * 
 */
class INVENTOR_API SoInteractiveSwitch : public SoSwitch
{

  SO_NODE_HEADER(SoInteractiveSwitch);

 public:
  /**
   * Index of the child to traverse during interaction, 
   * (or one of #SO_SWITCH_NONE, #SO_SWITCH_INHERIT, or #SO_SWITCH_ALL).
   */
  SoSFInt32 whichInteractiveChild;

  /**
   * Creates an interactive-switch node with default settings.
   */
  SoInteractiveSwitch();

  /**
   * Constructor that takes approximate number of children.
   */
  SoInteractiveSwitch(int nChildren);

  /**
   * Overrides method in SoSwitch to return FALSE if there is no
   * selected child or the selected child does not affect the state.
   */
  virtual SbBool affectsState() const;

SoINTERNAL public:
  
  static void initClass();
  static void exitClass();

protected:
  /** Destructor. */
  virtual ~SoInteractiveSwitch();

  /** Traverses correct child. If matchIndex >=0, child index must match */
  virtual void doChild(SoAction *action, int matchIndex = -1);
};

#endif // _SO_INTERACTIVE_SWITCH_


