/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
 **=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_LIGHT_
#define  _SO_LIGHT_

#include <Inventor/nodes/SoNode.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFColor.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/SbViewportRegion.h>

class SoCamera;

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoLight
//
//  Abstract base light source node.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * Abstract base class for all light source nodes.
 *
 * @ingroup LightNodes
 *
 * @DESCRIPTION
 *   SoLight is the abstract base class for all light nodes. A light node defines an
 *   illumination source that may affect subsequent shapes in the scene graph,
 *   depending on the current lighting style. Light sources are affected by the
 *   current transformation. A light node under a separator does not affect any
 *   objects outside that separator.
 *
 *   You can also use a node kit to create a light. See SoLightKit.
 *
 *   An SoEnvironment node can be used to specify additional options related to lighting:
 *   - The ambient light component of the Phong reflection model.
 *   - Light attenuation for SoPointLight and SoSpotLight nodes.
 *   - Ambient occlusion.
 *
 *   Shadows @BR
 *   - A light may cast shadows on any shapes included in an SoShadowGroup placed later in the scene graph.
 *     See the field #shadowIntensity and SoShadowGroup for details on shadowing attributes and limitations.
 *
 *   Lighting is enabled by default (default model is PER_VERTEX_PHONG). When lighting is enabled, there must
 *   be light nodes (e.g. SoDirectionalLight) in the scene graph for objects to be easily visible.
 *   When using the Open Inventor viewer classes (e.g. SoWinExaminerViewer), the viewer will create
 *   a "headlight" (directional light attached to the camera) if no other light nodes exist when 
 *   setSceneGraph() is called.
 *
 *   By default, lighting is only computed for the "front" faces of geometry, so the
 *   back face of the geometry will be dark.  The front face is determined by the normal vector,
 *   if specified, or by the vertex ordering (see SoShapeHints) if normals are computed by Open
 *   Inventor. To enable "two sided" lighting (lighting both front and back faces) see SoShapeHints.
 *
 *   The maximum number of lights depends on the hardware, but at least eight
 *   simultaneous lights are supported.
 *
 * @FILE_FORMAT_DEFAULT
 *    This is an abstract class. See the reference page of a derived class for the
 *   format and default values.
 *
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction @BR
 *        Activates this light (if so specified) during traversal. All shape nodes that
 *        come after this light in the scene graph are illuminated by this light.
 *
 *
 * @SEE_ALSO
 *    SoDirectionalLight,
 *    SoEnvironment,
 *    SoLightKit,
 *    SoLightModel,
 *    SoMaterial,
 *    SoPointLight,
 *    SoSpotLight,
 *    SoShadowGroup
 *
 *
 */
class INVENTOR_API SoLight : public SoNode
{

  SO_NODE_ABSTRACT_HEADER( SoLight );

public:
  /**
   * Determines whether the source is active or inactive. When inactive, the source
   * does not illuminate at all.
   */
  SoSFBool on;

  /**
   * Illumination intensity of light source. Valid values range from 0.0 (no
   * illumination) to 1.0 (maximum illumination).
   */
  SoSFFloat intensity;

  /**
   * Light source illumination color.
   */
  SoSFColor color;

  /**
   * Specifies the intensity of shadows cast by this light.
   *
   * This field only has an effect on objects that are inside an SoShadowGroup node.
   *
   * The shadow intensity is a value between 0 and 1.
   * With a value of 0, the light casts no shadows.
   * With a value of 1, the light casts shadows at the maximum intensity, defined by the value of SoShadowGroup::intensity.
   * The default value is 1.
   *
   * @FIELD_SINCE_OIV 2025.2
   */
  SoSFFloat shadowIntensity;

SoEXTENDER public:
  /** @copydoc SoNode::callback */
  virtual void callback( SoCallbackAction* action );

  /** @copydoc SoNode::doAction */
  virtual void doAction( SoAction* action );

SoINTERNAL public:
  /** Return a camera associated to the light */
  virtual void getCameras(SoState* SO_UNUSED_PARAM(state), SoPath* SO_UNUSED_PARAM(scene),
                          float SO_UNUSED_PARAM(visibilityLength), SoNodeList& SO_UNUSED_PARAM(lightCams))
  {
  }

  static void initClass();
  static void exitClass();

  // Returns the light Id.
  int32_t getLightId() const;

  struct LightConfiguration
  {
    bool enabled;
    SbVec4f position;

    SbVec4f spotDirection;
    float spotExponent;
    float spotCutoff;

    SbVec4f ambientIntensity;
    SbVec4f diffuseIntensity;
    SbVec4f specularIntensity;

    float constantAttenuation;
    float linearAttenuation;
    float quadraticAttenuation;

    LightConfiguration()
    {
      enabled = false;
      position = SbVec4f( 0, 0, 0, 0 );
      spotDirection = SbVec4f( 0, 0, -1, 0 );
      spotExponent = 0.0f;
      spotCutoff = 180.0f;
      ambientIntensity = SbVec4f( 0, 0, 0, 1 );
      diffuseIntensity = SbVec4f( 1, 1, 1, 1 );
      specularIntensity = SbVec4f( 1, 1, 1, 1 );
      constantAttenuation = 1.0f;
      linearAttenuation = 0.0f;
      quadraticAttenuation = 0.0f;
    }
  };

  LightConfiguration lightConfiguration;

protected:
  struct Members
  {
    Members();
    ~Members();
    void unref();

    int32_t m_lightId;

    /** Cameras used for shadowmapping */
    SoNodeList m_cameras;
  };
  Members m_members;

  /** Return members  */
  static SoLight::Members* getMembers( const SoLight* sg );

  SoLight(); // Makes this abstract

  virtual ~SoLight();
};

#endif /* _SO_LIGHT_ */
