/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_LIGHT_MODEL_
#define  _SO_LIGHT_MODEL_

#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/elements/SoLightModelElement.h>

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoLightModel
//
//  Light model node.
//
//////////////////////////////////////////////////////////////////////////////

/**
Node that defines the lighting model to use when rendering.

@ingroup LightNodes

@DESCRIPTION
  This node defines the lighting model to use when rendering subsequent
  shapes. The lighting model is specified in the #model field, which is
  PER_VERTEX_PHONG by default.

  SoLightModel supports several types of light model. The following images provide examples
  of rendering the same scene using a different light model.

  @TABLE_0B
      @TR @IMAGE BunnyBaseColor.png width=400
      @TD @IMAGE BunnyPhong.png width=400
      @TD @IMAGE BunnyPBS.png width=400
	  @TR #BASE_COLOR @TD #PER_PIXEL_PHONG @TD #PHYSICALLY_BASED
  @TABLE_END

  When the light model is not #BASE_COLOR, there must be light nodes (e.g. SoDirectionalLight) in the
  scene graph for objects to be easily visible.
  When using the Open Inventor viewer classes (e.g. SoWinExaminerViewer), the viewer will create
  a "headlight" (directional light attached to the camera) if no other light nodes exist when 
  setSceneGraph() is called. When using the Open Inventor viewer components (e.g. SceneExaminer),
  a "headlight" is automatically added to the scene graph.

  You can use a SoEnvironment node to specify additional lighting options:
  - The ambient light component of the Phong reflection model.
  - Light attenuation for SoPointLight and SoSpotLight nodes.
  - Ambient occlusion.

@anchor SoLightModel-base-color-def
# Definition of base color

  All the light models supported by Open Inventor use the concept of base color composed 
  of 3 channels red green blue (rgb). It corresponds to the color of an object before
  being altered by light effects. Unfortunatly the term *base color* is not always
  used in the API and documentation, sometimes it is referred to as *diffuse color*, 
  *rgb color*, a *rgb* or just *color*.

  The following fields define the same concept of *base color* of an object: 
    - SoPhysicalMaterial#baseColor
    - SoMaterial#diffuseColor
    - SoBaseColor#rgb
    - SoPackedColor#orderedRGBA
	- SoColorMap#colorMap
    - SoBufferedShape#colorBuffer
    - SoVertexProperty#orderedRGBA

  The following C++ code samples are equivalent and define a single red opaque base color:

@code
    // using SoMaterial
    mySoMaterial->diffuseColor = SbColor(1,0,0);
	mySoMaterial->transparency = 0 ;

	// using SoPhysicalMaterial
    mySoPhysicalMaterial->baseColor = SbColorRGBA(1,0,0,1);

	// using SoBaseColor
	mySoBaseColor->rgb = SbColor(1,0,0);

	// using SoPackedColor
	mySoPackedColor->orderedRGBA = SbColorRGBA(1,0,0,1).getPackedValue();

	// using SoColorMap
	mySoColorMap->colorMap.setValues(0, 4, new float[4]{ 1, 0, 0, 1 });

	// using SoBufferedShape
	mySoBufferedShape->colorBuffer = new SoCpuBufferObject(new float[4]{ 1, 0, 0, 1 }, 4 * sizeof(float));

	// using SoVertexProperty
	mySoVertexProperty->orderedRGBA = SbColorRGBA(1,0,0,1).getPackedValue();
@endcode

  Changing the value of any of those fields overrides the *base color* of an object.
  This means that if you set 2 different fields to specify a color, the current definition of 
  a color during the scene graph traversal overrides the previous one.
  For instance, the following code sample specifies an opaque red color that overrides the
  semi opaque green.

\if_cpp
@code
    mySoPhysicalMaterial->baseColor = SbColorRGBA(0,1,0,0.5f); // semi opaque green color
    mySoMaterial->diffuseColor = SbColor(1,0,0); // red color
    mySoMaterial->transparency = 0 ;
    myRoot->addChild(mySoPhysicalMaterial);
    myRoot->addChild(mySoMaterial); // this red material overrides the green color
@endcode
\endif
\if_java
@code
    mySoPhysicalMaterial.baseColor.setValue(new SbColorRGBA(0,1,0,0.5f)); // semi opaque green color
    mySoMaterial.diffuseColor.setValue(new SbColor(1,0,0)); // red color
    mySoMaterial.transparency.setValue(0);
    myRoot.addChild(mySoPhysicalMaterial);
    myRoot.addChild(mySoMaterial); // this red material overrides the green color
@endcode
\endif
\if_dotnet
@code
    mySoPhysicalMaterial.baseColor.SetValue(new SbColorRGBA(0,1,0,0.5f)); // semi opaque green color
    mySoMaterial.diffuseColor.SetValue(new SbColor(1,0,0)); // red color
    mySoMaterial.transparency.SetValue(0);
    myRoot.addChild(mySoPhysicalMaterial);
    myRoot.addChild(mySoMaterial); // this red material overrides the green color
@endcode
\endif

# Light model and property nodes

  According to the light model selected, the following notes explain what is taken into account in the
  scene graph, and what is ignored:
  
  Using model #BASE_COLOR
     - Objects are drawn using the base color only. There is no lighting.
     - If a node related to the material is visited after SoBaseColor during the traversal,
       the rgb defined by the SoBaseColor may change.
     - The base color is specified as described [above](#SoLightModel-base-color-def).
     - The SoMaterial fields are ignored except diffuseColor and transparency.
     - The SoPhysicalMaterial fields are ignored except baseColor.
     - The SoLight instances are ignored.
     - The SoEnvironment instances are ignored.

  Using #PER_VERTEX_PHONG or #PER_PIXEL_PHONG
     - The lighting takes into account all fields of SoMaterial.
     - If a node related to the material is visited after the SoMaterial during the traversal,
       the diffuseColor and transparency defined by the SoMaterial may change
     - The base color is specified as described [above](#SoLightModel-base-color-def).
     - The SoPhysicalMaterial fields are ignored except baseColor.
     - The SoLight instances are used.
     - The SoEnvironment instances are used.

  Using #PHYSICALLY_BASED
     - The lighting takes into account all fields of SoPhysicalMaterial.
     - If a node related to the material is visited after the SoPhysicalMaterial during the traversal,
       the baseColor defined by the SoPhysicalMaterial may change
     - The base color is specified as described [above](#SoLightModel-base-color-def).
     - The SoMaterial fields are ignored except diffuseColor and transparency.
     - The SoLight instances are used.
     - The SoEnvironment instances are used.

# Two side lighting

  By default, lighting is only computed for the "front" faces of the geometry, leaving the 
  back face of the geometry dark.  If the normal vector is specified, it determines the front face.
  If normals are computed by Open Inventor, the vertex ordering (see SoShapeHints) determines the front 
  face. To enable "two sided" lighting (lighting both front and back faces) see SoShapeHints.

# Light model and VolumeViz

  The SoLightModel node controls lighting for most VolumeViz shapes, but *not* for SoVolumeRender.
  The lighting for volume rendering is controlled by the SoVolumeRenderingQuality node.

  Two sided lighting is automatically enabled for "slice" shapes, such as SoOrthoSlice.

@FILE_FORMAT_DEFAULT
   LightModel {
   @TABLE_FILE_FORMAT
      @TR model   @TD PER_VERTEX_PHONG
   @TABLE_END
   }

@ACTION_BEHAVIOR
   SoGLRenderAction, SoCallbackAction  
       Sets the current lighting model in the state.

@SEE_ALSO
   SoBaseColor,
   SoEnvironment,
   SoLight,
   SoDirectionalLight,
   SoPointLight,
   SoSpotLight,
   SoPhysicalMaterial,
   SoMaterial

 */
class INVENTOR_API SoLightModel : public SoNode {

  SO_NODE_HEADER(SoLightModel);

 public:
   /** Lighting model */
  enum Model {
    /**
     *  Use only the base color of an object.
     *  Effectively lighting is turned off and light nodes are not considered.
     *  This mode is adapted to render annotation, for data visualization, for 2D scenes, and other special cases.
     */
    BASE_COLOR = SoLightModelElement::BASE_COLOR,
    /**
     *  Use Phong lighting model 
     */
    SoDEPRECATED_ENUM_VALUE(10000, "PHONG is replaced by PER_VERTEX_PHONG to avoid ambiguities with PER_PIXEL_PHONG")
    PHONG = SoLightModelElement::PHONG,
    /**
     *  Use the Vertex-based Phong lighting model (default).
     *  Lighting is computed at each vertex of the shape and the
     *  color of each pixel (fragment) is determined by interpolating between
     *  these values. This model produces less realistic results than pixel-based
     *  lighting, especially for Point and Spot lights.
     */
    PER_VERTEX_PHONG = SoLightModelElement::PER_VERTEX_PHONG,
    /**
     *  Use the Pixel-based Phong lighting model.
     *  A lighting computation is done for each pixel (fragment) affected by the shape. 
     */
    PER_PIXEL_PHONG = SoLightModelElement::PER_PIXEL_PHONG,

    /** Use the physically based lighting model */
    PHYSICALLY_BASED = SoLightModelElement::PHYSICALLY_BASED,
  };

  // Fields
  /**
   * Lighting model to use
   * @useenum{Model}. Default is PER_VERTEX_PHONG.
   */
  SoSFEnum model;

  /**
   * Creates a light model node with default settings.
   */
  SoLightModel();

  /**
   * Set the state of the override field.
   * see SoNode::setOverride doc.
   */
  inline virtual void setOverride(const SbBool state)
  { override.setValue(state); }

  /**
   * Returns the state of the override field.
   */
  inline virtual SbBool isOverride() const
  { return override.getValue(); }

 SoEXTENDER public:
  virtual void doAction(SoAction *action);
  virtual void GLRender(SoGLRenderAction *action);
  virtual void callback(SoCallbackAction *action);

 SoINTERNAL public:
  static void initClass();
  static void exitClass();

  SoSFBool override;

 protected:
  virtual ~SoLightModel();
};

#endif /* _SO_LIGHT_MODEL_ */

