/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : R. ALBOU (May 1998)
**=======================================================================*/

/* include files */
#ifndef _SO_MARKER_SET_
#define _SO_MARKER_SET_

#include <Inventor/nodes/SoPointSet.h>
#include <Inventor/fields/SoMFInt32.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoMFFloat.h>

/*! \cond PRIVATE */
class SbThreadMutex;
class SoMarker;

namespace markerset {
	class SoMarkerSetRenderer;
}
class SoMarkerSizeCache;
/*! \endcond */

/*----------------------------------------------------------------------------*/

/**
 * @VSGEXT Extension of PointSet that draws a small bitmap (symbol) at each 3D location.
 *
 * @ingroup ShapeNodes
 *
 * @DESCRIPTION
 *
 *   This class draws a marker (small image) at each of a set of points located at
 *   the coordinates specified by the #vertexProperty field (from SoVertexShape)
 *   or the current inherited 3D coordinates. For best performance, the
 *   #vertexProperty field is recommended.  SoMarkerSet uses the coordinates in
 *   order, starting with the coordinate at #startIndex. The number of coordinates 
 *   to use (the number of markers in the set) is specified by the #numPoints field.
 *   The default value of this field is -1, which means to use all the coordinates,
 *   so it may not be necessary to explicitly set this field.
 *
 *   You specify the marker to be drawn at each point using the field #markerIndex,
 *   which indexes into the global list of marker definitions. There is a set of
 *   pre-defined markers already in the global list.  You can add your own marker
 *   definitions to this list using the #addMarker functions.  Marker definitions in
 *   the global list can be used by any SoMarkerSet or SoIndexedMarkerSet node.
 *   You can specify fewer index values than the number of markers to be drawn
 *   (the number of coordinates).  In this case the node will cycle through the provided
 *   index values as many times as necessary.  If all the markers in the set will use
 *   the same definition it is only necessary to set the first value in markerIndex.
 *
 *   You can specify a scale factor (> 0) for each marker using the field #markerScale.
 *   You can specify fewer scale values than the number of markers to be drawn, but
 *   in this case the "missing" values are assumed to be 1.  You can also specify a
 *   scale factor to be applied to every marker in the set using the field #markerGlobalScale.
 *   This scale factor is applied in addition to the individual scale factor (if any).
 *
 *   A marker is an image defined by a bitmap and optional color values.  If no
 *   color values are given then image pixels corresponding to "1" bits are drawn
 *   using the current material color (see SoMaterial) and pixels corresponding
 *   to "0" bits are not drawn (whatever is behind the marker shows through).  If
 *   color values are given then each pixel in the image is assigned an RGBA value.
 *   
 *   The coordinates of the markers are transformed by the current cumulative
 *   transformation.
 *
 *   Lighting is not applied to points (i.e., they are rendered with light model BASE_COLOR)
 *   unless the application explicitly sets normal vectors using SoVertexProperty or SoNormal.
 *
 *   Automatic simplification (by skipping points) is controlled by the SoComplexity
 *   field @I value@i.  Only values < 0.5 have any effect.
 *
 *   Note that custom markers defined with addMarker are not automatically
 *   saved in the Open Inventor file when an SoWriteAction is applied to the
 *   scene graph (and therefore are not defined if the scene graph is reloaded).
 *   To save marker definitions in an Open Inventor file use the SoMarker node.
 *
 *   Limitations: @BR
 *   - Since Open Inventor 9.3, markers are rendered using a high performance shader
 *     implementation by default. However the shader implementation requires support
 *     for the OpenGL geometry shader (GL_ARB_geometry_shader4) and texture array
 *     (GL_EXT_texture_array) features (or the equivalent extensions).  If necessary
 *     markers will still be rendered, but using the previous drawPixels() implementation
 *     which is slower.
 *
 *   Shape Antialiasing type is SoShape::POINTS.
 *
 *   Note that SoMarkerSet does not cast a shadow (SoShadowGroup).
 *
 *   The following markers are defined by default:
 *
 *   @IMAGE MARKERS.gif
 *
 *
 * @FILE_FORMAT_DEFAULT
 *    MarkerSet {
 *    @TABLE_FILE_FORMAT
 *       @TR #markerIndex       @TD 0
 *       @TR #markerScale       @TD 1
 *       @TR #markerGlobalScale @TD 1
 *       @TR #vertexProperty    @TD NULL
 *       @TR #startIndex        @TD 0
 *       @TR #numPoints         @TD -1
 *    @TABLE_END
 *    }
 *
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction @BR
 *        Draws markers based on the current coordinates.
 *
 *    SoGetPrimitiveCountAction @BR
 *        Increments the image count by 1 for each marker in the marker set.
 *
 *
 * @SEE_ALSO
 *    SoCoordinate3,
 *    SoFullSceneAntialiasing,
 *    SoIndexedMarkerSet,
 *    SoMarker,
 *    SoPointSet,
 *    SoVertexProperty
 *
 *
 */
class INVENTOR_API SoMarkerSet : public SoPointSet {
  SO_NODE_HEADER(SoMarkerSet) ;

 public:
  /**
   * Specifies the marker index (type of marker) for each marker.
   * Default is 0. Predefined markers are in enum #MarkerType.
   * You can specify fewer index values than the number of markers to be drawn
   * (the number of coordinates).  In this case the node will cycle through the provided
   * index values as many times as necessary.  For example, if all the markers in the set
   * will use the same definition it is only necessary to set the first value in markerIndex.
   *
   * Note: This is an integer valued field. When writing an application you can use
   * enum values to set the field. However in a .iv file you can @I only@i use integer values. @BR
   * For example, you cannot write:
   * \par
   * \code
   * MarkerSet {
   *   markerIndex DIAMOND_FILLED_9_9
   * }
   * \endcode
   *
   * You must write:
   * \par
   * \code
   * MarkerSet {
   *   markerIndex 82
   * }
   * \endcode
   */
  SoMFInt32 markerIndex;

  /** 
   * Specifies the scale factor applied to each marker.
   * Default is 1.  
   * You can specify fewer scale values than the number of markers to be drawn.
   * In this case the "missing" values are looped from the first scale value each necessary time.
   * You can also specify a
   * scale factor to be applied to every marker in the set using the field #markerGlobalScale.
   *
   * Supported only when using shader implementation (this is the default).
   *
   * @FIELD_SINCE_OIV 9.3
   */
  SoMFFloat markerScale;

  /**
   * Specifies a global scale factor applied to all markers.
   * Default is 1. This scale factor is applied in addition to the individual scale factors
   * (if any) specified in the #markerScale field.
   *
   * Supported only when using shader implementation (this is the default).
   *
   * @FIELD_SINCE_OIV 9.3
   */
  SoSFFloat markerGlobalScale;

  /**
   * Constructor.
   */
  SoMarkerSet() ;

  /** Marker type */
  enum MarkerType {
    /** Cross 5x5 */
    CROSS_5_5,
    /** Plus 5x5 */
    PLUS_5_5,
    /** Minus 5x5 */
    MINUS_5_5,
    /** Slash 5x5 */
    SLASH_5_5,
    /** Backslash 5x5 */
    BACKSLASH_5_5,
    /** Bar 5x5 */
    BAR_5_5,
    /** Star 5x5 */
    STAR_5_5,
    /** Y 5x5 */
    Y_5_5,
    /** Lightning 5x5 */
    LIGHTNING_5_5,
    /** Well 5x5 */
    WELL_5_5,

    /** Circle line 5x5 */
    CIRCLE_LINE_5_5,
    /** Square line 5x5 */
    SQUARE_LINE_5_5,
    /** Diamond line 5x5 */
    DIAMOND_LINE_5_5,
    /** Triangle line 5x5 */
    TRIANGLE_LINE_5_5,
    /** Rhombus line 5x5 */
    RHOMBUS_LINE_5_5,
    /** Hourglass line 5x5 */
    HOURGLASS_LINE_5_5,
    /** Satellite line 5x5 */
    SATELLITE_LINE_5_5,
    /** Pine line 5x5 */
    PINE_TREE_LINE_5_5,
    /** Caution line 5x5 */
    CAUTION_LINE_5_5,
    /** Ship line 5x5 */
    SHIP_LINE_5_5,

    /** Circle filled 5x5 */
    CIRCLE_FILLED_5_5,
    /** Square filled 5x5 */
    SQUARE_FILLED_5_5,
    /** Diamond filled 5x5 */
    DIAMOND_FILLED_5_5,
    /** Triangle filled 5x5 */
    TRIANGLE_FILLED_5_5,
    /** Rhombus filled 5x5 */
    RHOMBUS_FILLED_5_5,
    /** Hourglass filled 5x5 */
    HOURGLASS_FILLED_5_5,
    /** Satellite filled 5x5 */
    SATELLITE_FILLED_5_5,
    /** Pine filled 5x5 */
    PINE_TREE_FILLED_5_5,
    /** Caution filled 5x5 */
    CAUTION_FILLED_5_5,
    /** Ship filled 5x5 */
    SHIP_FILLED_5_5,

    /** Cross 7x7 */
    CROSS_7_7,
    /** Plus 7x7 */
    PLUS_7_7,
    /** Minus 7x7 */
    MINUS_7_7,
    /** Slash 7x7 */
    SLASH_7_7,
    /** Backslash 7x7 */
    BACKSLASH_7_7,
    /** Bar 7x7 */
    BAR_7_7,
    /** Star 7x7 */
    STAR_7_7,
    /** Y 7x7 */
    Y_7_7,
    /** Lightning 7x7 */
    LIGHTNING_7_7,
    /** Well 7x7 */
    WELL_7_7,

    /** Circle line 7x7 */
    CIRCLE_LINE_7_7,
    /** Square line 7x7 */
    SQUARE_LINE_7_7,
    /** Diamond line 7x7 */
    DIAMOND_LINE_7_7,
    /** Triangle line 7x7 */
    TRIANGLE_LINE_7_7,
    /** Rhombus line 7x7 */
    RHOMBUS_LINE_7_7,
    /** Hourglass line 7x7 */
    HOURGLASS_LINE_7_7,
    /** Satellite line 7x7 */
    SATELLITE_LINE_7_7,
    /** Pine line 7x7 */
    PINE_TREE_LINE_7_7,
    /** Caution line 7x7 */
    CAUTION_LINE_7_7,
    /** Ship line 7x7 */
    SHIP_LINE_7_7,

    /** Circle  filled 7x7 */
    CIRCLE_FILLED_7_7,
    /** Square filled 7x7 */
    SQUARE_FILLED_7_7,
    /** Diamond filled 7x7 */
    DIAMOND_FILLED_7_7,
    /** Triangle filled 7x7 */
    TRIANGLE_FILLED_7_7,
    /** Rhombus filled 7x7 */
    RHOMBUS_FILLED_7_7,
    /** Hourglass filled 7x7 */
    HOURGLASS_FILLED_7_7,
    /** Satellite filled 7x7 */
    SATELLITE_FILLED_7_7,
    /** Pine filled 7x7 */
    PINE_TREE_FILLED_7_7,
    /** Caution filled 7x7 */
    CAUTION_FILLED_7_7,
    /** Ship filled 7x7 */
    SHIP_FILLED_7_7,

    /** Cross 9x9 */
    CROSS_9_9,
    /** Plus 9x9 */
    PLUS_9_9,
    /** Minus 9x9 */
    MINUS_9_9,
    /** Slash 9x9 */
    SLASH_9_9,
    /** Backslash 9x9 */
    BACKSLASH_9_9,
    /** Bar 9x9 */
    BAR_9_9,
    /** Star 9x9 */
    STAR_9_9,
    /** Y 9x9 */
    Y_9_9,
    /** Lightning 9x9 */
    LIGHTNING_9_9,
    /** Well 9x9 */
    WELL_9_9,

    /** Circle line 9x9 */
    CIRCLE_LINE_9_9,
    /** Square line 9x9 */
    SQUARE_LINE_9_9,
    /** Diamond line 9x9 */
    DIAMOND_LINE_9_9,
    /** Triangle line 9x9 */
    TRIANGLE_LINE_9_9,
    /** Rhombus line 9x9 */
    RHOMBUS_LINE_9_9,
    /** Hourglass line 9x9 */
    HOURGLASS_LINE_9_9,
    /** Satellite line 9x9 */
    SATELLITE_LINE_9_9,
    /** Pine line 9x9 */
    PINE_TREE_LINE_9_9,
    /** Caution line 9x9 */
    CAUTION_LINE_9_9,
    /** Ship line 9x9 */
    SHIP_LINE_9_9,

    /** Circle filled 9x9 */
    CIRCLE_FILLED_9_9,
    /** Square filled 9x9 */
    SQUARE_FILLED_9_9,
    /** Diamond filled 9x9 */
    DIAMOND_FILLED_9_9,
    /** Triangle filled 9x9 */
    TRIANGLE_FILLED_9_9,
    /** Rhombus filled 9x9 */
    RHOMBUS_FILLED_9_9,
    /** Hourglass filled 9x9 */
    HOURGLASS_FILLED_9_9,
    /** Satellite filled 9x9 */
    SATELLITE_FILLED_9_9,
    /** Pine filled 9x9 */
    PINE_TREE_FILLED_9_9,
    /** Caution filled 9x9 */
    CAUTION_FILLED_9_9,
    /** Ship filled 9x9 */
    SHIP_FILLED_9_9
  };

  /**
   * Return the number of defined markers
   */
  static int getNumDefinedMarkers();

  /**
   * Add a new marker with index @B markerIndex @b. If the marker exists, it is
   * replaced. @B size @b is the size of the marker in pixels. @B bytes @b is the
   * marker bitmap. The bitmap is arranged row by row from left to right and top
   * to bottom (according to the parameter @B isUpToDown @b). Each byte in the
   * bitmap corresponds to eight pixels.  Open Inventor makes a copy of the bitmap.
   *
   * @B isLSBFirst @b: If TRUE, bits are ordered within a byte from least significant
   * to most significant; otherwise the first bit in each byte is the most
   * significant one.
   *
   * @B isUpToDown @b: If TRUE, marker bitmap is described from top to bottom
   * (bytes[0] is the left top corner of the bitmap), otherwise from bottom to
   * top (bytes[0] is the bottom left corner).
   */
  static void addMarker(int markerIndex, const SbVec2s &size, const unsigned char *bytes,
                        SbBool isLSBFirst = TRUE, SbBool isUpToDown = TRUE);

  /**
   * Add a new colored marker with index @B markerIndex @b. If the marker exists, it is
   * replaced. @B size @b is the size of the marker in pixels. @B bytes @b is the
   * marker bitmap. @B orderedRGBA @b is an array of per-pixel color values to apply
   * to the marker.  The bitmap is arranged row by row from left to right and top to
   * bottom (according to the parameter @B isUpToDown @b). Each byte in the bitmap
   * corresponds to eight pixels. Each value in orderedRGBA corresponds to one pixel.
   * Open Inventor makes a copy of the bitmap and color values.
   *
   * @B isLSBFirst @b: If TRUE, bits are ordered within a byte from least significant
   * to most significant; otherwise the first bit in each byte is the most
   * significant one.
   8
   * @B isUpToDown @b: If TRUE, marker bitmap and color array are described from top
   * to bottom (bytes[0] is the left top corner of the bitmap), otherwise from bottom
   * to top (bytes[0] is the bottom left corner).
   */
  static void addMarker(int markerIndex, const SbVec2s &size,
                        const unsigned char *bytes,
                        const uint32_t *orderedRGBA,
                        SbBool isLSBFirst = TRUE, SbBool isUpToDown = TRUE);

  /**
   * Add a new marker with index @B markerIndex @b using an existing marker definition.
   * If the marker exists, it is replaced.
   */
  static void addMarker(int markerIndex, SoMarker* marker );

  /**
   * Retrieve the description of the marker with index @B markerIndex @b. Returns
   * TRUE if the marker exists, otherwise returns FALSE and the parameters are
   * not modified.
   * [OIV-WRAPPER-ARG IN,OUT,OUT&ARRAY{size[0] % 8 != 0? ((int)(size[0] / 8) + 1)*size[1]: size[0] / 8 * size[1]},OUT]
   * [OIVJAVA-WRAPPER PACK{MarkerInfo}]
   */
  static SbBool getMarker(int markerIndex,
                          SbVec2s &size,
                          const unsigned char *&bytes,
                          SbBool &isLSBFirst);

  /**
   * Retrieve the description of the colored marker with index @B markerIndex @b.
   * Returns TRUE if the marker exists, otherwise returns FALSE and the
   * parameters are not modified.  If the marker does not have color values,
   * orderedRGBA will point to a single uint32_t containing the value zero.
   * [OIV-WRAPPER-ARG IN,OUT,OUT&ARRAY{size[0] % 8 != 0? ((int)(size[0] / 8) + 1)*size[1]: size[0] / 8 * size[1]},ARRAY{size[0]*size[1]},OUT]
   * [OIVJAVA-WRAPPER NAME{getColoredMarker},PACK{ColoredMarkerInfo}]
   */
  static SbBool getMarker(int markerIndex,
                          SbVec2s &size,
                          const unsigned char *&bytes,
                          const uint32_t *&orderedRGBA,
                          SbBool &isLSBFirst);

  /**
   * Retrieve the definition of the marker with index @B markerIndex @b. Returns
   * NULL if the marker does not exist.
   */
  static SoMarker* getMarker( int markerIndex );

  /**
   * Remove the marker with index @B markerIndex @b. Returns TRUE if the marker
   * was successfully removed, FALSE otherwise.
   */
  static SbBool removeMarker(int markerIndex);

protected:
  // Generates markers representing marker set
  virtual void generatePrimitives(SoAction *action);

  // Destructor.
  virtual ~SoMarkerSet();

SoEXTENDER public:
  virtual void  GLRender(SoGLRenderAction *action);
  virtual void  getPrimitiveCount(SoGetPrimitiveCountAction *action);

  // Computes bounding box
  virtual void  computeBBox( SoAction *action, SbBox3f &box, SbVec3f &center );

  static SbBool isMarkerBitSet(int markerIndex, int bitNumber);

SoINTERNAL public:
  static void initClass();
  static void exitClass();

  // manage notification
  virtual void notify(SoNotList *list);

private:

  /** Markerset renderer. */
  markerset::SoMarkerSetRenderer* m_MarkerSetShaderRenderer;

  /** Helper function for updating marker set renderer */
  void updateRenderer(markerset::SoMarkerSetRenderer* renderer, SoNotList* list);

  /** Factory method for choosing and initializing renderer objects. */
  markerset::SoMarkerSetRenderer* chooseRenderer(SoState* state);

  // Cache storage for marker sizes during rayPick action
  SoRayPickAction* m_currentPickAction;
  SoMarkerSizeCache* m_markerSizeCache;
};

#endif // !_SO_MARKER_SET_


