/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_MATERIAL_
#define  _SO_MATERIAL_

#include <Inventor/fields/SoMFColor.h>
#include <Inventor/fields/SoMFFloat.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/nodes/SoNode.h>

/**
 Surface material definition node.

 @ingroup LightNodes

 @DESCRIPTION
   This node defines the current surface material properties for all subsequent
   shapes. SoMaterial sets several components of the current material during
   traversal.  The ambientColor, diffuseColor, emissiveColor, specularColor
   and shininess fields are interpreted according to the classic OpenGL lighting
   model. The transparency field is effectively the inverse of "opacity" or
   "alpha value".

   As described @ref SoLightModel-base-color-def "here", the #diffuseColor and #transparency of
   this class can be overridden by other property nodes.

   If the current SoLightModel#model is set to BASE_COLOR or PHYSICALLY_BASED,
   only the #diffuseColor and #transparency fields are used from this SoMaterial class.

   Multiple values can be specified for the #diffuseColor and
   #transparency fields. Different shapes interpret materials with multiple values
   differently. To bind materials to shapes, use an SoMaterialBinding node.

# Several other nodes can be used to set diffuse color and transparency for geometry.
   
   If the other color values are not needed, these nodes use a little less memory
   than an SoMaterial node, especially if multiple color values are specified.
   Generally SoVertexProperty is the most efficient mechanism and may provide better
   performance than using SoMaterial.  Note that for these nodes transparency is
   set as "alpha value" (inverse of transparency).
   - Diffuse color (only) can also be specified using an SoBaseColor node.
   - Diffuse color and transparency can also be specified using an SoPackedColor node.
   - Diffuse color and transparency can also be specified for polygonal geometry using
     the orderedRGBA field of SoVertexProperty.

# Lighting and material RGB values:
   
   The color components specified for lights mean something different than for 
   materials. For a light, the numbers correspond to a percentage of full 
   intensity for each color. If the R, G, and B values for a light's color are 
   all 1.0, the light is the brightest possible white. If the values are 0.5, 
   the color is still white, but only at half intensity, so it appears gray. 
   If R=G=1 and B=0 (full red and green with no blue), the light appears yellow.
   The intensity can also be modulated using the #SoLight::intensity field.

   For materials, the numbers correspond to the reflected percentages of those 
   colors. So if R=1, G=0.5, and B=0 for a material, that material reflects all 
   the incoming red light, half the incoming green, and none of the incoming 
   blue light. In other words, if an OpenGL light has components (LR, LG, LB), 
   and a material has corresponding components (MR, MG, MB), then, ignoring all 
   other reflectivity effects, the light that arrives at the eye is given by 
   (LR*MR, LG*MG, LB*MB).  As a result, for example, shining a pure blue light
   on a pure red cone has no visible effect.

   Similarly, if you have two lights that send (R1, G1, B1) and (R2, G2, B2) 
   to the eye, the components are added, giving (R1+R2, G1+G2, B1+B2). If any 
   of the sums are greater than 1 (corresponding to a color brighter than the 
   hardware can display), the component is clamped to 1.

# Override material:

   To force all geometry following/below this node to use specific color and
   transparency values, call the setOverride() method with true.  Overriding
   the diffuse color and transparency overrides the color and transparency
   values in other nodes including SoPhysicalMaterial, SoPackedColor and SoVertexProperty.
   This can be useful, for example, to highlight a selected object.

   It is also possible to override only a subset of the SoMaterial fields.
   If, for example, you only want to override the diffuse color, but not the other
   values, call setIgnored(true) on the fields that you do not want to override.
   But note that this selective override technique only works on other SoMaterial
   nodes!  For SoPackedColor and SoVertexProperty, the diffuseColor and transparency
   values are bound together and cannot be overridden separately.

# VolumeViz shapes:

   For scalar (non-RGBA) volumes, the color and alpha value of a voxel is affected
   by two nodes. SoMaterial's diffuseColor field specifies the "base" color and
   alpha values for all voxels. SoTransferFunction specifies color and alpha values
   based on the voxel value.  The final voxel color and alpha (before lighting and other
   effects) is computed by multiplying these two color and alpha values. The default
   material is 0.8, 0.8, 0.8, 1 (fully opaque gray). The 0.8 value for R, G and B
   allows lighting to increase the brightness of the voxel. For slice rendering
   without lighting, the application may want to set the material to 1, 1, 1, 1 so
   that only the SoTransferFunction affects the voxel color and alpha. Effectively
   the material alpha value (aka transparency) is a "global" multiplier that can
   be used to increase or decrease the transparency of all voxels in the volume.

 @FILE_FORMAT_DEFAULT
    Material {
    @TABLE_FILE_FORMAT
       @TR ambientColor    @TD 0.2 0.2 0.2
       @TR diffuseColor    @TD 0.8 0.8 0.8
       @TR specularColor   @TD 0 0 0
       @TR emissiveColor   @TD 0 0 0
       @TR shininess       @TD 0.2
       @TR transparency    @TD 0
    @TABLE_END
    }

 @ACTION_BEHAVIOR
    SoGLRenderAction,  SoCallbackAction  
        Sets the ambient color, the diffuse color, the specular color, the emissive
        color, the shininess, and the transparency of the current material.
    Sets: SoMaterialElement

 @SEE_ALSO
    SoBaseColor,
    SoLightModel,
    SoMaterialBinding,
    SoPackedColor,
    SoVertexProperty,

 */
class INVENTOR_API SoMaterial : public SoNode {

  SO_NODE_HEADER(SoMaterial);

 public:
  // Fields
  /**
   * Ambient color of the surface.
   * Default is 0.2 0.2 0.2.
   *
   * Ambient reflectance affects the overall color of the object. Because diffuse reflectance
   * is brightest where an object is directly illuminated, ambient reflectance is most noticeable
   * where an object receives no direct illumination. An object's total ambient reflectance is
   * affected by the global ambient light (see SoEnvironment) and ambient light from individual
   * light sources. Like diffuse reflectance, ambient reflectance is not affected by the position
   * of the camera.
   */
  SoMFColor           ambientColor;
  /**
   * Diffuse color(s) of the surface. A diffuse color is similar to a base color as described 
   * [here](#SoLightModel-base-color-def).
   * Default is 0.8 0.8 0.8.
   *
   * Diffuse reflectance plays the most important role in determining the appearance of an object.
   * It's affected by the color of the incident light(s) and the angle of each incident light
   * relative to the object's normal direction. (It's most intense where the incident light
   * falls perpendicular to the surface.) The position of the camera doesn't affect diffuse
   * reflectance at all.
   */
  SoMFColor           diffuseColor;
  /**
   * Specular color of the surface.
   * Default is 0 0 0.
   *
   * Specular reflection from an object produces highlights. Unlike ambient and diffuse reflection,
   * the amount of specular reflection does depend on the location of the camera - it's brightest
   * along the direct angle of reflection. To see this, imagine looking at a metallic ball outdoors
   * in the sunlight. As you move your head, the highlight created by the sunlight moves with you
   * to some extent. However, if you move your head too much, you lose the highlight entirely.
   *
   * This field specifies the color of the reflected light. The #shininess field controls the size
   * and brightness of the highlight.
   */
  SoMFColor           specularColor;
  /**
   * Emissive color of the surface.
   * Default is 0 0 0.
   *
   * Emissive color makes an object appear to be giving off light of that color, independent of
   * any lights (or lack of lights) in the scene.  It can be useful for highlighting selected
   * objects in the scene.
   */
  SoMFColor           emissiveColor;
  /**
   * Shininess coefficient of the surface. Values can range from 0.0 to 1.0.
   * Default is 0.2.
   *
   * The dot product of the vector reflected by the surface normal and 
   * the inverted light vector is raised to the "Shininess" power. 
   * The higher the shininess number, the smaller the resulting specular 
   * highlight turns out to be.
   */
  SoMFFloat           shininess;

  /**
   * Transparency value(s) of the surface. Values can range from 0.0 for opaque
   * surfaces to 1.0 for completely transparent surfaces. Default is 0 (opaque).
   *
   * Transparency is the inverse of "opacity" or "alpha" value.
   */
  SoMFFloat           transparency;

  /**
   * Creates a material node with default settings.
   */
  SoMaterial();

  /**
   * Set the state of the override field.
   * see SoNode::setOverride doc.
   */
  inline virtual void setOverride(const SbBool state)
  { override.setValue(state); }

  /**
   * Returns the state of the override field.
   */
  inline virtual SbBool isOverride() const
  { return override.getValue(); }

 SoEXTENDER public:
  virtual void        doAction(SoAction *action);
  virtual void        GLRender(SoGLRenderAction *action);
  virtual void        callback(SoCallbackAction *action);
  virtual void        write(SoWriteAction *writeAction);
  virtual void        rayPick(SoRayPickAction *action);

 SoINTERNAL public:
  static void         initClass();
  static void         exitClass();

  SoSFBool override;

 protected:
  virtual ~SoMaterial();

 private:

#if SoDEPRECATED_BEGIN(10000)
   /**
   * DirectViz rendering only. Specifies that all the shapes using this material node will
   * receive shadows from other shapes when rendering using DirectViz.
   *
   * @FIELD_SINCE_OIV 6.1
   */
  SoDEPRECATED_FIELD(10000,"No longer used.")
  SoSFBool receiveShadow;

   /**
   * DirectViz rendering only. Specifies the reflective color of the shape.
   * When set to black (0,0,0), the object is not reflective.
   * To simulate a perfect mirror, the reflectiveColor should be set to white (1, 1, 1),
   * and all other color field values to black (0, 0, 0).
   * The reflective component is added to the overall color of the object.
   *
   * @FIELD_SINCE_OIV 6.1
   */
  SoDEPRECATED_FIELD(10000,"No longer used.")
  SoMFColor reflectiveColor;

  // friend class to access these private fields and avoid #35326 regression
  friend class SoIfWeeder;
  friend class SoIfBuilder;
  friend class SoIfShape;
  friend class SoIfMerger;
#endif /** @DEPRECATED_END */

};

#endif /* _SO_MATERIAL_ */

