/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (Aug 2013)
**=======================================================================*/


#ifndef  _SO_MULTIPLE_INSTANCE_
#define  _SO_MULTIPLE_INSTANCE_

#include <Inventor/nodes/SoMultipleInstanceBase.h>
#include <Inventor/fields/SoSFUInt32.h>
#include <Inventor/fields/SoMFInstanceParameter.h>
#include <Inventor/SbPImpl.h>

class SoShaderProgram;
class SoGLShaderProgramElement;

SO_PIMPL_PUBLIC_DECLARATION(SoMultipleInstance)

/** 
* Group node that renders multiple instances of its children.
* 
* @ingroup GroupNodes
* 
* @DESCRIPTION
*   This group node renders its children multiple times applying
*   different per-instance attributes.
*   
*   This node issues a single multi-instance OpenGL draw call that automatically applies
*   per-instance parameters (SoInstanceParameter objects) set in the #parameters field.
*   Some predefined parameter names can be used for common instance parameters like
*   position, rotation, scale factor, or color. Parameter objects with these names are
*   automatically managed and are recognized and used by the default shaders.
*   Applications can define additional parameter objects to be used by custom shaders.
*   Custom shaders can also be used to set per-instance colors and so on.
*
*   The predefined instance parameters are typically set by the application to position the
*   instances in space.
*   The application does not need to set predefined instance parameter values if the
*   default values are OK. For example, you may not need to set per-instance rotations.
*   In that case you can just set the desired parameters for translating and/or scaling instances.
*   If the application does not set any instance parameters, all the instances will be drawn
*   at the same 3D position without any rotation or scaling factor applied.
*
*   Some methods are provided by SoInstanceParameter to create these per-instance parameters:
*   see the SoInstanceParameter and SoMFInstanceParameter documentation.
*
*   The predefined names for the SoInstanceParameter @a name field are the following.
*   However we recommend using the SoInstanceParameter static method getPredefinedParameterName()
*   to ensure future compatibility.
*   - "OivShapeInstanceTranslation": @BR
*     Data is an array of SbVec3f translation values (3 floats) to apply to the instances.
*
*   - "OivShapeInstanceScale": @BR
*     Data is an array of SbVec3f scale factor values (3 floats) to apply to the instances.
*
*   - "OivShapeInstanceRotation": @BR
*     Data is an array of SbRotation values (4 floats representing a quaternion) to apply to the instances.
*
*   - "OivShapeInstanceColor": @BR
*     Data is an array of SbColor values (3 floats) to apply to the instances.
*
*   - "OivShapeInstanceTransparency": @BR
*     Data is an array of float values to apply to the instances.
*
*   - "OivShapeInstanceVisibility": @BR
*     Data is an array of unsigned int32 values to apply to the instances.

*   The OivShapeInstanceTranslation, OivShapeInstanceScale and OivShapeInstanceRotation
*   parameters (if present) are combined together to create an equivalent matrix.
*   This matrix is available to application shader programs using the GLSL function
*   \code mat4 OivInstanceMatrix() \endcode
*   If no instance parameters were set, the identity matrix is returned
*
*   The OivShapeInstanceColor and OivShapeInstanceTransparency parameters affect, respectively,
*   the diffuse color and the transparency of each instance. Note that when one of these parameters is defined,
*   it overrides the diffuse color or the transparency value of a SoMaterial node.
*
* @B Notes:@b
* - The number of elements in each SoVertexShaderParameter must be greater than or 
*   equal to numInstances * SoVertexShaderParameter::instanceDivisor. If not,
*   the attributes will be ignored with warning messages in debug mode.
*
* - Traversing the <B>N</B>th child sets the current switch value to @B N @b, for
*   use with inherited switch values (see SoSwitch).
*
* - Although SoMultipleInstance is derived from SoGroup, it @I behaves@i like
*   an SoSeparator (traversal state is pushed and popped).
*
* @B Limitations:@b
*   - Multiple instance nodes cannot currently be nested. @BR
*     SoMultipleInstance, SoMultipleCopy and SoArray nodes may not be children.
*
*   - This node requires at least version 3.3 of OpenGL.
*
* \if_cpp
* @EXAMPLE
* \code
* // Geometry to be instanced.
* SoAlgebraicCone* geometry = new SoAlgebraicCone();
* 
* // Position data for multiple instances.
* int numPositions = 4;
* SoCpuBufferObject* positionBuffer = new SoCpuBufferObject();
*   positionBuffer->setSize( numPositions * sizeof(SbVec3f) );
*   SbVec3f* positions = (SbVec3f*)positionBuffer->map( SoBufferObject::SET );
*   positions[0].setValue( -2, -2, 0 );
*   positions[1].setValue(  2, -2, 0 );
*   positions[2].setValue(  2,  2, 0 );
*   positions[3].setValue( -2,  2, 0 );
*   positionBuffer->unmap();
* 
* // Parameter object for multiple instance positions.
* SoInstanceParameter* positionParam =
*   SoInstanceParameter::createPredefinedParameter( SoInstanceParameter::TRANSLATION, positionBuffer );
* 
* // Multiple instance node.
* SoMultipleInstance* multiInstance = new SoMultipleInstance();
*   multiInstance->numInstances = numPositions;
*   multiInstance->parameters   = positionParam;
*   multiInstance->addChild( geometry );
* 
* // Add to scene graph
* root->addChild( multiInstance );
* \endcode
* \endif
*
* @FILE_FORMAT_DEFAULT
*    MultipleInstance {
*    @TABLE_FILE_FORMAT
*       @TR numInstances       @TD 1
*       @TR numBatches         @TD 1
*       @TR parameters         @TD
*    @TABLE_END
*    }
* 
* @SEE_ALSO
*    SoInstanceParameter, SoVertexShaderParameter, SoBufferedShape,
*    SoAlgebraicShape, SoShaderProgram, SoMultipleCopy, SoArray, SoSwitch
* 
* @NODE_SINCE_OIV 9.5
*/
class INVENTOR_API SoMultipleInstance : public SoMultipleInstanceBase
{
  SO_NODE_HEADER(SoMultipleInstance);
  SO_PIMPL_PUBLIC_HEADER(SoMultipleInstance)

public:
  /**
  * Number of shape instances to render. Default is 1.
  */
  SoSFUInt32 numInstances;

 /**
  * When this field is set to 1 (default), all the instances are rendered in
  * one batch.
  * When greater than 1, rendering will be done in numBatches passes,
  * in order to spend less time in the driver code.
  * This is useful when rendering GPU costly shapes (complex shader, big number of vertices...),
  * because the process may spend too much time in the driver leading to a driver 
  * timeout on Windows systems.
  *
  * Limitation: When #numBatches is greater than 1, only instance parameters with a divisor equal to 1 are supported.
  *
  * Default is 1.
  */
  SoSFUInt32 numBatches;

  /**
  * List of per-instance attributes of type SoVertexShaderParameters.
  * The predefined names can be queried using SoInstanceParameter::getPredefinedParameterName().
  *
  * The number of elements in each SoVertexShaderParameter must be greater than or 
  * equal to numInstances * SoInstanceParameter::divisor,
  * if not, the attributes will be ignored with warning messages in debug mode.
  *
  * Some predefined SoVertexShaderParameter::name are automaticly managed/interpreted:
  * - OivShapeInstanceTranslation:
  *      Handles an array of vec3f translation to apply to vertices of each instance.
  *
  * - OivShapeInstanceScale:
  *      Handles an array of vec3f scale factor applied to vertices of each instance.
  *
  * - OivShapeInstanceRotation:
  *      Handles an array of rotation vec4f to apply to vertices of each instance.
  *
  * - OivShapeInstanceColor:
  *      Handles an array of vec3f color applied to vertices of each instance.
  *
  * - OivShapeInstanceTransparency:
  *      Handles an array of float transparency applied to vertices of each instance.
  *
  * OivShapeInstanceTranslation, OivShapeInstanceScale, OivShapeInstanceRotation
  * are combined together to create an equivalent matrix 
  * available in GLSL shaders with the function
  * \code mat4 OivInstanceMatrix() \endcode
  * If no parameters are present then an identity matrix is returned.
  *
  * OivShapeInstanceColor affects the diffuse color of each instance.
  */
  SoMFInstanceParameter parameters;

  /**
  * Creates a multiple instance node with default settings.
  */
  SoMultipleInstance();

SoEXTENDER public:
  /** Implements SoGLRenderAction traversal. */
  virtual void GLRender( SoGLRenderAction *action );

  /** Implements SoGetBoundingBoxAction traversal. */
  virtual void getBoundingBox( SoGetBoundingBoxAction *action );

SoINTERNAL public:
  /** @copydoc SoNode::initClass() */
  static void initClass();

  /** @copydoc SoNode::exitClass() */
  static void exitClass();

  /** @copydoc SoNode::notify() */
  virtual void notify(SoNotList* notificationList);

protected:
  /** Protected destructor */
  virtual ~SoMultipleInstance();
    
  // needed to access SoMultipleInstanceImpl from this class
  friend class SoInstanceMatrixAccessorAttributes;
    
};

#endif /* _SO_MULTIPLE_INSTANCE_ */

