/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : R. ALBOU (Jun 1998)
**=======================================================================*/

                                                             /* include files */
#ifndef  _SO_NURBS_PROPERTY_
#define  _SO_NURBS_PROPERTY_

#include <Inventor/fields/SoSFBitMask.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/fields/SoSFColor.h>
#include <Inventor/fields/SoSFUShort.h>
#include <Inventor/fields/SoSFFloat.h>

#include <Inventor/nodes/SoNode.h>

class SoGLRenderAction ;
class SoCallbackAction ;

/*----------------------------------------------------------------------------*/
/**
 * Node that defines additional properties for rendering a NURBS surface.
 * 
 * @ingroup PropertyNodes
 * 
 * @DESCRIPTION
 *   This node defines an additional set of properties used by SoNurbsSurface when
 *   rendering NURBS surfaces.
 *   
 *   For a large NURBS surface, it is sometimes more effective to display two central
 *   isoparametric curves rather than facet boundaries (as in wireframe mode) because
 *   the latter may provide an overly complicated view of the surface.
 *   
 *   This node allows you to specify the NURBS draw style; the color of the trim
 *   curves, boundaries and isoparametric curves; the line pattern of the
 *   isoparametric curves; the number of sample points between trim curves
 *   control points; the type of tessellation; angle and distance for adaptive
 *   tessellation. 
 *   
 *   The NURBS draw style is a bitmask specifying which combination of the following
 *   representations to use for displaying the NURBS: normal display, trim curves and
 *   boundaries, and/or central isoparametric curves.
 *   
 *   Turning @I off @i the normal NURBS surface display, and turning @I on @i the
 *   display of trim curves, boundaries and isoparametric curves can dramatically
 *   speed up rendering of large models.
 *
 *   When in ADAPTIVE tessellation mode, regions with a low curvature will be less tessellated
 *   than regions with a high curvature. 
 * 
 * @FILE_FORMAT_DEFAULT
 *    NurbsProperty {
 *    @TABLE_FILE_FORMAT
 *       @TR drawStyle               @TD NORMAL
 *       @TR numSamplePoints         @TD -1
 *       @TR color                   @TD 1 1 1
 *       @TR isoParamCurvesPattern   @TD 0xFFFF
 *       @TR tessellationType        @TD CONSTANT
 *       @TR tessellationDistance    @TD 0.01
 *       @TR tessellationAngle       @TD 0.35
 *    @TABLE_END
 *    }
 *
 * @ACTION_BEHAVIOR
 *    Sets: SoNurbsPropertyElement
 * 
 * @SEE_ALSO
 *    SoNurbsSurface
 * 
 * 
 */
class INVENTOR_API SoNurbsProperty : public SoNode {  
  SO_NODE_HEADER(SoNurbsProperty) ;

 public:

   /** Draw style */
  enum DrawStyle {
    /**
     *  Draw the NURBS surface normally 
     */
    NORMAL                      = 0x01,
    /**
     *  Draw the boundaries of the surface and trim curves 
     */
    BOUNDARIES_AND_TRIM_CURVES  = 0x02,
    /**
     *  Draw two isoparametric curves in the middle of the surface (one for U = 0.5 and
     * one for V=0.5) 
     */
    CENTRAL_ISO_PARAM_CURVES    = 0x04
  } ;

  /** Tesselation type */
  enum TessellationType { 
    /**
     *  The tessellation step size is constant and the same for both the
     *  U and V axes. The tessellation step size depends on SoComplexity::value.
     */
    CONSTANT,
    /**
     * The tessellation depends on the curvature of the geometry.
     * Regions with a low curvature will be less tessellated
     * than regions with a high curvature. 
     */
    ADAPTIVE
  };
  
  // Fields
  /**
   * Indicates which part of the NURBS surface to render.
   * Use enum #DrawStyle. Default is NORMAL.
   */
  SoSFBitMask  drawStyle ;
  /**
   * #numSamplePoints is used during trim curve tessellation. It specifies the
   * exact number of points to generate between two trim curve control points. If set
   * to -1 (the default), the number of points generated is computed from the
   * SoComplexity node. Setting this field to a small value (less than 3) can reduce
   * the number of triangles generated by the tessellation and thus increase the
   * drawing performance of trimmed NURBS surfaces.
   * 
   */
  SoSFInt32    numSamplePoints ;
  /**
   * Contains the color used to render trim curves and isoparametric curves.
   * 
   */
  SoSFColor    color ;
  /**
   * Sets a specific line pattern for isoparametric curves.
   * 
   */
  SoSFUShort    isoParamCurvesPattern ;

  /**
   * Specifies the tessellation type.
   * * @useenum{TessellationType}. Default is CONSTANT.
   *
   * @FIELD_SINCE_OIV 6.0
   */
  SoSFEnum tessellationType;

  /**
   * Used when tessellationType is set to ADAPTIVE. 
   * The maximum tessellation step size (curvilinear distance) is this fraction of the 
   * diagonal of the bounding box of the surface or the group of surfaces (SoNurbsGroup).
   * Increasing this value make the tessellation more dependent on #tessellationAngle.
   * Default value is 0.01. That is, the distance must not be more than 1% of the
   * diagonal of the bounding box.
   * 
   * @FIELD_SINCE_OIV 6.0
   */
  SoSFFloat tessellationDistance;

  /**
   * Used when tessellationType is set to ADAPTIVE. Specifies the maximum angle (in radians) between
   * the normals of two adjacent tessellated faces. Decreasing this value adds more tessellation 
   * points on flat parts of the surface.
   * Default value is 0.35
   *
   * @FIELD_SINCE_OIV 6.0
   */
  SoSFFloat tessellationAngle;

  /**
   * Constructor.
   */
  SoNurbsProperty() ;

 SoEXTENDER public:
  virtual void  doAction(SoAction *action);
  virtual void  callback(SoCallbackAction *action);
  virtual void  GLRender(SoGLRenderAction *action);
  virtual void  rayPick(SoRayPickAction *action);
  virtual void  getPrimitiveCount(SoGetPrimitiveCountAction *action);
  virtual void  getBoundingBox(SoGetBoundingBoxAction *action);
  
  /*----------------------------------------------------------------------------*/
  
 SoINTERNAL public:
  static void initClass() ;
  static void exitClass() ;
  
 protected:
  // Destructor
  virtual ~SoNurbsProperty() ;
} ;
/*----------------------------------------------------------------------------*/

#endif /* _SO_NURBS_PROPERTY_ */

