/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_PACKED_COLOR_
#define  _SO_PACKED_COLOR_

#include <Inventor/fields/SoMFUInt32.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/caches/SoNodeDependencies.h>

class SoColorPacker;

/**
 Node that defines base colors using packed representation.

 @ingroup PropertyNodes

 @DESCRIPTION
   SoPackedColor is similar to SoBaseColor in that it sets the base color
   component of the current material. However, it also changes the transparency
   component. The color and transparency information is packed into unsigned 32-bit
   integers: 0xrrggbbaa, where aa represents the alpha (0x00 = fully transparent,
   0xff = opaque), and rr, gg, and bb represent the red, blue, and green components
   of the color, respectively.

   SoPackedColor uses less memory than SoBaseColor or SoMaterial to store multiple
   color and transparency values. It can be used in cases where space is critical.
   However, for geometry nodes that have a *vertexProperty* field, the best
   practice is to use an SoVertexProperty node (which also has an orderedRGBA field)
   to specify diffuse color and transparency.

   As described @ref SoLightModel-base-color-def "here", the #orderedRGBA field of
   this class can be overridden by other property nodes.

 @FILE_FORMAT_DEFAULT
    PackedColor {
    @TABLE_FILE_FORMAT
       @TR orderedRGBA   @TD 0xccccccff
    @TABLE_END
    }

 @ACTION_BEHAVIOR
    SoGLRenderAction,  SoCallbackAction  
    Sets the current base (diffuse) color(s) in the state.

 @SEE_ALSO
    SoBaseColor,
    SoVertexProperty,
    SoMaterial

 */
class INVENTOR_API SoPackedColor : public SoNode {

  SO_NODE_HEADER(SoPackedColor);

 public:
  // Fields
  /**
   * Defines the packed base colors.
   * 
   */
  SoMFUInt32          orderedRGBA;

  /**
   * Creates a packed color node with default settings.
   */
  SoPackedColor();

  /**
   * Set the state of the override field.
   * see SoNode::setOverride doc.
   */
  inline virtual void setOverride(const SbBool state)
  { override.setValue(state); }

  /**
   * Returns the state of the override field.
   */
  inline virtual SbBool isOverride() const
  { return override.getValue(); }

 SoEXTENDER public:
  virtual void        doAction(SoAction *action);
  virtual void        GLRender(SoGLRenderAction *action);
  virtual void        callback(SoCallbackAction *action);
  SbBool              isTransparent()
    {return transparent;}

 SoINTERNAL public:
  static void         initClass();
  static void         exitClass();

  SoSFBool override;

  // check for transparency when field changes.
  virtual void        notify(SoNotList *list);
  
 protected:
  virtual ~SoPackedColor();
    
 private:
  // store whether transparent or not
  SbBool transparent;

  void computeReplacementColors(SoState* state);
  void updateTransparencies(SoState* state);

  SoRef<SoNodeDependencies> m_replacementColorDependencies;
  std::vector<uint32_t> m_replacementColors;
  std::vector<float> m_cachedTransparencies;
    
};

#endif /* _SO_PACKED_COLOR_ */

