/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_PICK_STYLE_
#define  _SO_PICK_STYLE_

#include <Inventor/SbString.h>
#include <Inventor/elements/SoPickStyleElement.h>
#include <Inventor/elements/SoPickMethodElement.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/nodes/SoNode.h>

/**
 * Picking style node.
 * 
 * @ingroup PropertyNodes
 * 
 * @DESCRIPTION
 *   This node determines how subsequent geometry nodes in the scene graph are to be
 *   picked, as indicated by the #style and #method fields.
 *   
 *   Note that this is the only way to change the pick behavior of shapes; drawing
 *   style, complexity, and other rendering-related properties have no effect on
 *   picking.
 * 
 * @FILE_FORMAT_DEFAULT
 *    PickStyle {
 *    @TABLE_FILE_FORMAT
 *       @TR style   @TD SHAPE
 *       @TR method  @TD AUTO
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoRayPickAction,  SoCallbackAction @BR
 *        Sets the current pick style and method in the state. Sets: SoPickStyleElement
 *                                                                   SoPickMethodElement
 * 
 * 
 * @SEE_ALSO
 *    SoComplexity,
 *    SoDrawStyle,
 *    SoRayPickAction
 * 
 * 
 */
class INVENTOR_API SoPickStyle : public SoNode {

  SO_NODE_HEADER(SoPickStyle);

 public:
   /** Picking style */
  enum Style {
    /**
     *  Points on the surfaces of shapes may be picked 
     */
    SHAPE           = SoPickStyleElement::SHAPE,
    /**
     *  Points on the surfaces of 3D bounding boxes of shapes may be picked 
     */
    BOUNDING_BOX    = SoPickStyleElement::BOUNDING_BOX,
    /**
     *  Subsequent objects are ignored by pick traversal. 
     */
    UNPICKABLE      = SoPickStyleElement::UNPICKABLE
  };

  /**
   * Picking style.
   * @useenum{Style}. Default is SHAPE.
   */
  SoSFEnum            style;

  /** 
   * Picking method.
   * Specifying a method does not change the result of picking, but may affect performance.
   * Some shapes like SoPointSet and MoMeshSkin implement a fast GPU Picking technique
   * that may be faster than CPU picking (depending on the use case).
   * When Method is AUTO, Open Inventor will automatically determine which picking method
   * to use. However, you can force use of the CPU or GPU picking techniques.
   * Please refer to SoRayPickAction documentation for more details about GPU Picking. */
  enum Method {
    /**
    *  Select the best method to use when picking a shape.
    */
    AUTO = SoPickMethodElement::AUTO,
    /**
    *  Always use GPU accelerated picking if the shape supports it.
    *  The documentation of the SoPickAction describes the shapes that
    *  support GPU picking and some required conditions.
    */
    GPU = SoPickMethodElement::GPU,
    /**
    *  Always use CPU picking even if the shape supports another picking method.
    */
    CPU = SoPickMethodElement::CPU
  };

  /**
  * Specifies the method used to pick shapes.
  * @useenum{Method}. Default is AUTO.
  */
  SoSFEnum            method;

  /**
   * Creates a pick style node with default settings.
   */
  SoPickStyle();

  /**
   * Set the state of the override field.
   * see SoNode::setOverride doc.
   */
  inline virtual void setOverride(const SbBool state)
  { override.setValue(state); }

  /**
   * Returns the state of the override field.
   */
  inline virtual SbBool isOverride() const
  { return override.getValue(); }

 SoEXTENDER public:
  virtual void        doAction(SoAction *action);
  virtual void        callback(SoCallbackAction *action);
  virtual void        pick(SoPickAction *action);
  virtual void        GLRender( SoGLRenderAction *action );

 SoINTERNAL public:
  static void         initClass();
  static void         exitClass();

  SoSFBool override;

 protected:
  virtual ~SoPickStyle();
};

#endif /* _SO_PICK_STYLE_ */

