/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2021 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_POINT_SET_
#define  _SO_POINT_SET_

#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/nodes/SoNonIndexedShape.h>
#include <Inventor/SbPImpl.h>

class SoState;
class SoTextureCoordinateBundle;
class SoTextureCoordinate3Bundle;

SO_PIMPL_PUBLIC_DECLARATION(SoPointSet)

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoPointSet
//
//  Set of points, using the current coordinates. The startIndex field
//  gives the starting coordinate index for the first point. The number of
//  points to draw is determined by the numPoints field.
//
//////////////////////////////////////////////////////////////////////////////

// This value, when used in the numPoints field, means use the rest of
// the coordinates as points
#define SO_POINT_SET_USE_REST_OF_POINTS (-1)

/**
 * Point set shape node.
 * 
 * @ingroup ShapeNodes
 * 
 * @DESCRIPTION
 *   This node represents a set of points located at the coordinates specified by the
 *   #vertexProperty field (from SoVertexShape) or the current inherited
 *   coordinates. For optimal performance, the #vertexProperty field is
 *   recommended.
 *   
 *   SoPointSet uses the coordinates in order, starting with the coordinate at #startIndex.
 *   The number of coordinates to use (the number of points in the set) is 
 *   specified by the #numPoints field.  The default value of this field is -1, 
 *   which means to use all the coordinates, so it may not be necessary to 
 *   explicitly set this field.
 *   
 *   The coordinates of the point set are transformed by the current cumulative
 *   transformation.
 *
 *   Lighting is not applied to points (i.e., they are rendered with light model BASE_COLOR)
 *   unless the application explicitly sets normal vectors using SoVertexProperty or SoNormal.
 *   Note that this is specific to direct rendering of point sets.  If you render polygonal
 *   geometry, e.g. SoFaceSet, in POINTS mode (SoDrawStyle), normal vectors are computed by
 *   Open Inventor and lighting is enabled.
 *
 *   NOTE: If you want to apply lighting in a shader, you still need to explicitly set normal
 *   vectors, otherwise lighting will be disabled for the point set.
 *
 *   Point size is controlled by the SoDrawStyle field @I pointSize@i.
 *
 *   Automatic simplification (by skipping points) is controlled by the SoComplexity
 *   field @I value@i.  Only values < 0.5 have any effect.
 *   
 *   Treatment of the current material and normal binding is as follows: PER_PART,
 *   PER_FACE, and PER_VERTEX bindings bind one material or normal to each point. The
 *   default material binding is OVERALL. The default normal binding is PER_VERTEX.
 *
 *
 *   Shape Antialiasing type is SoShape::POINTS.
 *
 * @B Limitations:@b
 *   - By default, SoPointSet ignores the vertex ordering set on the state by SoShapeHints nodes
 *     in order to avoid a performance hit on some GeForce cards. However, this default behavior
 *     can be changed using the environment variable OIV_POINT_SET_USE_VERTEX_ORDERING
 *     (see SoPreferences).
 * 
 * @FILE_FORMAT_DEFAULT
 *    PointSet {
 *    @TABLE_FILE_FORMAT
 *       @TR vertexProperty   @TD NULL
 *       @TR startIndex       @TD 0
 *       @TR numPoints        @TD -1
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction @BR
 *        Draws points based on the current coordinates, normals, materials, drawing
 *        style, and so on.
 * 
 *    SoRayPickAction @BR
 *        Picks points based on the current coordinates and transformation. Details about
 *        the intersection are returned in an SoPointDetail.
 * 
 *    SoGetBoundingBoxAction @BR
 *        Computes the bounding box that encloses all points in the set with the current
 *        transformation applied to them. Sets the center to the average of the
 *        coordinates of all points.
 * 
 *    SoCallbackAction @BR
 *        If any point callbacks are registered with the action, they will be invoked for
 *        each point in the set.
 * 
 * 
 * @SEE_ALSO
 *    SoCoordinate3,
 *    SoDrawStyle,
 *    SoFullSceneAntialiasing,
 *    SoIndexedPointSet,
 *    SoPointDetail,
 *    SoVertexProperty
 * 
 * 
 */
class INVENTOR_API SoPointSet : public SoNonIndexedShape
{
  SO_NODE_HEADER(SoPointSet);
  SO_PIMPL_PUBLIC_HEADER(SoPointSet)

public:
  /**
   * Number of points.
   */
  SoSFInt32 numPoints;

  /**
   * Creates a point set node with default settings.
   */
  SoPointSet();

SoEXTENDER public:
  /** Implements SoGLRenderAction action */
  virtual void GLRender( SoGLRenderAction* action );

  /**
   * Implements SoGetBoundingBoxAction action
   * Redefined this to tell open caches that they contain points
   */
  virtual void getBoundingBox( SoGetBoundingBoxAction* action );

  /** Implements SoGetPrimitiveCountAction action */
  virtual void getPrimitiveCount( SoGetPrimitiveCountAction* action );

  // Computes bounding box
  virtual void computeBBox( SoAction* action, SbBox3f& box, SbVec3f& center );

SoINTERNAL public:
  
  /** register in database */
  static void initClass();
  /** unregister from database */
  static void exitClass();
  
  // return num vertices
  virtual int getNumVert( SoState* state = NULL );

  /** Blow vpCache when vertexProperty field changes */
  virtual void notify( SoNotList* list );

  //return number of primitives
  virtual int getNumPrim( SoState* state = NULL );

protected:
  // Generates points representing point set
  virtual void generatePrimitives( SoAction* action );

  // check if normal should be generated or not.
  virtual void shouldGenerateNormals( SoGLRenderAction* action, const SoShapeStyleElement *shapeStyle );

  // No tangent generation
  virtual void shouldGenerateTangents( SoGLRenderAction* /*action*/, const SoShapeStyleElement* /*shapeStyle*/ ) {}
 
  // check if glrender should be called
  virtual SbBool shouldGLRender( SoGLRenderAction* action, SbBool isPointsOrLines = FALSE );

  //Free memory
  virtual void exitRender( SoState* state,
                           int stateFlag,
                           SoTextureCoordinateBundle* tcb,
                           SoTextureCoordinate3Bundle* tc3b );

  // count the number of primitives in the shape
  virtual void countPrim( SoState* state = NULL );

  // really do the rendering part
  virtual void doRendering( SoGLRenderAction* action, const SoShapeStyleElement* shapeStyle );

  // says that rendering is PointOrLine
  virtual SbBool isRenderingPointsOrLines( SoGLRenderAction* action );

  // Overrides standard method to create an SoPointDetail instance
  virtual SoDetail* createPointDetail( SoRayPickAction* action,
                                       const SoPrimitiveVertex* v,
                                       SoPickedPoint* pp);

  /** Destructor */
  virtual ~SoPointSet();

};

inline SbBool 
SoPointSet::isRenderingPointsOrLines( SoGLRenderAction * )
{
  return TRUE;
}

#endif /* _SO_POINT_SET_ */

