/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2021 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : C. OGNIER (Dec 2004)
**=======================================================================*/


#ifndef  _SO_RENDER_TO_TEXTURE_PROPERTY_
#define  _SO_RENDER_TO_TEXTURE_PROPERTY_

#include <Inventor/nodes/SoNode.h>
#include <Inventor/fields/SoMFNode.h> 
#include <Inventor/fields/SoSFEnum.h> 
#include <Inventor/fields/SoSFColor.h> 
#include <Inventor/fields/SoMFPath.h> 
#include <Inventor/fields/SoSFVec2s.h>
#include <Inventor/fields/SoSFTrigger.h>
#include <Inventor/actions/SoGLRenderAction.h> 
#include <Inventor/sensors/SoFieldSensor.h>
#include <Inventor/components/SoGLGraphicConfigTemplate.h>
#include <Inventor/sys/SoGLType.h>
#include <Inventor/helpers/SbGlContextHelper.h>

#include <SoDeprecationRules.h>

#ifndef HIDDEN_FROM_DOC
/*! \cond PRIVATE */
namespace inventor
{
  namespace impl
  {
    class SoTextureCubeMapImpl;
    class SoTextureImpl;
    class SoTexture2Impl;
    class SoViewingCubeImpl;
  }
}
/*! \endcond */
#endif

class SoTexture;

SO_PIMPL_PUBLIC_DECLARATION(SoRenderToTextureProperty)


/**
 * 
 * @VSGEXT Class for creating a texture by rendering a scene graph
 * 
 * @ingroup General
 * 
 * @DESCRIPTION
 * This class is used to define an image for texture mapping which is 
 * created by rendering a scene graph.
 *
 * It can be used with the classes SoTexture2 and SoTextureCubeMap.
 * Create the SoRenderToTextureProperty node and assign it to the
 * @I renderToTextureProperty@i field of the texture mapping node.
 *
 * The SoRenderToTarget node can also be used to create texture images.
 *
 * One of the two fields, node or path, must be defined.
 *
 * The scene graph should normally contain a camera and at least one light.
 * Unlike SoRenderToTarget, the children of SoRenderToTextureProperty do
 * not inherit the camera, lights, etc from the main scene graph.
 *
 * Note that the default image size is 0,0 (effectively disables rendering).
 * 
 * @FILE_FORMAT_DEFAULT
 *    RenderToTextureProperty {
 *    @TABLE_FILE_FORMAT
 *       @TR node                @TD []
 *       @TR path                @TD ""
 *       @TR updatePolicy        @TD WHEN_NEEDED
 *       @TR backgroundColor     @TD 0 0 0
 *       @TR size                @TD 0 0
 *       @TR component           @TD RGB
 *    @TABLE_END
 *    }
 * 
 * @SEE_ALSO
 *  SoTexture2,
 *  SoTextureCubeMap
 *
 */
class INVENTOR_API SoRenderToTextureProperty : public SoNode
{
  SO_NODE_HEADER(SoRenderToTextureProperty);
  SO_PIMPL_FINAL_PUBLIC_HEADER(SoRenderToTextureProperty)
  
 public:
  
 SoRenderToTextureProperty();
 
 /** Update Policy enumeration values. */
 enum UpdatePolicy {
    /**
     * The texture is updated at each traversal of the shape it is mapped to.
     */
    EACH_FRAME,
    /**
     * Default.
     * The texture is updated only if the path or node fields have changed.
     */
    WHEN_NEEDED,
    /**
     * The texture is updated only if the trigger field is touched.
     */
    ON_TRIGGER
  };

 /** Rendering component enumeration values. */
  enum Component {
    /**
     * Default.
     * The RGB components are rendered to the texture.
     */
    RGB,
    /**
     * The RGB and alpha components are rendered to the texture.
     */
    RGB_TRANSPARENCY,
    RGB_ALPHA = RGB_TRANSPARENCY,
    /**
     * The depth component is rendered to the texture.
     */
    DEPTH
  };


  // Fields
  /** 
   * Specifies the scene graph(s) (SoNode(s)) to be rendered into the texture.
   * For use with SoTexture2, only one scene graph should be specified.
   * For use with SoTextureCube map, six scene graphs must be specified,
   * one for each face of the cube in the order:
   *
   * - imagePosX (left)
   * - imageNegX (right)
   * - imagePosY (bottom)
   * - imageNegY (top)
   * - imagePosZ (back)
   * - imageNegZ (front)
   *
   * The node field has priority over the path field. Null by default.
   */
  SoMFNode node;

  /** 
   * Specifies the scene graph(s) (SoPath(s)) to be rendered into the texture. 
   * For use with SoTexture2, only one path should be specified.
   * For use with SoTextureCube map, six paths must be specified,
   * one for each face of the cube in the order:
   *
   * - imagePosX (left)
   * - imageNegX (right)
   * - imagePosY (bottom)
   * - imageNegY (top)
   * - imagePosZ (back)
   * - imageNegZ (front)
   *
   * The node field has priority over the path field. Null by default.
   */
  SoMFPath path;

  /**
   * Specifies the policy for rendering to the texture. 
   * Use enum #UpdatePolicy. Default is WHEN_NEEDED.
   */
  SoSFEnum updatePolicy;

  /**
   * Background color used when rendering the scene graph. Black by default.
   */
  SoSFColor backgroundColor;

  /**
   * Size in pixels of the rendered texture. 
   * (0, 0) by default.
   */
  SoSFVec2s size;

  /**
   * Specifies the type of component(s) for the rendering. 
   * Use enum #Component. Default is RGB.
   */
  SoSFEnum component;

  /**
   * Trigger field to render the scene into the Pbuffer.
   * Effective only if updatePolicy is ON_TRIGGER.
   */
  SoSFTrigger trigger;

  /**
   * Specifies a custom render action to use for rendering the scene.
   */
  void setGLRenderAction(SoGLRenderAction* ra);

  /**
   * Returns the render action used for rendering the scene.
   */
  SoGLRenderAction* getGLRenderAction() const;

#if SoDEPRECATED_BEGIN(10000)

  SoDEPRECATED_METHOD(10000, "No more used. Related to deprecated PBuffer implementation.")
  void setGraphicConfigTemplate(SoGLGraphicConfigTemplate*) {}

  SoDEPRECATED_METHOD(10000, "No more used. Related to deprecated PBuffer implementation. Always return NULL")
  SoGLGraphicConfigTemplate* getGraphicConfigTemplate() { return NULL; }

  SoDEPRECATED_METHOD(10000, "No more used. Related to deprecated PBuffer implementation.")
  void setContextForSharing(SoGLContext*) {}

#endif /** @DEPRECATED_END */



SoEXTENDER public:
  virtual void GLRender(SoGLRenderAction *action);

SoINTERNAL public:
  static void initClass();
  static void exitClass();

  /** catch some fields changes */
  virtual void fieldHasChanged( SoField* field );
  virtual void notify( SoNotList *list );

  /**
   * Sets the rendering target texture.
   *
   * Automatically set when used with the renderToTextureProperty field of
   * SoTexture2 or SoTextureCubeMap.
   */
  void setTexture(SoTexture* texture);

protected:
  ~SoRenderToTextureProperty();

#ifndef HIDDEN_FROM_DOC
  // friendship to access SoRenderToTextureProperty::getImpl<>
  friend class SoTextureCubeMap;
  friend class inventor::impl::SoTextureCubeMapImpl;
  friend class SoTexture2;
  friend class inventor::impl::SoTextureImpl;
  friend class inventor::impl::SoTexture2Impl;
  friend class inventor::impl::SoViewingCubeImpl;
#endif // HIDDEN_FROM_DOC

};

#endif /* _SO_RENDER_TO_TEXTURE_PROPERTY_ */
