/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Nick Thompson (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_SEPARATOR_
#define  _SO_SEPARATOR_

#include <Inventor/nodes/SoGroup.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/fields/SoMFInt32.h>
#include <Inventor/SbPImpl.h>

class SoBoundingBoxCache;
class SoPrimitiveCountCache;

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoSeparator
//
//  Separator group node: state is saved before traversing children
//  and restored afterwards.
//
//////////////////////////////////////////////////////////////////////////////

#include <Inventor/fields/SoSFBitMask.h>
#include <Inventor/SbRenderUnitID.h>

SO_PIMPL_PUBLIC_DECLARATION(SoSeparator)

/**
 * Group node that saves and restores traversal state.
 * 
 * @ingroup GroupNodes
 * 
 * @DESCRIPTION
 *   This group node performs a push (save) of the traversal state before traversing
 *   its children and a pop (restore) after traversing them. This isolates the
 *   separator's children from the rest of the scene graph. A separator can include
 *   lights, cameras, coordinates, normals, bindings, and all other properties.
 *   Separators are relatively inexpensive (the push/pop operation is highly
 *   optimized), so they can be used freely within scenes.
 *
 *   Caching:
 *   
 *   The SoSeparator node provides caching of state during
 *   bounding box computation. This feature can be explicitly enabled or disabled
 *   by setting the #boundingBoxCaching field. By default,
 *   this field is set to AUTO, which means that Open Inventor decides whether
 *   to build a cache based on internal heuristics.
 *
 *   Culling:
 *
 *   Separators can also perform culling during rendering and picking. Culling skips
 *   over traversal of the separator's children if they are not going to be rendered
 *   or picked, based on the comparison of the separator's bounding box with the
 *   current view volume. Culling is controlled by the #renderCulling and
 *   #pickCulling fields. These are also set to AUTO by default; however,
 *   render culling can be expensive (and can interfere with render caching), so the
 *   AUTO heuristics leave it disabled unless specified otherwise.
 *   Note that if renderCulling is ON, the SoSeparator will prohibit auto-caching, so
 *   that no SoSeparator (or other render caching node) above this node in the scene graph
 *   will automatically create a render cache.
 *
 *  Fast edit:
 *
 *  The fast editing feature allows you to modify parts of a scene without 
 *  redrawing the entire scene. For example, you could use it to interactively 
 *  move a small object in a large scene that takes a long time to redraw. 
 *  Fast editing must be enabled using SoGLRenderAction::setFastEditSavePolicy.
 * 
 *  The #fastEditing field enables fast editing for the sub-graph under the
 *  SoSeparator. 
 *  Possible values are DISABLE, KEEP_ZBUFFER, and CLEAR_ZBUFFER. 
 *  Using KEEP_ZBUFFER means that the fast edit geometry is depth buffered 
 *  against the other objects in the scene, and using CLEAR_ZBUFFER means 
 *  that the fast edit geometry will be drawn on top of the other objects of 
 *  the scene. If several SoSeparators have the CLEAR_ZBUFFER flag set, they 
 *  are drawn in the order in which they appear in the scene. The last 
 *  separator in the scene will be topmost on the screen.
 *
 *  Fast edit performance:
 *  - This feature is most efficient when the parts of a scene graph to be 
 *    fast edited contain a relatively small number of triangles. 
 *  - The topmost SoSeparator of your scene graph should not enable the fast editing feature.
 *    Otherwise, performance will be very poor. 
 *  - Currently, the entire viewport is saved and restored during fast editing. 
 *    Therefore, fast editing performance is linked to the size of the viewport: the larger 
 *    the viewport, the longer it will take to save and restore it.
 *
 * Fast edit limitations:
 *  - Hardware: Fast editing performance depends on your graphics hardware and driver. 
 *    To use the fast editing feature, the ARB_bufferRegion OpenGL extension 
 *    is required.
 *  - Clip planes: When a fast edit object is moved outside the clip planes limit, fast 
 *    editing is temporarily disabled and the entire scene is redrawn.
 *  - All fast editing sub-graphs are rendered even if only one fast editing sub-graph has changed.
 *  - Transparency: If the scene contains partially transparent objects, there is a
 *    possibility of visual artifacts regardless of what transparency algorithm is used.
 *    The fast edit shapes are necessarily rendered on top of the rest of the scene, which
 *    may not be the order required for correct transparency.
 *  - Stereo rendering: Fast edit is not compatible with stereo rendering. Fast edit is disabled
 *    when switching to stereo rendering.
 *
 * @FILE_FORMAT_DEFAULT
 *    Separator {
 *    @TABLE_FILE_FORMAT
 *       @TR boundingBoxCaching   @TD AUTO
 *       @TR renderCulling        @TD AUTO
 *       @TR pickCulling          @TD AUTO
 *       @TR fastEditing          @TD DISABLE
 *       @TR renderUnitID         @TD -1
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction,  SoCallbackAction,  SoGetBoundingBoxAction,  SoGetMatrixAction,  SoHandleEventAction,  SoRayPickAction,  SoSearchAction @BR
 *        Saves the current traversal state, traverses all children, and restores the
 *        previous traversal state.
 *        Sets: SoCacheElement
 * 
 * @SEE_ALSO
 *    SoSelection,
 *    SoTransformSeparator,
 * 
 */
class INVENTOR_API SoSeparator : public SoGroup
{
  SO_NODE_HEADER(SoSeparator);
  SO_PIMPL_PUBLIC_HEADER(SoSeparator)

 public:

  /** Possible values for caching. */
  enum CacheEnabled {
    /**
     *  Never build a cache 
     */
    OFF,
    /**
     *  Always try to build a cache 
     */
    ON,
    /**
     *  Decide whether to cache based on some heuristic 
     */
    AUTO
  };

#if SoDEPRECATED_BEGIN(10300)
  /**
   * Whether to cache during rendering traversal.
   * @useenum{CacheEnabled}. Default is AUTO.
   */
  SoDEPRECATED_FIELD( 10300, "This field is ignored. Render caching is set internally to ensure best performance." )
  SoSFEnum renderCaching;
#endif /** @DEPRECATED_END */

  /**
   * Whether to cache during bounding box traversal.
   * @useenum{CacheEnabled}. Default is AUTO.
   */
  SoSFEnum boundingBoxCaching;

  /**
   * Whether to cull during rendering traversal.
   * @useenum{CacheEnabled}. Default is AUTO.
   */
  SoSFEnum renderCulling;

  /**
   * Whether to cull during picking traversal.
   * @useenum{CacheEnabled}. Default is AUTO.
   */
  SoSFEnum pickCulling;

  /** Fast editing policy enumeration values. */
  enum FastEditing {
    /**
     * This SoSeparator node doesn't have Fast Edit behavior.
     */
    DISABLE = 0x01,

    /**
     * This SoSeparator node has a Fast Edit behavior and the children nodes are drawn considering the depth Buffer.
     */
    KEEP_ZBUFFER = 0x02,

    /**
     * This SoSeparator node has a Fast Edit behavior and the children nodes are drawn after the depth Buffer has been cleared.
     */
    CLEAR_ZBUFFER = 0x03
  };

  /**
   * Specifies the fast edit mode of the separator node.
   * Use enum #FastEditing. Default is DISABLE.
   *
   * @FIELD_SINCE_OIV 5.0
   */
  SoSFEnum fastEditing;

  /** 
   * Used during the ScaleViz depth compositing process, this field specifies which 
   * render unit (OIRU) will render the sub scene graph below this separator.
   *
   * - 0, the sub scene graph is rendered on the local machine. 
   * 
   * - > 0, the sub scene graph is rendered by the OIRU with the
   *   specified ID. If the specified renderUnitId is larger than the number
   *   of available render units, 
   *   the sub scene graph is distributed to the specified render unit ID modulo the 
   *   number of render units actually available. That is:@BR @BR
   *   RenderUnitUsed = ((renderUnitId-1)%(ActualNumberOfRenderUnits))+1
   *
   * - -1 (SO_RENDERUNITID_NONE), this separator is not assigned to any OIRU.@BR @BR
   *   NOTE: This is the default value, which means that by default the sub scene graph 
   *   is *not* distributed to any node. In order to use depth compositing, 
   *   you must modify renderUnitId from the default, otherwise nothing will be rendered.
   *
   * - -2 (SO_RENDERUNITID_INHERIT), this separator inherits its value
   *   from parent SoSeparator.
   *
   * - -3 (SO_RENDERUNITID_ALL), the sub scene graph is rendered by all OIRUs.
   *
   * @FIELD_SINCE_OIV 6.1
   */
  SoSFInt32 renderUnitId;

  /**
   * Creates a separator node with default settings.
   */
  SoSeparator();

  /**
   * Constructor that takes approximate number of children.
   */
  SoSeparator(int nChildren);

 SoEXTENDER public:
  // Overrides default method on SoNode
  virtual SbBool      affectsState() const;

  // Implement actions
  virtual void        doAction(SoAction *action);
  virtual void        callback(SoCallbackAction *action);
  virtual void        getBoundingBox(SoGetBoundingBoxAction *action);
  virtual void        getMatrix(SoGetMatrixAction *action);
  virtual void        GLRender(SoGLRenderAction *action);
  virtual void        handleEvent(SoHandleEventAction *action);
  virtual void        rayPick(SoRayPickAction *action);
  virtual void        search(SoSearchAction *action);
  virtual void        getPrimitiveCount(SoGetPrimitiveCountAction *action);
  virtual void        clearCacheList();

  // These methods make render traversal faster by implementing
  // different rendering paths corresponding to different action
  // path codes.
  virtual void        GLRenderBelowPath(SoGLRenderAction *action);
  virtual void        GLRenderInPath(SoGLRenderAction *action);
  virtual void        GLRenderOffPath(SoGLRenderAction *action);

 SoINTERNAL public:
  static void         initClass();
  static void         exitClass();

  SB_THREAD_TLS_HEADER();

  virtual void        notify(SoNotList *list);

  /**
   * return current BBox cache.
   * Particulary useful to reuse already construct bbox cache.
   * (ie. see SoExtSelection optimization)
   */
  SoBoundingBoxCache* getBoundingBoxCache() const
  { return bboxCache; }

#ifdef ADD_MFIELD_DEPLIST
  SoMFInt32 renderUnitDependencies;
#endif

  /** Mark all caches and all their parent caches invalid. */
  void invalidateAllParentCaches();

  /** Mark all caches and all their children caches invalid. */
  void invalidateAllChildrenCaches();

  void setRenderCachePolicy(uint32_t policy);

  uint32_t getRenderCachePolicy() const;

protected:
  // Returns TRUE if this separator can be culled because it is
  // outside the cull volume.  The cullResults flags come from the
  // GLRenderAction->getCullTestResults() method, and track the
  // results of previous cull tests to save work when we know we're
  // completely inside the view volume.
  virtual SbBool cullTest(SoGLRenderAction *action, int &cullResults);

  virtual SbBool readInstance(SoInput *in, unsigned short flags);

  virtual ~SoSeparator();

private:

  SoBoundingBoxCache* bboxCache; // Cache for bounding boxes

  SoPrimitiveCountCache *primCountCache;      // Cache for primitive count
        
  SbBool bboxCacheIsOpen;

  // bba is in thread local storage now. 
  struct MTstruct {
    SoGetBoundingBoxAction *bba;
  };

  void commonConstructor();

  float m_depthOffsetCachedValue;

#if SoDEPRECATED_BEGIN(10000)
  /**
  * Whether to cache during DirectViz rendering traversal.
  * @useenum{CacheEnabled}. Default is AUTO.
  * @BR
  * @BR
  * @FIELD_SINCE_OIV 6.1
  */
  SoDEPRECATED_FIELD(10000,"No longer used.")
  SoSFEnum directVizCaching;
#endif /** @DEPRECATED_END */
};

#ifdef _MSC_VER
#pragma warning( pop )
#endif

#endif /* _SO_SEPARATOR_ */


