/*==============================================================================
**    THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),     **
**              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                 **
**                                                                            **
**  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS   **
**  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR   **
**  WRITTEN AUTHORIZATION OF FEI S.A.S.                                       **
**                                                                            **
**                        RESTRICTED RIGHTS LEGEND                            **
**  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS  **
**  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN  **
**  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT  **
**  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN  **
**  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         **
**                                                                            **
**                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                    **
**                              BORDEAUX, FRANCE                              **
**                            ALL RIGHTS RESERVED                             **
==============================================================================*/

#pragma once

#include <Inventor/nodes/SoNode.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFNode.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/SbEnums.h>
#include <Inventor/nodes/SoShaderParameter.h>

class SoTexture;

/**
 * @VSGEXT Shader parameter node storing a texture image.
 *
 * @ingroup ShaderNodes
 *
 * @DESCRIPTION
 * This node allows the application to bind a texture to an image unit for the
 * purpose of reading and writing it from shaders. #texture specifies an
 * existing texture node to bind to the image unit.
 *
 * If #texture is an SoTextureCubeMap or an SoTexture3 with SoTexture3::ARRAY
 * layout, then it is possible to bind either the entire array, or only a single
 * layer of the array to the image unit. In such cases, if #layer is -1, the
 * entire array is attached to the image unit. Otherwise, #layer specifies the
 * layer of the array to attach to the image unit.
 *
 * #accessMode specifies the access types to be performed by shaders and may be
 * set to #READ, #WRITE, or #READ_WRITE to indicate read-only, write-only or
 * read-write access, respectively. Violation of the access type specified in
 * #accessMode (for example, if a shader writes to an image bound with
 * #accessMode set to #READ) will lead to undefined results, possibly including
 * program termination.
 *
 * SoShaderParameterImage nodes are usually added to the field SoShaderProgram#images
 * to affect only that specific shader program, but can also be inserted in the scene
 * graph and affect all subsequent shader programs.
 *
 * See <a href="https://www.khronos.org/opengl/wiki/Image_Load_Store">Image Load Store</a>
 * section of the OpenGL wiki for more information about using this feature in GLSL. @BR
 *
 * As an example, here is how you could define a 2D write-only texture in GLSL:
 * @code
 *  restrict writeonly uniform image2D myTexture;
 * @endcode
 * Use "restrict" to allow the compiler to make optimizations when you know that
 * this variable is the only one that will access the underlying texture. @BR
 * Also note that the name of the variable ("myTexture" in the example) should
 * correspond to the value of the field #name.
 *
 * @FILE_FORMAT_DEFAULT
 *    ShaderParameterImage {
 *      @TABLE_FILE_FORMAT
 *        @TR name           @TD ""
 *        @TR identifier     @TD 0
 *        @TR texture        @TD NULL
 *        @TR layer          @TD -1
 *        @TR accessMode     @TD READ
 *      @TABLE_END
 *    }
 *
 *
 * @SEE_ALSO
 *    SoShaderParameter,
 *    SoShaderParameterBufferObject,
 *    SoUniformShaderParameter,
 *    SoShaderObject,
 *    SoShaderProgram
 *
 * @NODE_SINCE_OIV 10.12
 */
class INVENTOR_API SoShaderParameterImage : public SoShaderParameter
{
  SO_NODE_HEADER(SoShaderParameterImage);
  RENDERER_RESOURCE(SoShaderParameterImage);

public:
  /**
   * Specifies the texture to bind to the image unit.
   * Default is NULL.
   */
  SoSFNode texture;

  /**
   * Specifies the layer of texture to be bound to the image unit.
   * Default is -1 (bind all layers).
   */
  SoSFInt32 layer;

  /**
   * Image access modes.
   */
  enum AccessMode
  {
    /**
     * The shader can only read from the image.
     */
    READ = SbEnums::READ,

    /**
     * The shader can only write to the image.
     */
    WRITE = SbEnums::WRITE,

    /**
     * The shader can read from and can write to the image.
     */
    READ_WRITE = SbEnums::READ_WRITE,
  };

  /**
   * Specifies the type of access that will be performed on the image.
   * @useenum{AccessMode}.
   * Default is #READ.
   */
  SoSFEnum accessMode;

  SoShaderParameterImage();

SoEXTENDER public:
  void GLRender(SoGLRenderAction* action);

SoINTERNAL public:
  static void initClass();
  static void exitClass();

  void setAssignedUnit(int unit)
  {
    m_assignedUnit = unit;
  }

  int getAssignedUnit() const
  {
    return m_assignedUnit;
  }

  static const SbString& getGLSLFormatLayout(SoTexture* texture);

  void fieldHasChanged(SoField* field) override;

  /** Update internal parameters that depend on state */
  void updateFromState(SoState* state);

  /**
   * Returns the SoTexture node that will be used as the source of read & write
   * operations by the image parameter.
   */
  SoTexture* getTexture() const;

protected:
  virtual ~SoShaderParameterImage();

private:
  int m_assignedUnit;
  SoTexture* m_texture;
};

