/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : R.ALBOU (Sep 2001)
**=======================================================================*/
#ifndef  _SO_SHADOW_GROUP_
#define  _SO_SHADOW_GROUP_


#include <Inventor/nodes/SoGroup.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/SbMatrix.h>
#include <Inventor/SbPImpl.h>

class SoGLShadowMapping ;
class SoCache ;
class SoGLTexture;

SO_PIMPL_PUBLIC_DECLARATION(SoShadowGroup)

/**
* @VSGEXT Shadow casting group node
*
* @ingroup GroupNodes
*
* @DESCRIPTION
* This group node performs real-time shadow casting. Because shadows depends on a lot of OpenInventor elements,
* it is recommended to put children of this group under a SoSeparator in order to have a cache.
* The shadow casting computation takes into account all shapes contained in this group.
* It ignores shapes in other groups.
* Depending on the parameters of the lights in the scene, a shape within this group may cast and receive shadows:
*   - on/from itself (self-shadowing)
*   - on/from other shapes (also within this group)
* The current shadow style determines if shapes can cast shadows and/or have shadows cast onto them (see SoShadowStyle).
*
* This class defines a single shadowing method.
* The #method field allows for future shadowing methods to be added.
*
* - VARIANCE_SHADOW_MAP: This mode displays soft shadows. The #smoothFactor
*   field increases or decreases the softness.
*   It does not display aliasing, but on large scenes shadows will progressively fade away.
*   Increasing #quality reduces this problem.
*
* Lights @BR
* - Shadows are cast by SoLight nodes placed before the SoShadowGroup node in the scene graph.
* - SoLight nodes included in the SoShadowGroup node do not cast any shadow.
* - If there are no SoLight nodes before the SoShadowGroup node, no shadow is cast.
* - The shape of the shadows depend on the parameters of the SoLight nodes and on the type of light.
*   Only SoDirectionalLight, SoPointLight and SoQuadAreaLight light types cast shadows.
* - SoShadowGroup supports shadowing with up to 8 simultaneous lights.
* - SoLight#shadowIntensity controls the intensity of the shadows cast by that light node.
*   This intensity is multiplied by the value of SoShadowGroup#intensity to give the final intensity of that light's shadow.
*
* Shaders @BR
* Only the VARIANCE_SHADOW_MAP method is available with shaders (custom or internal). In order to use shadows
* with user defined GLSL shaders (SoShaderProgram or SoVolumeShader), the following functions must
* be used in the shader programs. @BR
* In the vertex shader:
* - void OivSetupShadowVertex(): Must be called in the vertex shader.
*
* In the fragment shader:
* - float OivComputeShadow(): Returns a scalar in [0-1].
*   0 means fragment is fully shadowed, 1 means no shadow.  Should be used to modify the
*   fragment color.
* - void OivGenerateShadowMap(): Must be called during the shadowmap pass in order to create the
*   shadowmap.  During this pass, the shader must not write to gl_FragColor.  Use the uniform
*   @I OivShadowPass @i to know when the shadowmap is being computed.
*
* The following uniform is available in vertex and fragment shaders:
* - uniform bool OivShadowPass: Is true if the shadowmap is currently being rendered
*
* The following code can be used as a skeleton for a GLSL shader that works with shadows:
* \code
* // The vertex shader:
* void OivSetupShadowVertex();
* void main()
* {
*   ..userCode..
*   //Needed for shadowing
*   OivSetupShadowVertex();
* }
*
* // The fragment shader:
* //If true we are in shadowmap generation pass
* uniform bool OivShadowPass;
*
* void OivGenerateShadowMap();
* float OivComputeShadow();
* void main()
* {
*   if ( !OivShadowPass ) {
*     ..compute fragment color here..
*     // Define the final color
*     gl_FragColor.xyz = fragColor.xyz * OivComputeShadow();
*     gl_FragColor.w = fragColor.w;
*   }
*   else {
*     // Output the shadowmap during the shadow map pass
*     OivGenerateShadowMap();
*   }
* } \endcode
*
* Transparency: @BR
* Transparent objects are treated as follows, depending on the transparency type:
*
*  - NO_SORT: Transparent objects cast a shadow and are shadowed but the shadow intensity doesn't
*    depend on the transparency value (the same shadow is displayed for full transparent shapes and
*    opaque shapes).
*
*  - OPAQUE_FIRST, SORTED_OBJECT and SORTED_PIXEL: Same as previous case for VARIANCE_SHADOW_MAP.
*
* Texture units: @BR
* In VARIANCE_SHADOW_MAP mode, this node reserves texture unit OIV_SHADOW_TEXTURE_UNIT0
* for its rendering (see SoPreferences to set these). If this value is not set
* the application can specify which texture
* units OpenInventor should automatically use by setting environment variables (see SoPreferences).
* The texture units between OIV_FIRST_RESERVED_TEXTURE_UNIT and
* OIV_FIRST_RESERVED_TEXTURE_UNIT + SoShaderProgram::getNumReservedTextures() -1
* inclusive are reserved for internal OpenInventor use.
* If OIV_FIRST_RESERVED_TEXTURE_UNIT is not set, its default value is
* SoFragmentShader::getMaxTextureImageUnit()-SoShaderProgram::getNumReservedTextures().
* Note: The value returned by SoShaderProgram::getNumReservedTextures() may change between
* OpenInventor versions.  The total number of available texture units depends on the graphics
* hardware.
*
* Hardware requirements: @BR
* The @B Shadow  @b and @B Depth Texture @b OpenGL extensions (standard in OpenGL 1.4) are
* used if they are available and generally improve performance.<BR>
* In order to increase quality and performance of the shadows, OpenGL PBuffers are used.
* PBuffers are a limited resource on some systems.  Set @B OIV_PBUFFER_ENABLE @b to 0 to
* disallow use of PBuffers.
*
* A DirectX 10 graphics board is needed to support VARIANCE_SHADOW_MAP.
*
* Use the isSupported() static method to determine if the current graphics board
* supports shadowing.
*
* @B Limitations @b
*  - SoShadowGroup cannot be nested by design.
*  - The shadows cast by an SoQuadAreaLight do not take the size of its area into account.
*    An SoQuadAreaLight casts the same shadows as an SoPointLight positioned at the center of the quadrangle.
*  - SoShadowGroup does not support SoSpotLight nodes and they do not cast any shadow.
*
* Notes:
* - Geometry cannot receive shadows if lighting is disabled. @BR
*   For example if an SoLightModel node sets the lighting model to BASE_COLOR.
* - Geometry cannot receive shadows if a texture is applied using the
*   DECAL or REPLACE models.
*
* @FILE_FORMAT_DEFAULT
*    ShadowGroup {
*    @TABLE_FILE_FORMAT
*               @TR isActive             @TD TRUE
*               @TR isShapesBefore       @TD FALSE
*               @TR intensity            @TD 0.5
*               @TR precision            @TD 0.5
*               @TR quality              @TD 0.5
*               @TR shadowCachingEnabled @TD TRUE
*               @TR visibilityRadius     @TD 1.0
*               @TR visibilityFlag       @TD LONGEST_BBOX_EDGE_FACTOR
*               @TR smoothFactor         @TD 1
*               @TR minVariance          @TD 1e-5
*               @TR method               @TD VARIANCE_SHADOW_MAP
*               @TR lightBleedingReduction @TD 0.01
*    @TABLE_END
*    }
*
* @SEE_ALSO
*    SoShadowStyle,
*    SoLight
*
*
*/
class INVENTOR_API SoShadowGroup : public SoGroup
{
  SO_NODE_HEADER(SoShadowGroup);
  SO_PIMPL_PUBLIC_HEADER(SoShadowGroup)

public:
  /**
  * #visibilityRadius interpretation
  */
  enum VisibilityFlag {
    /**
    *
    * #visibilityRadius is interpreted as an
    * absolute value.
    */
    ABSOLUTE_RADIUS,

    /**
    * The longest edge of the scene
    * bounding box will be multiplied by the value of #visibilityRadius. The
    * resulting value
    * will be the radius within which shadows are computed.
    */
    LONGEST_BBOX_EDGE_FACTOR
  } ;

  /**
   * Shadowing techniques
   */
  enum ShadowingMethod {
    /**
     * Shadows with soft edges
     */
    VARIANCE_SHADOW_MAP,
    #ifndef HIDDEN_FROM_DOC
    LAST_METHOD
    #endif
  };

  /**
  * Default constructor.
  */
  SoShadowGroup();

  /**
  * Constructor that takes approximate number of children.
  */
  SoShadowGroup(int nChildren);

  /**
  * Activates/deactivates the shadowing.
  * If deactivated, this #SoShadowGroup behaves like an #SoGroup.
  */
  SoSFBool        isActive ;

  /**
  * Specifies the intensity of the shadow.
  * 0 = lowest intensity (the shadow is invisible),
  * 1 = highest intensity (the shadow is solid black).
  */
  SoSFFloat   intensity ;

  /**
  * Specifies the precision of the shadow.
  * 0 = lowest precision, 1 = highest precision.
  * With lower precisions, some parts of the scene may be unlit where
  * they should be lit, or vice versa.
  * Increasing the precision will reduce performance.
  *
  * NOTE: If your graphics driver does not support
  * the required OpenGL extensions, you may see low precision results
  * even when this field is set to the highest precision.
  */
  SoSFFloat   precision ;

  /**
  * Specifies the quality of the shadow.
  *
  * The texture used for shadowmap rendering will be of size 2*quality*ViewportSize.
    * Increasing the quality will reduce performance and increase memory consumption.
  */
  SoSFFloat   quality ;

  /**
  * Indicates if a cache should be used for computing the shadows.
  * Using a cache will improve rendering performance if the lights of your
  * scene and your scene under this group node do not change.
  */
  SoSFBool    shadowCachingEnabled ;

  /**
  * Shadows are only computed within "visibility radius" distance
  * from the camera position.
  *
  * This radius equals:
  *   - the longest edge of the bounding box of the scene multiplied by #visibilityRadius
  *     if #visibilityFlag equals #LONGEST_BBOX_EDGE_FACTOR.
  *
  *   - #visibilityRadius if #visibilityFlag equals #ABSOLUTE_RADIUS.
  *
  * When #visibilityRadius = 1 and #visibilityFlag = #LONGEST_BBOX_EDGE_FACTOR
  * (default values), shadowing is computed throughout the entire scene.
  *
  * Note: With VARIANCE_SHADOW_MAP method, try to increase the #quality field first.
  */
  SoSFFloat   visibilityRadius ;

  /**
  * Specifies how #visibilityRadius is interpreted.
  * Use enum #VisibilityFlag. Default is LONGEST_BBOX_EDGE_FACTOR.
  */
  SoSFEnum    visibilityFlag ;


  /**
   * Set the smoothness of shadows. Higher values give smoother shadows.
   * Default is 1.
   *
   * @FIELD_SINCE_OIV 8.1
   */
  SoSFInt32 smoothFactor;

  /**
   * Increasing this value will decrease
   * possible self-shadowing artifacts but will make shadows fade away. The default
   * value is safe for most scenes. Increasing the #quality field is also preferable before
   * tweaking this value.
   * A special case is for SoVolumeRender node which has a minimum of 0.001 or IVVR_SHADOW_MIN_VARIANCE if set.
   * Default is 1e-5 and range is [0-1].
   *
   * @FIELD_SINCE_OIV 8.1
   */
  SoSFFloat minVariance;

  /**
   * In some cases, a halo may appear around shadows
   * intersecting each other. Increasing this value will decrease this effect.
   * Default value is safe for most scenes.
   * Default is 0.01 and range is [0-1].
   *
   * @FIELD_SINCE_OIV 8.1
   */
  SoSFFloat lightBleedingReduction;

  /**
   * Specifies the shadowing technique to use.
   * @useenum{ShadowingMethod}. Default is VARIANCE_SHADOW_MAP.
   *
   * @FIELD_SINCE_OIV 8.1
   */
  SoSFEnum method;

  /**
  * Indicates if shadow casting is supported by your graphic board.
   * When using a debug build of Open Inventor, some "no context available"
   * warning messages may be generated. You can ignore them or see
   * SoGLExtension for an example of using SoGLContext to avoid them.
  */
  static SbBool isSupported(SoShadowGroup::ShadowingMethod method = SoShadowGroup::VARIANCE_SHADOW_MAP) ;


SoEXTENDER public:
  /** Called during GLRender action*/
  virtual void GLRender(SoGLRenderAction *action);

  /*----------------------------------------------------------------------------*/

SoINTERNAL public:
  typedef std::vector<int> StackIndexVector;

  static void initClass() ;
  static void exitClass() ;

  const std::vector<SbMatrix>& getLightViewMatrices() const;

  const std::vector<SbMatrix>& getLightViewProjectionMatrices() const;

  const std::vector<SbVec2f>& getLightsNearFar() const;

  const std::vector<SbVec2i32>& getLightMatricesOffsetsAndNums() const;

  const std::vector<float>& getLightShadowIntensities() const;

  int getLightMatrixIndex() const;

  /** Return texture map unit used for shadowmap */
  static int getShadowMapTextureUnit(SoGLRenderAction* action);

  /** Destroy light caches */
  void invalidateLightCaches();

  /** Return false if shadowmap needs to be recomputed
   *  and update cache as needed. Take passed elements index in account */
  bool isCacheValid(SoState* state, SoCache* &cache, const StackIndexVector& elemsToCheck);


  /** Return true if shadow is active according to SoInteractiveComplexity */
  bool isShadowActive(SoState* state) const;

  /** Return quality according to SoInteractiveComplexity */
  float getQuality(SoState* state) const;

  /**
   * Handle field change
   */
  virtual void notify(SoNotList *list);

  /** Return the shadowmap texture size */
  SbVec2i32 getShadowMapSize() const;

protected:
  virtual ~SoShadowGroup();

  /**
   * @copydoc SoGroup::destroy
   */
  void destroy() override;

private:

#if SoDEPRECATED_BEGIN(9620)
  /**
  * Used only for legacy SHADOW_MAP method.
  * Kept for compatibility purposes with old iv files.
  */
  SoDEPRECATED_MEMBER(9620, "No longer used since SHADOW_MAP is not supported")
  SoSFBool        isShapesBefore;
#endif  /** @DEPRECATED_END */

  /** Add dependency on various elements */
  void addElementsToCache(SoState* state, SoCache* cache) const;

  /** Allocate the good SoShadowGroupImpl accordind to method field*/
  void chooseShadowingMethod();

  /** The suitable implementation is created depending on the method.*/
  inventor::impl::SoShadowGroupImpl* createImplementation() const;

  void commonInit();

  /** Texture unit define by SoPreferences */
  static int s_defaultTextureUnit;
};
/*----------------------------------------------------------------------------*/
#endif // _SO_SHADOW_GROUP_


