
/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_SHAPE_
#define  _SO_SHAPE_

#include <Inventor/SbBox.h>
#include <Inventor/SbMatrix.h>
#include <Inventor/actions/SoRayPickAction.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/SbPImpl.h>

class SoDetail;
class SbVec2s;
class SoFaceDetail;
class SoPickedPoint;
class SoPrimitiveVertex;
class SoState;
class SoMFInt32;
class PickedPointListImpl;

typedef SoPrimitiveVertex *jySoPrimitiveVertexPtr;
typedef SoDetail *jySoDetailPtr;

SO_PIMPL_PUBLIC_DECLARATION(SoShape)

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoShape
//
//  Abstract base shape node class, which provides some convenience
//  functions for its subclasses.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * Abstract base class for all shape nodes.
 * 
 * @ingroup ShapeNodes
 * 
 * @DESCRIPTION
 *   This node is the abstract base class for all shape (geometry) nodes. All classes
 *   derived from SoShape draw geometry during render traversal.
 * 
 * @FILE_FORMAT_DEFAULT
 *    This is an abstract class. See the reference page of a derived class for the
 *   format and default values.
 * 
 * @SEE_ALSO
 *    SoCone,
 *    SoCube,
 *    SoCylinder,
 *    SoFullSceneAntialiasing,
 *    SoIndexedNurbsCurve,
 *    SoIndexedNurbsSurface,
 *    SoNurbsCurve,
 *    SoNurbsSurface,
 *    SoShapeHints,
 *    SoShapeKit,
 *    SoSphere,
 *    SoText2,
 *    SoText3,
 *    SoVertexShape
 * 
 * 
 */
class INVENTOR_API SoShape : public SoNode
{
  SO_NODE_ABSTRACT_HEADER(SoShape);
  SO_PIMPL_PUBLIC_HEADER(SoShape)

 public:
  /**
   * Whether to ignore this node during bounding box traversal. Default is FALSE.
   */
  SoSFBool boundingBoxIgnoring;

  /**
   * Overrides default method on SoNode
   */
  virtual SbBool affectsState() const;

  /** 
   * Basic type for antialiased rendering for this shape (Do not consider the SoDrawStyle
   * property currently in the state). Default shape type is POLYGONS.
   */

  enum ShapeType {
  /** The shape is render based on points (Ex: SoPointSet).*/
    POINTS,
  /** The shape is render based on lines (Ex: SoLineSet).*/
    LINES,
  /** The shape is render based on polygons (Ex: SoFaceSet).*/
    POLYGONS,
  /** The shape is a text and follows the special filter for texts.*/
    TEXT
  };

  /**
   * Gets the current shape Full Scene Antialiasing type.
   */
  ShapeType getShapeType();

  /**
   * Returns TRUE if the primitive restart feature is available.
   *
   * It is better to call this function with an SoGLContext created on
   * the device used for rendering or with a valid state. 
   *
   * Primitive restart (if available) can be used with SoBufferedShape.
   */
  static SbBool isPrimitiveRestartAvailable(SoState* state = NULL);

 SoEXTENDER public:

  /**
   * This type is used by the triangle shape generation methods
   * (beginShape, etc.)
   */

  enum TriangleShape {
    TRIANGLE_STRIP,
    TRIANGLE_FAN,
    TRIANGLES,
    POLYGON,
    SEGMENT_STRIP,
    POINT_SET
  };

  /**
   * These methods can be used by subclasses to generate triangles
   * more easily when those triangles are part of a larger
   * structure, such as a triangle strip, triangle fan, or
   * triangulated polygon, according to the TriangleShape enumerated
   * type. The sequence of calls is similar to GL's: begin a shape,
   * send vertices of that shape, then end the shape.
   *
   * If the face detail passed to beginShape() is NULL (the
   * default), the details in the vertices will be used as is.
   * Note that some vertices may be copied into local storage; the
   * detail pointers are copied as well, so the details themselves
   * should be consistent for the duration of the shape generation.
   *
   * If the face detail passed to beginShape() is non-NULL, the
   * details in the vertices are assumed to be SoPointDetails. Each
   * vertex of each triangle generated will contain a pointer to the
   * face detail, which will be filled with three copies of the
   * point details from the relevant vertices. Since copies of the
   * point details are made, the storage for each point detail
   * passed to shapeVertex() can be re-used by the caller.
   */
  void beginShape( SoAction *action, TriangleShape shapeType,
                   SoFaceDetail *faceDetail = NULL);

  /** 
   * add a primitive vertex to the shape prevously started with beginShape().
   * [OIV-WRAPPER-ARG IN]
   */
  void shapeVertex(const SoPrimitiveVertex *v);



  /**
   * end shape previously started with beginShape().
   */
  void endShape();
  
  /**
   * Implements bounding box method using virtual computeBBox()
   * method. Subclasses should not need to override this.
   */
  virtual void getBoundingBox(SoGetBoundingBoxAction *action);

  /**
   * Implements rendering by rendering each primitive generated by
   * subclass. May be overridden in subclass.
   */
  virtual void GLRender(SoGLRenderAction *action);

  /**
   * Implements picking along a ray by intersecting the ray with
   * each primitive generated by subclass. May be overridden in
   * subclass.
   */
  virtual void rayPick(SoRayPickAction *action);

  /**
   * Implements primitive generation for all shapes. Calls the 
   * generatePrimitives for the shape.
   */
  virtual void callback(SoCallbackAction *action);

  /**
   * This can be used by subclasses when the complexity type is
   * SCREEN_SPACE to determine how many window pixels are covered by
   * the shape. It returns in rectSize the number of pixels in the
   * window rectangle that covers the given 3D bounding box.
   */
  static void getScreenSize(SoState *state, const SbBox3f &boundingBox, SbVec2s &rectSize);

  /**
   * This can be used by dynamic shape subclasses that use
   * the complexity element, to compute how the decimation
   * element affects the complexity of the shape.
   * The shape passes in the complexity it would have
   * used and this function returns the new complexity to use
   * based on decimation element.
   */
  static float getDecimatedComplexity(SoState *state, float complexity);

  /**
   * Compute axis aligned bounding box (AABB) for subclass using information in the
   * given action (which may not necessarily be an
   * SoGetBoundingBoxAction). This is used by getBoundingBox() and
   * when rendering or picking a shape with bounding-box complexity.
   * Subclasses must define this method.
   */
  virtual void computeBBox(SoAction *action, SbBox3f &box, SbVec3f &center) = 0;

  /** 
   * Compute object oriented bounding box (OOB) for subclass using information in the
   * given action (which may not necessarily be an
   * SoGetBoundingBoxAction). This is used by getBoundingBox() and
   * when rendering or picking a shape with bounding-box complexity.
   *
   * This method returns an Object Oriented Bounding Box (OOB), which may
   * be more efficient for picking, colision detection, and culling compared 
   * to an axis aligned bounding box, at the cost of more computing time.
   * Subclasses should override this method, if not then the axis aligned
   * bounding bbox is used.
   * 
   * By default Open Inventor only uses the axis aligned bounding box (AABB).
   * OOB can be enabled using SoComplexity or the OIV_USE_OOB environment variable.
   *
   */
  virtual void computeBBox(SoAction *action, SbXfBox3d &box, SbVec3d &center)
  {
    // SubClass did not redefine this function so by default use AABB version
    SbBox3f boxf;
    SbVec3f centerf;
    computeBBox(action, boxf, centerf);

    box.setTransform(SbMatrixd::identity());
    box.makeEmpty();
    box.setBounds(SbVec3d(boxf.getMin().getValue()),SbVec3d(boxf.getMax().getValue()));
    center.setValue(centerf);
  }

  /**
   * Used by BoundingBoxAction to know if bounding box computation should 
   * be ignored or not.
   * By default the returned value is given by boundingBoxIgnoring field.
   */
  inline virtual SbBool isBoundingBoxIgnoring() const
  { return boundingBoxIgnoring.getValue(); }

  /**
   * Counts number of primitives produced by this shape.
   * Default action uses primitive callbacks, subclasses should
   * derive their own method if they can do it faster.
   */
  virtual void getPrimitiveCount(SoGetPrimitiveCountAction *action);

  // Gets multi-texturing info:
  //    - Indicates if the multi-texturing is active
  //    - The list of texture units where texture coordinate should be sent (that is
  //       where texture coordinates different from a function.)
  static SbBool getMultiTextureInfo(SoState *state, SbIntList &imageTextUnitList) ;
                                   
  /**
   * set the antialiasing type for this shape.
   */
  void setShapeType(SoShape::ShapeType st);

  // These methods invoke whatever callbacks are registered with the
  // primitive generation procedures. If primitives are being
  // generated for picking or rendering purposes, the appropriate
  // SoShape methods are called. If primitives are being generated
  // for the SoCallbackAction, the callbacks registered with the
  // action are called.
  void invokeTriangleCallbacks(SoAction *action,
                               const SoPrimitiveVertex *v1,
                               const SoPrimitiveVertex *v2,
                               const SoPrimitiveVertex *v3);

  void invokeLineSegmentCallbacks(SoAction *action,
                                  const SoPrimitiveVertex *v1,
                                  const SoPrimitiveVertex *v2);

  void invokePointCallbacks(SoAction *action,
                            const SoPrimitiveVertex *v);

 
 SoINTERNAL public:
  static void initClass();
  static void exitClass();

  // redefined for Shapes
  virtual void GLRenderP(SoGLRenderAction *action);

  SB_THREAD_TLS_HEADER();

  // It is possible to disable projection for a given shape even if the element is set
  SbBool m_forceDisableProjection;

  void setForceDisableProjection( SbBool disable ){m_forceDisableProjection = disable;}

  static bool isVertexArrayAvailable(SoState* state=NULL);
  static bool isMultiDrawArrayAvailable(SoState* state=NULL);
  static bool isVertexBufferAvailable(SoState* state=NULL);

  // Returns 0 if not available, 1 if NV version, 2 if core
  static int getPrimitiveRestartMode(SoState* state = NULL);

  static int getMaxPrimitiveLoop();

  static int getMinVertexNumForVA();

  static int getMinVertexNumForVBO();

  static int getMinVertexNumForVAVBONoCache();

  static bool isDesindexingAllowed();

  static bool isTrisStripAllowedForQuadMesh();

  static SbBool isDetailDebugEnabled();

  static bool isCreaseAngleFixSet();

  // This endShape is used by SoIndexedFaceSet and SoFaceSet to create hole
  // in a face.
  void endShape( int wt, SoMFInt32* indexes );

  // Returns TRUE if the shape should be rendered. Subclasses can
  // call this in their rendering methods to determine whether to
  // continue.
  SbBool shouldPrimitiveCount(SoGetPrimitiveCountAction *action);

  virtual int getNumVert(SoState *state = NULL ){ return getNumPrim( state ); }

  virtual int getNumPrim( SoState *state = NULL );

  inline SbBool isRenderable(SoGLRenderAction* action)
  {
    return shouldGLRender(action, FALSE);
  }

 protected:

  SoShape();

  virtual SbBool checkShapeHints(SoGLRenderAction *action);

  inline virtual void countPrim( SoState *state = NULL );

  virtual SbBool checkTransparency(SoGLRenderAction *action, SbBool isPointsOrLines);

  virtual SbBool isRenderingPointsOrLines(SoGLRenderAction *action);

  virtual SbBool shouldRenderNow(SoGLRenderAction *action,SbBool &isPointsOrLines);

  //-------------------------------------------------------------------------------

  // This method MUST be defined by each subclass to generate
  // primitives (triangles, line segments, points) that represent
  // the shape.
  virtual void generatePrimitives(SoAction *action) = 0;

  // Returns TRUE if the shape should be rendered. Subclasses can
  // call this in their rendering methods to determine whether to
  // continue.  
  virtual SbBool shouldGLRender(SoGLRenderAction *action,
                                SbBool isPointsOrLines = FALSE);

  // Implementation of shouldGLRender to split the general case
  // to tessellationShader support case. shouldGLRender always
  // return false if tessellation shaders are on the state. This
  // method is called for every shape with no tessellation shaders on the
  // state and bufferedShape.
  SbBool shouldGLRenderCore(SoGLRenderAction *action,
                      SbBool isPointsOrLines = FALSE,
                      SbBool hasTessellationShader = FALSE);

  // Returns FALSE if the shape is invisible. If the complexity
  // is BOUNDING_BOX, it renders the cuboid and returns FALSE.
  // Otherwise returns TRUE.
  SbBool handleInvisibleOrBBox(SoGLRenderAction *action,
                                SbBool isPointsOrLines = FALSE);

  // Returns TRUE if the shape may be picked. Subclasses can call
  // this in their picking methods to determine whether to continue.
  SbBool shouldRayPick(SoRayPickAction *action);



  /// These can be used by solid shapes to indicate so during
  // rendering. They take care of setting up shape hints in the
  // state. Calls to these two methods should bracket the rendering
  // code.
  void beginSolidShape(SoGLRenderAction *action);
  void endSolidShape(SoGLRenderAction *action);

  // Computes a picking ray in the object space of the shape
  // instance. The picking ray is stored in the SoRayPickAction for
  // later access by the subclass. The second form takes a matrix to
  // concatenate with the current transformation matrix. It can be
  // used, for example, if a shape has sizing or positioning info
  // built into it.
  void computeObjectSpaceRay(SoRayPickAction *action);
  void computeObjectSpaceRay(SoRayPickAction *action,
                             const SbMatrix &matrix);

  // These methods are called during picking to create details
  // representing extra info about a pick intersection. The default
  // methods implemented in SoShape return NULL. Subclasses can
  // override these methods to set up specific detail instances that
  // contain the extra info. Subclasses can get whatever information
  // they need from the SoPickedPoint, and can set information in it
  // as well.
  virtual SoDetail *createTriangleDetail(SoRayPickAction *action,
                                         const SoPrimitiveVertex *v1,
                                         const SoPrimitiveVertex *v2,
                                         const SoPrimitiveVertex *v3,
                                         SoPickedPoint *pp);
  virtual SoDetail *createLineSegmentDetail(SoRayPickAction *action,
                                            const SoPrimitiveVertex *v1,
                                            const SoPrimitiveVertex *v2,
                                            SoPickedPoint *pp);
  virtual SoDetail *createPointDetail(SoRayPickAction *action,
                                      const SoPrimitiveVertex *v,
                                      SoPickedPoint *pp);
  


  virtual ~SoShape();
    
  // Applies GLRender action to the bounding box surrounding the shape.
  // This is used to render shapes when BOUNDING_BOX complexity is on.
  virtual void GLRenderBoundingBox(SoGLRenderAction *action);

  // Changes the GL_MULTISAMPLE_ARB OpenGL bit value. This method is used to filter the
  // antialiasing when traversing the shape. (See SoFullSceneAntialiasing for more).
  virtual void applyFullSceneAntialiasingFilter(SoState* state);

  // isProjectCoordNeeded = true if projection need to be done by SoShape
  // isProjectCoordNeeded = false if projection is already handled somewhere (ex: vertexShape)
  void setNeedToProjectCoord(bool isProjectCoordNeeded);

 protected:
  
  static int s_keepReset;
  static int s_forceSend;
  static SbBool s_noClip;

private:

  // Applies rayPick action to the bounding box surrounding the shape.
  // This is used to pick shapes when BOUNDING_BOX complexity is on.
  void rayPickBoundingBox(SoRayPickAction *action);

  void commonConstructor();
};

inline SbBool 
SoShape::checkShapeHints(SoGLRenderAction *)
{
  return FALSE;
}

void
SoShape::countPrim( SoState* /*state*/ )
{}

#endif /* _SO_SHAPE_ */
