/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/


#ifndef  _SO_STEREO_CAMERA_
#define  _SO_STEREO_CAMERA_

#include <Inventor/fields/SoSFBool.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>

/**
 * Stereo camera node.
 *
 * @ingroup CameraNodes
 *
 * @DESCRIPTION
 *   A stereo camera defines a specific perspective camera for stereo support.
 *
 *   This class defines fields to store the stereo settings related to the camera:
 *   - Stereo offset: the distance of each eye from the camera position.
 *   - Stereo balance: the position of the zero parallax plane.
 *
 *   See the base class SoPerspectiveCamera for more information about the inherited fields.
 *
 * @FILE_FORMAT_DEFAULT
 *    StereoCamera {
 *    @TABLE_FILE_FORMAT
 *       @TR offset                @TD 0.7
 *       @TR balance               @TD 1.0
 *       @TR balanceNearFrac       @TD FALSE
 *       @TR absoluteAdjustments   @TD FALSE
 *    @TABLE_END
 *    }
 *
 * @ACTION_BEHAVIOR
 *    Sets: SoStereoElement
 *
 * @SEE_ALSO
 *    SbViewVolume,
 *    SoPerspectiveCamera,
 *    SoCameraInteractor
 *
 */

class INVENTOR_API SoStereoCamera : public SoPerspectiveCamera {

  SO_NODE_HEADER(SoStereoCamera);

public:

  /**
   * The stereo offset (the distance of each eye from the camera position).
   * The right eye is moved plus offset and the left eye is moved minus offset.
   * Default is 0.7.
   */
  SoSFFloat offset;

  /**
   * The stereo balance (the position of the zero parallax plane).
   * Default balance is 1.0.
   */
  SoSFFloat balance;

  /**
   * Specifies whether the balance value is defined as a fraction of the camera near distance.
   *
   * Note: Since the projection matrix always depends on the camera's near plane, in
   * some cases it may be necessary to detect changes to the camera near plane and
   * adjust by setting a new stereo balance value. Open Inventor will make these
   * adjustments automatically if the @B balanceNearFrac @b field is set to TRUE.
   * In this case the stereo balance value is defined as a fraction of the camera
   * near distance.
   *
   * Default nearFrac is FALSE.
   */
  SoSFBool balanceNearFrac;

  /**
   * Specifies if stereo adjustments are absolute. FALSE by default.
   *
   * The default non-absolute mode allows the stereo settings to be valid over a range
   * of different view volume settings. If you chose absolute mode, you are responsible
   * for modifying the stereo settings (if necessary) when the view volume changes.
   *
   * When absolute mode is TRUE, stereo offset and balance are used as shown in the
   * following pseudo-code for the right eye view:
   * \code
   * StereoCameraOffset = offset.getValue();
   * FrustumAsymmetry   = balance.getValue();
   *
   * glTranslated (-StereoCameraOffset, 0, 0);
   * glFrustum (FrustumLeft + FrustumAsymmetry, FrustumRight + FrustumAsymmetry,
   *            FrustumBottom, FrustumTop, NearClipDistance, FarClipDistance);
   * \endcode
   * The left eye view is symmetric.
   *
   * When absolute mode is FALSE, stereo offset and balance are used as shown in
   * the following pseudo-code for the right eye view:
   *
   * Xrange is right minus left (i.e., first two arguments of glFrustum) and
   * multiply that difference by the ratio of the distance to the desired plane
   * of zero parallax to the near clipping plane distance.
   * \code
   * StereoCameraOffset   = Xrange * 0.035 * offset.getValue();
   * FrustumAsymmetry     = -StereoCameraOffset * balance.getValue();
   * ZeroParallaxDistance = (NearClipDistance + FarClipDistance)/0.5;
   *
   * FrustumAsymmetry *= NearClipDistance / ZeroParallaxDistance;
   *
   * glTranslated (-StereoCameraOffset, 0, 0);
   * glFrustum (FrustumLeft + FrustumAsymmetry, FrustumRight + FrustumAsymmetry,
   *            FrustumBottom, FrustumTop, NearClipDistance, FarClipDistance);
   * \endcode
   * The left eye view is symmetric.
   */
  SoSFBool absoluteAdjustments;

  /**
   * Creates a stereo camera node with default settings.
   */
  SoStereoCamera();

#if SoDEPRECATED_BEGIN(9700)

  SoDEPRECATED_METHOD_NOWARN( 9700, "Use offset field instead." )
  virtual void setStereoAdjustment( float adjustment );

  SoDEPRECATED_METHOD_NOWARN( 9700, "Use absoluteAdjustments field instead." )
  virtual void setStereoAbsoluteAdjustments( SbBool absolute );

  SoDEPRECATED_METHOD_NOWARN( 9700, "Use balance and balanceNearFrac fields instead." )
  virtual void setBalanceAdjustment( float adjustment, SbBool nearFrac = false );

  /**
   * Allows the camera to render in stereo.
   * Do nothing here as this camera is always allowed to render in stereo.
   */
  SoDEPRECATED_METHOD_NOWARN( 9700, "Unused" )
  virtual void allowStereo( SbBool allowed );

#endif /** @DEPRECATED_END */

SoINTERNAL public:

  static void initClass();
  static void exitClass();

  virtual void notify( SoNotList* list );

protected:

  virtual ~SoStereoCamera();

  virtual float getEyeOffset( SoState* state = NULL );
  virtual void updateStereoAdjustments( SoState* state = NULL );

private:

  /** Set stereo parameters in stereo element. */
  void setStereoElement( SoState* state );

};

#endif /* _SO_STEREO_CAMERA_ */
