/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Nick Thompson (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_SWITCH_
#define  _SO_SWITCH_

#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/nodes/SoGroup.h>

/**
 * @file SoSwitch.h
 * Switch group node: traverses only the child indicated by integer
 * "whichChild" field. If this field is #SO_SWITCH_NONE, no children
 * are traversed, while #SO_SWITCH_INHERIT means inherit the index
 * from the current switch state, doing a modulo operation to make
 * sure the child index is within the correct bounds of the switch
 * node.
 */


/**
 * Possible value of SoSwitch::whichChild : Don't traverse any children
 *
 * @relates SoSwitch
 */
#define SO_SWITCH_NONE          (-1)

/**
 * Possible value of SoSwitch::whichChild : Inherit value from state
 *
 * @relates SoSwitch
 */
#define SO_SWITCH_INHERIT       (-2)

/**
 * Possible value of SoSwitch::whichChild : Traverse all children
 *
 * @relates SoSwitch
*/
#define SO_SWITCH_ALL           (-3)

SO_PIMPL_PUBLIC_DECLARATION( SoSwitch )

/**
 * Group node that traverses one chosen child.
 * 
 * @ingroup GroupNodes
 * 
 * @DESCRIPTION
 *   This group node traverses exactly one, none or all of its children. It
 *   implements an operation similar to the switch statement in C. One can use this
 *   node to switch on and off the effects of some properties, to switch between
 *   different properties, to switch between different geometry or to control the
 *   visibility of geometry.
 *
 *   The #whichChild field specifies the index of the child to traverse, where
 *   the first child has index 0. There are several special values that can be used:
 *   - #SO_SWITCH_NONE (-1, the default) means do not traverse any children.
 *
 *   - #SO_SWITCH_INHERIT (-2) allows the index to be inherited from a
 *     previously-encountered SoSwitch node or from certain other nodes (such as
 *     SoArray or SoMultipleCopy) that set the switch value.
 *
 *   - #SO_SWITCH_ALL (-3) traverses all children, making the switch behave exactly
 *     like a regular SoGroup.
 *
 *   There are several extended versions of this node with additional/different behavior:
 *   - SoInteractiveSwitch: During a user interaction, for example moving the camera,
 *     traverses a different (or no) child specified by its whichInteractiveChild field.
 *
 *   - SoMultiSwitch: Traverses any number of its children (specified in a list of indices).
 *
 *   - SoPathSwitch: Only traverses its children if the current path (when it is traversed)
 *     matches or contains a path specified in its @I path@i field.
 *
 *   - SoMultiPathSwitch: Only traverses its children if the current path matches or
 *     contains any of the paths specified in its @I paths@i field.
 *
 *   @B NOTES: @b
 *   - The default value for #whichChild is #SO_SWITCH_NONE, meaning
 *     that @I nothing@i is displayed by default.  In most cases you will want to set some
 *     other initial value in this field.
 *
 *   - When the value of #whichChild is #SO_SWITCH_INHERIT, if the inherited value is
 *     larger than the number of children in the current node, the actual value used
 *     is "modulo" the number of children.
 *
 *   - Some actions have an option that allows them to ignore the value of #whichChild.
 *     For example, calling setSearchingAll(true) on an SoSearchAction causes it to
 *     traverse all the children of SoSwitch nodes.
 *
 *   - SoSwitch is commonly used to control the visibility of a geometry.  For example
 *     an application might allow the user to create multiple slices through a volume
 *     (SoOrthoSlice) or mesh (MoLogicalSlice) data set, but also allow the user to
 *     control the visibility of each slice separately.
 *     Using SoSwitch with #SO_SWITCH_NONE is much more efficient than removing geometry
 *     from the scene graph because previously built caches are preserved. Using SoSwitch
 *     is also much more efficient than using draw style INVISIBLE (SoDrawStyle) because
 *     the children are not traversed.
 *
 *   - This node is an SoGroup, @B not@b an SoSeparator, and therefore does not
 *     provide render caching.  Generally if the children contain geometry, it is good 
 *     practice to make each child an SoSeparator to make render caching more effective.
 * 
 * @FILE_FORMAT_DEFAULT
 *    Switch {
 *    @TABLE_FILE_FORMAT
 *       @TR whichChild   @TD -1
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction,  SoCallbackAction,  SoGetBoundingBoxAction,  SoGetMatrixAction,  SoHandleEventAction,  SoRayPickAction @BR
 *        Traverses the chosen child or children. Sets: SoSwitchElement
 * 
 *    SoSearchAction @BR
 *        If the action's Searching-All flag is set, always traverses all children.
 *        Otherwise, traverses just the chosen child or children.
 * 
 * 
 * @SEE_ALSO
 *    SoArray,
 *    SoLevelOfDetail,
 *    SoMultipleCopy,
 *    SoPathSwitch,
 *    SoMultiSwitch,
 *    SoInteractiveSwitch
 * 
 * 
 */
class INVENTOR_API SoSwitch : public SoGroup
{

  SO_NODE_HEADER(SoSwitch);
  SO_PIMPL_PUBLIC_HEADER( SoSwitch )

 public:
  /**
   * Index of the child to traverse, or one of #SO_SWITCH_NONE, #SO_SWITCH_INHERIT, or
   * #SO_SWITCH_ALL.
   * 
   */
  SoSFInt32 whichChild;

  /**
   * Creates a switch node with default settings.
   */
  SoSwitch();

  /**
   * Constructor that takes approximate number of children.
   */
  SoSwitch(int nChildren);

 SoEXTENDER public:
  /**
   * Overrides method in SoNode to return FALSE if there is no
   * selected child or the selected child does not affect the state.
   */
   virtual SbBool affectsState() const;

  // Implement actions
  virtual void doAction(SoAction *action);
  virtual void callback(SoCallbackAction *action);
  virtual void GLRender(SoGLRenderAction *action);
  virtual void pick(SoPickAction *action);
  virtual void getBoundingBox(SoGetBoundingBoxAction *action);
  virtual void handleEvent(SoHandleEventAction *action);
  virtual void getMatrix(SoGetMatrixAction *action);
  virtual void search(SoSearchAction *action);
  virtual void write(SoWriteAction *action);
  virtual void getPrimitiveCount(SoGetPrimitiveCountAction *action);

 SoINTERNAL public:
  static void initClass();
  static void exitClass();

  virtual void notify(SoNotList *list);

  /**
   * Returns TRUE if a node has an effect on the traversal path. The default
   * method returns FALSE. Node classes such as SoSwitch that allow
   * to traverse scene in different ways returns TRUE.
   */
  virtual bool affectsPath() const;

 protected:
  /** Destructor. */
  virtual ~SoSwitch();

  // Traverses correct child. If matchIndex >=0, child index must match
  virtual void doChild(SoAction *action, int matchIndex = -1);

 private:
  void commonConstructor();
};

#endif /* _SO_SWITCH_ */

