/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : Nicolas DAGUISE (Oct 2004)
**=======================================================================*/


#ifndef  _SO_TEXT_3_
#define  _SO_TEXT_3_

#include <Inventor/SbBox.h>
#include <Inventor/fields/SoMFString.h>
#include <Inventor/fields/SoSFBitMask.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoMFVec3f.h>
#include <Inventor/nodes/SoShape.h>

class SoAnnoText3;
class SoAsciiText;

SO_PIMPL_PUBLIC_DECLARATION( SoText3 );

/**
 * 3D text shape node.
 * 
 * @ingroup ShapeNodes
 * 
 * @DESCRIPTION
 *   This node defines one or more strings of 3D text. In contrast with SoText2, 3D
 *   text can be rotated, scaled, lighted, and textured like all other 3D shapes. 
 *
 *   Optionally (setting the #parts field) each character in a 3D text string may
 *   be a true three dimensional object. This shape is created by extruding an outlined
 *   version of the character (in the current typeface) along the current profile, as
 *   defined by nodes derived from SoProfile. The default text profile, if none is
 *   specified, is a straight line segment one unit long.
 *   
 *   The text position is always at the local origin (0,0,0).  A transform node, 
 *   e.g. SoTranslation, must be used to position the text in 3D space. 
 *
 *   The text is always rendered in its local XY plane.  A transform node, e.g. 
 *   SoRotation, may be used to rotate the text plane in 3D space.  
 *
 *   Text supports both single byte Latin1 (ISO8859-1) characters and multi-byte 
 *   UNICODE characters.  ISO8859-1 is an 8-bit character set which is a superset
 *   of the older 7-bit ASCII character set. See the SbString class for more information.
 *   
 *   @B Rendering: @b
 *
 *   The typeface and size (default = 10) are specified using an SoFont node.
 *   The scale factor, if any, in the current transform is applied to this size.
 *
 *   3D text characters may be rendered using lines, polygons or textured rectangles
 *   depending on settings in the SoFont node.  Textured rendering is recommended.
 *   It generally has the best performance (minimum geometry) and the best image
 *   quality (because antialiasing is applied).
 *   - Line rendering is selected by specifying a "stroke font" name in the SoFont
 *     @I name @i field. In this case line attributes apply instead of polygon
 *     attributes and text is not extruded.
 *   - Polygon rendering is selected by setting the SoFont @I renderStyle@i field
 *     to POLYGON (the default) or POLYGON_AND_OUTLINE. 
 *   - Textured quad rendering is selected by setting the SoFont @I renderStyle@i
 *     field is set to TEXTURE or TEXTURE_DELAYED. @BR
 *
 *   Rendering styles: Additional styles can be applied using an SoTextProperty node, 
 *   including underline, strikethrough, background and background outline.
 *
 *   Complexity: The quality (degree of tesselation) of SoText3 line and polygon
 *   rendering depends on the SoComplexity node's @I value@i field.  The quality
 *   of SoText3 textured rendering depends on the SoComplexity node's @I value@i and
 *   @I textureQuality@i fields, plus the SoTextProperty node's @I textureQualityRange@i
 *   field. If there is no SoComplexity the textureQuality is 1. See SoFont for more discussion.
 *
 *   Material: SoText3 uses the current set of materials when rendering
 *   but only OVERALL, PER_PART and PER_PART_INDEXED. Any other binding will behave like OVERALL.
 *   If the material binding is OVERALL, then the whole text is drawn with the first material.
 *   If it is PER_PART or PER_PART_INDEXED, the front part of the text is drawn with the first
 *   material, the sides with the second, and the back with the third.
 *
 *   Lighting: By default lighting is only computed for the "front" face of geometry,
 *   including SoText3, meaning that the back face of the text will be black when
 *   lighting is enabled.  To illuminate both sides, enable "two sided" lighting using
 *   the SoShapeHints node's @I vertexOrdering@i field.  But note that, unlike most geometry,
 *   you should set the vertexOrdering field to CLOCKWISE, not COUNTERCLOCKWISE.
 *   
 *   Texture: Textures are applied to 3D text as follows. On the front and back faces of the
 *   text, the texture origin is at the base point of the first string; the base
 *   point is at the lower left for justification LEFT, at the lower right for RIGHT,
 *   and at the lower center for CENTER. The texture is scaled equally in both S and
 *   T dimensions, with the font height representing 1 unit. S increases to the right
 *   on the front faces and to the left on the back faces. On the sides, the texture
 *   is scaled the same as on the front and back. S is equal to 0 at the rear edge of
 *   the side faces. The T origin can occur anywhere along each character, depending
 *   on how that character's outline is defined.
 *
 *   Performance:
 *   A scene graph containing a large number of SoText3 nodes will generally have much
 *   better rendering performance using a text specific caching mechanism called
 *   @I accumulation rendering@i mode. However some features are not available
 *   and significant time may be required to build (or rebuild) this cache.
 *   See the SoTextProperty node @I enableTextAccumulation@i field for more information.
 *
 *   @B Justification: @b
 *
 *   Horizontal alignment of the text can be specified using the #justification
 *   field.  Both horizontal and vertical alignment can be controlled using an 
 *   SoTextProperty node, however the justification setting in this node overrides
 *   the horizontal setting unless the justification field is set to INHERITED.
 *
 *   @B Picking: @b
 *
 *   Text can be picked (SoRayPickAction) like any other geometry. If a text
 *   node has been picked then SoPickedPoint can return an SoTextDetail object.
 *   This object contains information about which string (in a multi-string text
 *   node) was picked and which character within the string was picked.
 *
 *   Antialiasing: @BR
 *   Shape Antialiasing type is SoShape::TEXT.
 * 
 * @FILE_FORMAT_DEFAULT
 *    Text3 {
 *    @TABLE_FILE_FORMAT
 *       @TR string          @TD ""
 *       @TR spacing         @TD 1
 *       @TR justification   @TD LEFT
 *       @TR parts           @TD FRONT
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction @BR
 *        Draws text based on the current font, profiles, transformation, drawing style,
 *        material, texture, complexity, and so on.
 * 
 *    SoRayPickAction @BR
 *        Performs a pick on the text. The string index and character position are
 *        available from the SoTextDetail.
 * 
 *    SoGetBoundingBoxAction @BR
 *        Computes the bounding box that encloses the text.
 * 
 *    SoCallbackAction @BR
 *        For non-stroke fonts,if any triangle callbacks are registered with the action,
 *        they will be invoked for each successive triangle used to approximate the text
 *        geometry.
 *        
 *        For stroke fonts, if any line segment callbacks are registered with the action,
 *        they will be invoked for each successive line segment used to approximate the
 *        text geometry.
 * 
 * 
 * @SEE_ALSO
 *    SoComplexity,
 *    SoFont,
 *    SoFullSceneAntialiasing,
 *    SoProfile,
 *    SoText2,
 *    SoTextDetail,
 *    SoTextProperty
 * 
 */
class INVENTOR_API SoText3 : public SoShape {

  SO_PIMPL_PUBLIC_HEADER( SoText3 );

  SO_NODE_HEADER(SoText3);

 public:
   /** Justification types */
  enum Justification {
    /**
     *  Left edges of all strings are aligned 
     */
    LEFT   = 0x01,
    /**
     *  Centers of all strings are aligned 
     */
    CENTER = 0x03,
    /**
     *  Right edges of all strings are aligned 
     */
    RIGHT  = 0x02,
    /** 
     *  The text justification is inherited from the current SoTextProperty::alignmentH value.
     */
    INHERITED = 0x04
  };

  /** Text3 part values. */
  enum Part {
    /**
     *  Front faces of characters 
     */
    FRONT = 0x01,
    /**
     *  Extruded sides of characters 
     */
    SIDES = 0x02,
    /**
     *  Back faces of characters 
     */
    BACK  = 0x04,
    /**
     *  All parts 
     */
    ALL   = 0x07
  };

  // Fields
  /**
   * The text string(s) to display. Each string will appear on its own line.
   * 
   */
  SoMFString string;

  /**
   * Defines the distance (in the negative y direction) between the base points of
   * successive strings, measured with respect to the current font height. A value of
   * 1 indicates single spacing, a value of 2 indicates double spacing, and so on.
   * 
   */
  SoSFFloat spacing;

  /**
   * Which parts of text are visible. Note that, for speed, the default for this
   * field is FRONT only.
   * 
   */
  SoSFBitMask parts;

  /**
   * Indicates placement and alignment of strings. With LEFT justification, the left
   * edge of the first line is at the (transformed) origin, and all left edges are
   * aligned. RIGHT justification is similar. CENTER justification places the center
   * of the first string at the (transformed) origin, with the centers of all
   * remaining strings aligned under it.
   * Use enum #Justification. Default is LEFT.
   */
  SoSFEnum justification;

  /**
   * Creates a 3D text node with default settings.
   */
  SoText3();

 SoEXTENDER public:

  /**
   * Method to get the bounds of the given character in the given
   * string.  This must be called during the application of an
   * action; use a callbackAction applied to a Path to the text node
   * if you have to (this is really designed to be used in a Text3
   * manipulator, which will have direct access to the state).
   * The bounds are based on the font metric information, not the
   * geometric bounding box of the character.  So, for example,
   * while a space character has an empty bounding box,
   * getCharacterBounds() will return a box as high as the font
   * height, as wide as a space, and as deep as the profile.
   */
  SbBox3f getCharacterBounds(SoState* state, int stringIndex, int charIndex);

  virtual void GLRender(SoGLRenderAction* action);
  virtual void rayPick(SoRayPickAction* action);
  virtual void getPrimitiveCount(SoGetPrimitiveCountAction* action);
  
  /**
   * Computes bounding box of text.
   */
  virtual void computeBBox(SoAction* action, SbBox3f& box, SbVec3f& center);

  // Creates a text detail when picking:
  SoDetail* createTriangleDetail( SoRayPickAction*,
                                  const SoPrimitiveVertex*,
                                  const SoPrimitiveVertex*,
                                  const SoPrimitiveVertex*,
                                  SoPickedPoint* );

SoINTERNAL public:
  static void initClass();
  static void exitClass();

  void setAsciiText(SoAsciiText*);
  SoAsciiText* getAsciiText() const;
  virtual void notify(SoNotList* list);

protected:
    
  /**
   * Destructor.
   */
  virtual ~SoText3();

  /**
   * Returns TRUE if the shape should be rendered. Subclasses can
   * call this in their rendering methods to determine whether to
   * continue.
   */
  virtual SbBool shouldGLRender(SoGLRenderAction* action, SbBool isPointsOrLines = FALSE);

  /**
   * Generates primitives.
   */
  virtual void generatePrimitives(SoAction* );

  friend class SoAnnoText3;
  friend class SoAsciiText;

private:
  void commonConstructor();
};

#endif // _SO_TEXT_3_

/**/


