/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Nicolas DAGUISE (Nov 2004)
**=======================================================================*/



#ifndef  _SO_TEXT_PROPERTY_
#define  _SO_TEXT_PROPERTY_

#include <Inventor/nodes/SoNode.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFVec2i32.h>
#include <Inventor/fields/SoSFBitMask.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoMFColorRGBA.h>
#include <Inventor/fields/SoMFBool.h>
#include <Inventor/elements/SoTextStyleElement.h>

/**
 * @VSGEXT Text property node.
 * 
 * @ingroup PropertyNodes
 * 
 * @DESCRIPTION
 *   This node specifies properties used for text rendering.
 *
 *   The application can specify text alignment and orientation using the #alignmentH, #alignmentV,
 *   and #orientation fields.  
 *
 *   Note: @BR
 *     Horizontal alignment can also be
 *     specified in some text nodes, for example SoText2 and SoText3, using
 *     the @I justification @i field.  The value in the text node's field
 *     is used unless that field is explicitly set to INHERIT.  To allow
 *     SoTextProperty to control horizontal justification, set the text node's
 *     justification field to INHERITED.
 *
 *   The application can also request use of font kerning using the #kerning field. Kerning consists
 *   of modifying the spacing between two successive glyphs according to their outlines. For
 *   examples, a "T" and a "y" can be moved closer together as the top of the "y" fits nicely
 *   under the upper right bar of the "T". Kerning depends on the specific font being used.
 *
 *   Different text rendering styles such as underline, strikethrough and background rectangle
 *   can be specified using the #style field.  A color can be specified for each style using
 *   the #styleColors field.  The current material does not affect the color of a text rendering
 *   style unless the corresponding value in the #styleColorsUseCurrentMaterial field is set to true.
 *   Text rendering styles apply to the entire string.
 *
 * \par
 * @TABLE_1B
 *   @TR Text rendering effects
 *   @TR @IMAGE TextProperties.png
 * @TABLE_END
 *
 *   Notes:
 *   - Underline/overline/strike always has the same position relative to the
 *     text string, regardless of the vertical justification setting.
 *
 *   - The underline/overline/strike position is determined by the font
 *     and the results should be similar to what you see in other
 *     applications using the same font (e.g. MS-Word on Windows).
 *     For some fonts the underline will intersect character descenders,
 *     for example on the lower case 'g'.  For some fonts the overline will
 *     intersect the top portion of some characters.
 *
 *   - When using BACK_FRAME and BACK_FRAME_LINE you will usually want to
 *     set the margin field to a small value, for example 0.05.
 *
 *   - Color can be specified separately for each style. 
 *     However underline and overline are often intended to be the same
 *     color as the text. It can be convenient to set the 
 *     styleColorsUseCurrentMaterial field so that the current SoMaterial
 *     controls both the text and the lines.
 *
 *   - Note that the color for each style is an RGBA value, allowing
 *     transparency to be applied.  This is particularly useful for the
 *     BACK_FRAME attribute so the text background rectangle can be
 *     semi-transparent.
 *
 *   - As backFrameLineWidth is increased, the width of the line expands
 *     outward, away from the string (the backframe box stays the same size).
 *
 *   - Be sure to use the values in the StyleColor enum to set colors and
 *     inheritance flags, not the values in the Style enum. For example,
 *     use UNDERLINE_COLOR, not UNDERLINE.  Open Inventor will not report
 *     an error, but the desired effect will not occur.
 * 
 * @FILE_FORMAT_DEFAULT
 *    TextProperty {
 *    @TABLE_FILE_FORMAT
 *       @TR aliasingFactor  @TD 1.0
 *       @TR alignmentH   @TD LEFT
 *       @TR alignmentV   @TD BASE
 *       @TR backFrameLineWidth       @TD 0.0
 *       @TR kerning      @TD FALSE
 *       @TR margin       @TD 0.0
 *       @TR orientation  @TD LEFT_TO_RIGHT
 *       @TR textureQualityRange  @TD 150 300
 *       @TR style        @TD NONE
 *       @TR styleColors  @TD 0.8 0.8 0.8 1
 *       @TR styleColorsUseCurrentMaterial @TD FALSE
 *    @TABLE_END
 *    }
 *
 * @ACTION_BEHAVIOR
 *    Sets: SoTextAlignmentHElement, SoTextAlignmentVElement, SoTextBackFrameLineWidthElement,
 *    SoTextOrientationElement, SoTextKerningElement, SoTextMarginElement
 * 
 * @SEE_ALSO
 *    SoAnnoText3, SoText3, SoText2, SoAsciiText, SoFont
 * 
 * 
 */
class INVENTOR_API SoTextProperty : public SoNode {

 SO_NODE_HEADER(SoTextProperty) ;
  
 public:
  /**
   * Text property constructor.
   */
  SoTextProperty();

  /**
  * Set the state of the override field.
  * see SoNode::setOverride doc.
  */
  inline virtual void setOverride(const SbBool state)
  { override.setValue(state); }

  /**
   * Returns the state of the override field.
   */
  inline virtual SbBool isOverride() const
  { return override.getValue(); }

  /** 
   * Defines the anti-aliasing factor to apply to the textured text rendering.
   * Default is 1.0.
   * Default value can be changed using the environment variable OIV_TEXT_ALIASING_FACTOR.
   *
   * The anti-aliasing applied transfer functions are:
   * - if aliasingFactor <= 1.0 => Max( (v + f - 1) / f , 0.0)
   * - if aliasingFactor > 1.0 => Min( (v * f) , 1.0)
   *
   * A value of 0 essentially turns off antialiasing. Only pixels that are full opacity will be rendered.
   * Value 1 is the native antialiasing provided by the FreeType library.
   * Values < 1 scale down the opacity of semi-transparent pixels,
   * so lower opacity pixels become completely transparent, effectively
   * making the character glyphs sharper but narrower. Values > 1 scale up
   * the opacity of semi-transparent pixels, effectively making the
   * characters sharper but wider.  Values greater than 5 have little
   * additional visual effect.
   */ 
  SoSFFloat aliasingFactor;

  /** Possible horizontal alignment values. */
  enum AlignmentH {
    /**  
     * Left edges of all strings are aligned. (default)
     */
    LEFT = 0x01,

    /**  
     * Horizontal centers of all strings are aligned.
     */
    CENTER = 0x03,

    /**  
     * Right edges of all strings are aligned.
     */
    RIGHT = 0x02
  };

  /** 
   * Indicates horizontal placement and alignment of strings. 
   * Use the #AlignmentH enum. Default value is LEFT. With LEFT alignment, the left
   * edge of the first line is at the (transformed) origin, and all left edges are
   * aligned. RIGHT alignment is similar. CENTER alignment places the center
   * of the first string at the (transformed) origin, with the centers of all
   * remaining strings aligned under it. 
   * Default value can be changed using the environment variable OIV_TEXT_ALIGNMENTH.
   *
   * Note: This field is only effective when the justification field of a text node is set to INHERITED.
   * When orientation is vertical, horizontal alignment is only applied on the first line.
   */
  SoSFEnum alignmentH;

  /** Possible vertical alignment values. */
  enum AlignmentV {
    /**  
     * Top edges of all strings are aligned.
     */
    TOP = 0x01,

    /**  
     * Vertical centers of all strings are aligned.
     */
    HALF = 0x03,

    /** 
     * All strings are aligned on their baseline. (default)
     */
    BASE = 0x04,

    /** 
     * Bottom edges of all strings are aligned.
     */
    BOTTOM = 0x05
  };

  /** 
   * Indicates vertical placement and alignment of strings. 
   * Use the #AlignmentV enum. Default is BASE. With TOP alignment, the top
   * edge of the first line is at the (transformed) origin, and all top edges are
   * aligned. BOTTOM alignment is similar. HALF alignment places the center
   * of the first string at the (transformed) origin, with the centers of all
   * remaining strings aligned under it.
   * Default value can be changed using the environment variable OIV_TEXT_ALIGNMENTV.
   * When orientation is horizontal, vertical alignment is only applied on the first line.
   */
  SoSFEnum alignmentV;

  /**
   * Defines the width of the line when BACK_FRAME_LINE has been enabled for the current
   * style. The backFrameLineWidth value is pixel size for SoText2 text, and is a 1/40
   * of the current font size for SoText3 text. Default value is 1.0
   * Default value can be changed using the environment variable OIV_TEXT_BACKFRAMELINE_WIDTH.
   *
   * @FIELD_SINCE_OIV 9.2
   */
  SoSFFloat backFrameLineWidth;

  /** 
   * Set this field to TRUE to apply kerning on text rendering. For more information about kerning, 
   * refer to the FreeType documentation located at
   * http://freetype.org/freetype2/docs/glyphs/glyphs-4.html
   * Default value is FALSE.
   */
  SoSFBool kerning ;

  /** Possible text orientation values. */
  enum Orientation {
    /** 
     * Text string is left to right oriented. (default)
     */
    LEFT_TO_RIGHT = 0x01,

    /** 
     * Text string is right to left oriented.
     */
    RIGHT_TO_LEFT = 0x02,

    /** 
     * Text string is top to bottom oriented.
     */
    TOP_TO_BOTTOM = 0x03,

    /** 
     * Text string is bottom to top oriented.
     */
    BOTTOM_TO_TOP = 0x04
  };

  /** 
   * Specifies the text rendering orientation. 
   * Use the #Orientation enum.  Default is LEFT_TO_RIGHT.
   * Default value can be changed using the environment variable OIV_TEXT_ORIENTATION.
   */
  SoSFEnum orientation;

  /** 
   * Defines a quality range of values taken into account during textured text
   * texture computation.
   * The size of the generated textures depends on this range, expressed in terms
   * of DPI, and depends on the SoComplexity::value.  Default is 150 to 300.
   * Default range values can be changed using the environment variables
   * OIV_MIN_TEXTURED_FONT_RES and OIV_MAX_TEXTURED_FONT_RES.
   *
   * @FIELD_SINCE_OIV 8.1
   */
  SoSFVec2i32 textureQualityRange;

  /** TextProperty style values. */
  enum Style
  {
    /** No effects. */
    NONE = SoTextStyleElement::NONE,
    /** UNDERLINE */
    UNDERLINE = SoTextStyleElement::UNDERLINE,
    /** STRIKETHROUGH */
    STRIKETHROUGH = SoTextStyleElement::STRIKETHROUGH,
    /** DOUBLE_STRIKETHROUGH */
    DOUBLE_STRIKETHROUGH = SoTextStyleElement::DOUBLE_STRIKETHROUGH,
    /** BACK_FRAME */
    BACK_FRAME = SoTextStyleElement::BACK_FRAME,
    /** BACK_FRAME_LINE */
    BACK_FRAME_LINE = SoTextStyleElement::BACK_FRAME_LINE,
    /** OVERLINE */
    OVERLINE = SoTextStyleElement::OVERLINE
  };

  /**
   * Specifies zero or more styles to be applied to text.  
   * Use the #Style enum. Default is NONE.
   *
   * Note: UNDERLINE, STRIKETHROUGH and DOUBLE_STRIKETHROUGH are not implemented
   * for vertical oriented text ( ie: #Orientation set to TOP_TO_BOTTOM or BOTTOM_TO_TOP).
   *
   * @FIELD_SINCE_OIV 8.1
   */
  SoSFBitMask style;

  /**
   * Index values for setting colors in the #styleColors field.
   */
  enum StyleColor
  {
    /** UNDERLINE_COLOR */
    UNDERLINE_COLOR,
    /** STRIKETHROUGH_COLOR */
    STRIKETHROUGH_COLOR,
    /** DOUBLE_STRIKETHROUGH_COLOR */
    DOUBLE_STRIKETHROUGH_COLOR,
    /** BACK_FRAME_COLOR */
    BACK_FRAME_COLOR,
    /** BACK_FRAME_LINE_COLOR */
    BACK_FRAME_LINE_COLOR,
    /** OVERLINE_COLOR */
    OVERLINE_COLOR,
    /** NUM_EFFECTS_COLORS */
    NUM_EFFECTS_COLORS
  };

  /** 
   * Specifies the color to use for each style. @BR
   * Use the index values defined in the #StyleColor enum (NOT the #Style enum) to assign the color to the desired
   * style.  For example, assign opaque red to the underline color.
   * \if_cpp
   *   \code
   *   SoTextProperty* node = new SoTextProperty();
   *   pNode->styleColors.set1Value( SoTextProperty::UNDERLINE_COLOR, SbColorRGBA(1,0,0,1) );
   *   \endcode
   * \endif
   * \if_dotnet
   *   \code
   *   // Note the Set1Value method expects an "int" as its first parameter
   *   SoTextProperty node = new SoTextProperty();
   *   node.styleColors.Set1Value( (int)SoTextProperty.StyleColorType.UNDERLINE_COLOR, new SbColorRGBA(1,0,0,1) );
   *   \endcode
   * \endif
   * \if_java
   *   \code
   *   // Note the set1Value method expects an "int" as its first parameter
   *   SoTextProperty node = new SoTextProperty();
   *   node.styleColors.set1Value( SoTextProperty.StyleColorType.UNDERLINE_COLOR.getValue(), new SbColorRGBA(1, 0, 0, 1) );
   *   \endcode
   * \endif
   * Default is (0.8,0.8,0.8,1) = the Open Inventor default diffuse color with alpha=1 (opaque).
   *
   * @FIELD_SINCE_OIV 8.1
   */
  SoMFColorRGBA styleColors;

  /** 
   * Set a margin (extra space) to apply around the text string for the
   * BACK_FRAME and BACK_FRAME_LINE styles.
   * The margin value is specified as a fraction of the font size.
   * Default value is 0.0. 
   * Default value can be changed using the environment variable OIV_TEXT_MARGIN.
   * @FIELD_SINCE_OIV 8.1
   */
  SoSFFloat margin;

  /**
   * This field allows to use the current SoMaterial instead of the value specified by #styleColors
   * for each available style (default gray). For example, if the entry corresponding to UNDERLINE_COLOR
   * is set to TRUE, underline will be rendered using the color inherited from SoMaterial.  This is
   * convenient if (as usual) the underline should be the same color as the text.
   * Use the index values defined in the #StyleColor enum (NOT the #Style enum) to set the value for the
   * desired style (see example code for #styleColors field).
   * @FIELD_SINCE_OIV 9.4
   */  
  SoMFBool styleColorsUseCurrentMaterial;

  /**
   * This field specifies the spacing between individual characters. The value is in font size units 
   * for 2D text (SoText2) and 3D text (SoText3). Default is 0.
   *   - 0 means no extra space with respect to the standard spacing defined in the font.
   *
   *   - A positive value is added to the default spacing (it is not relative or proportional).
   *
   *   - Negative values are ignored.
   */
  SoSFFloat characterSpacing;
  
SoEXTENDER public:
  virtual void  doAction(SoAction *action);
  virtual void  GLRender(SoGLRenderAction *action);
  virtual void  callback(SoCallbackAction *action);
  virtual void  pick(SoPickAction *action);
  virtual void  getBoundingBox(SoGetBoundingBoxAction *action);
  virtual void  getPrimitiveCount(SoGetPrimitiveCountAction *action);

SoINTERNAL public:
  // Init the class
  static void initClass();
  static void exitClass();

  SoSFBool override;

protected:
  virtual ~SoTextProperty();
} ;

/*----------------------------------------------------------------------------*/

#endif // _SO_TEXT_PROPERTY_

/**/


