/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Nick Thompson (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
** Modified by : Paul Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_DATA_SENSOR_
#define  _SO_DATA_SENSOR_

class SoBase;
class SoNode;
class SoPath;
class SoNotList;

#include <Inventor/sensors/SoDelayQueueSensor.h>
#include <Inventor/threads/SbThreadStorage.h>
#include <Inventor/threads/SbThreadSpinlock.h>
#include <Inventor/STL/list>

#ifdef _WIN32
#pragma warning(push)
#pragma warning(disable:4251)
#endif

/**
 * Abstract base class for sensors attached to parts of a scene.
 * 
 * @ingroup sensors
 * 
 * @DESCRIPTION
 *   Data sensors detect changes to scene graph objects (paths, nodes, or fields) and
 *   trigger their \if_dotnet delegate \else callback \endif when the object changes.
 *   The Open Inventor viewer classes, for example, attach an SoNodeSensor to the root of the
 *   application's scene graph in order to know when any part of the scene graph has been modified and a redraw is needed.
 *
 *   Data sensors provide methods that can be called in the
 *   \if_dotnet delegate \else callback \endif to determine exactly which node or field caused the
 *   sensor to be triggered.  However these methods only return valid information if the sensor priority
 *   was explicitly set to zero (default is 100).  Depending on the type of attached object, there are
 *   multiple possible triggers and only some of the get trigger info methods will return useful information.
 *
 *   Priority zero data sensors are triggered immediately after the change.
 *   Normal priority sensors are not triggered until the next time the "delay queue" is processed.
 *   Normally this happens when the viewer / renderArea is not rendering and there are no input events to be processed.
 *
 *   A data sensor will be triggered if its \if_dotnet Schedule() \else schedule() \endif method is
 *   called.  But the trigger node, trigger field, etc. queries will return null.  Generally this method
 *   is only useful for "at some future time" sensors like SoIdleSensor or SoAlarmSensor.
 *
 *   \if_java
 *   Data sensors provide a "final" task that is called when the object the data sensor is attached to is finalized.
 *   This task should not attempt to modify the object in any way. (see setFinalTask(Runnable))
 *   \else
 *   Data sensors provide a delete \if_dotnet delegate \else callback \endif that is called just before the object the
 *   data sensor is attached to is destroyed. The \if_dotnet delegate \else callback \endif should not attempt
 *   to modify the object in any way.
 *   \endif
 *
 * @EXAMPLE
 * Create a node sensor to detect when the camera has been modified.
 * \if_cpp
 *   \code
 *   SoCamera* camera = pViewer->getCamera();
 *   SoNodeSensor* nodeSensor = new SoNodeSensor();
 *   nodeSensor->setFunction(       nodeChangedCB   );
 *   nodeSensor->setDeleteCallback( nodeDestroyedCB );
 *   nodeSensor->setPriority( 0 );
 *   nodeSensor->attach( camera );
 *   \endcode
 * \endif
 * \if_dotnet
 *   \code
 *   SoCamera camera = viewer.GetCamera();
 *   SoNodeSensor nodeSensor = new SoNodeSensor();
 *   nodeSensor.Action         = nodeChangedCB;
 *   nodeSensor.DeleteCallback = nodeDestroyedCB;
 *   nodeSensor.SetPriority( 0 );
 *   nodeSensor.Attach( camera );
 *   \endcode
 * \endif
 * \if_java
 *   \code
 *   SoCamera camera = viewer.getArea().getCamera();
 *   SoNodeSensor nodeSensor = new SoNodeSensor();
 *   nodeSensor.setTask(      new nodeChangedCB(nodeSensor) );
 *   nodeSensor.setFinalTask( new nodeDestroyedCB(nodeSensor) );
 *   nodeSensor.setPriority( 0 );
 *   nodeSensor.attach( camera );
 *   \endcode
 * \endif
 *
 * \if_dotnet Delegate \else Method \endif called when the object is modified.
 * \if_cpp
 *   \code
 *   void nodeChangedCB( void* data, SoSensor* sensor )  
 *   {
 *     SoNodeSensor* nodeSensor = (SoNodeSensor*)sensor;
 *     SoNode* node = nodeSensor->getAttachedNode();
 *     std::cout << "Node Changed: " << node->getTypeId().getName().getString() << std::endl;
 *
 *     if (nodeSensor->getPriority() == 0) {
 *       SoField* field = nodeSensor->getTriggerField();
 *       SbName fieldName;
 *       node->getFieldName( field, fieldName );
 *       std::cout << "     Field: " << fieldName << std::endl;
 *     }
 *   }
 *
 *   void nodeDestroyedCB( void* data, SoSensor* sensor )  
 *   {
 *     SoNodeSensor* nodeSensor = (SoNodeSensor*)sensor;
 *     SoNode* node = nodeSensor->getAttachedNode();
 *     // . . .
 *     std::cout << "Node Destroyed: " << node->getTypeId().getName().getString() << std::endl;
 *   }
 *   \endcode
 * \endif
 * \if_dotnet
 *   \code
 *   public void nodeChangedCB( SoSensor sensor )
 *   {
 *       SoNodeSensor nodeSensor = (SoNodeSensor)sensor;
 *       SoNode node = nodeSensor.GetAttachedNode();
 *       Console.WriteLine("Node Changed: {0}", node.GetType().Name);
 *       if (nodeSensor.GetPriority() == 0)
 *       {
 *           SoField field = nodeSensor.GetTriggerField();
 *           String fieldName;
 *           node.GetFieldName( field, out fieldName );
 *           Console.WriteLine( "     Field: {0}", fieldName );
 *       }
 *   }
 *
 *   public void nodeDestroyedCB(SoSensor sensor)
 *   {
 *       SoNodeSensor nodeSensor = (SoNodeSensor)sensor;
 *       SoNode node = nodeSensor.GetAttachedNode();
 *       Console.WriteLine("Node Destroyed: {0}", node.GetType().Name);
 *   }
 *   \endcode
 * \endif
 * \if_java
 *   \code
 *   class nodeChangedCB implements Runnable {
 *       private SoNodeSensor m_nodeSensor = null;
 *       public nodeChangedCB( SoNodeSensor sensor ) {
 *           m_nodeSensor = sensor;
 *       }
 *       public void run() {
 *           SoNode node = m_nodeSensor.getAttachedNode();
 *           System.out.println( "Node changed: " + node.getClass().getSimpleName() );
 *           if (m_nodeSensor.getPriority() == 0) {
 *               SoField field = m_nodeSensor.getTriggerField();
 *               String fieldName = node.getFieldName( field );
 *               System.out.println( "     Field: " + fieldName );
 *           }
 *       }
 *   }
 *
 *   class nodeDestroyedCB implements Runnable {
 *       private SoNodeSensor m_nodeSensor = null;
 *       public nodeDestroyedCB( SoNodeSensor sensor ) {
 *           m_nodeSensor = sensor;
 *       }
 *       public void run() {
 *           if (m_nodeSensor != null) {
 *               SoNode node = m_nodeSensor.getAttachedNode();
 *               System.out.println( "Node destroyed: " + node.getClass().getSimpleName() );
 *           }
 *       }
 *   }
 *   \endcode
 * \endif
 *
 * @SEE_ALSO
 *    SoNodeSensor,
 *    SoPathSensor,
 *    SoFieldSensor,
 *    SoDelayQueueSensor
 * 
 * 
 */
class INVENTOR_API SoDataSensor : public SoDelayQueueSensor {

 public:
   /** Change type */
  enum ChangeType {
    /** Unspecified */
    UNSPECIFIED,
    /** Group add child */
    GROUP_ADD_CHILD,
    /** Group insert child */
    GROUP_INSERT_CHILD,
    /** Group replace child */
    GROUP_REPLACE_CHILD,
    /** Group remove child */
    GROUP_REMOVE_CHILD,
    /** Group remove all children */
    GROUP_REMOVE_ALL_CHILDREN,
    /** Field multivalue */
    FIELD_MULTIVALUE
  };

  /**
   * Constructor.
   */
  SoDataSensor();
  /**
   * Constructor that takes standard callback function and data.
   * [OIV-WRAPPER-NOT-WRAP]
   */
  SoDataSensor(SoSensorCB *func, void *data);

  // Destructor
#ifndef HIDDEN_FROM_DOC
  virtual ~SoDataSensor();
#endif // HIDDEN_FROM_DOC

  /**
   * Sets a callback that will be called when the object the sensor is sensing is
   * deleted.
   */
  void                setDeleteCallback(SoSensorCB *f, void *data = NULL)
    { deleteFunc = f; deleteData = data; }

  /**
   * If this is a priority 0 data sensor, returns the node that was modified
   * that caused this sensor to trigger. Returns NULL if the sensor was not triggered
   * because a node changed (for example, if schedule() is called on the
   * sensor) or if this sensor is not a priority 0 sensor. Note that because one
   * change to the scene graph may cause multiple nodes or fields to be modified
   * (because of field-to-field connections), the node returned may not be
   * the only one that changed.
   */
  SoNode *            getTriggerNode() const;

  /**
   * If this is a priority 0 data sensor, returns the field that was modified
   * that caused this sensor to trigger. Returns NULL if the sensor was not triggered
   * because a field changed (for example, if schedule() is called on the
   * sensor) or if this sensor is not a priority 0 sensor. Note that because one
   * change to the scene graph may cause multiple nodes or fields to be modified
   * (because of field-to-field connections), the field returned may not be
   * the only one that changed.
   */
  SoField *           getTriggerField() const;
    
  /**
   * If this is a priority 0 data sensor, returns a path to the node that caused this
   * sensor to trigger. Because recreating the path to the node that changed is
   * relatively expensive, @B setTriggerPathFlag(TRUE) @b must be called before the
   * sensor is scheduled. If it is not called, or if the sensor wasn't triggered
   * because a node changed, this returns NULL. NULL is also returned if this is not
   * a priority 0 sensor.
   */
  SoPath *            getTriggerPath() const;

  /**
   * Sets the flag that indicates whether the trigger path
   * (see getTriggerPath()) is available to \if_dotnet delegate \else callback \endif methods. This
   * is FALSE by default. Note that setting this to TRUE will add a
   * little overhead when the sensor is notified.
   */
  void setTriggerPathFlag(SbBool flag)
  { doTrigPath = flag; }

  /**
   * Sets the flag that indicates whether the trigger path fastEdit info
   * (see getTriggerFastEditInfo()) is available to \if_dotnet delegate \else callback \endif methods. This
   * is FALSE by default. Note that setting this to TRUE will add a
   * little overhead when the sensor is notified.
   * @M_SINCE 9.2.3
   */
  void setTriggerFastEditInfoFlag(SbBool flag)
  { doTrigFastEditInfo = flag; }

  /**
   * Queries the flag that indicates whether the trigger path
   * (see getTriggerPath()) is available to \if_dotnet delegates \else callbacks \endif\.
   */
  SbBool getTriggerPathFlag() const
  { return doTrigPath; }

  /**
   * Queries the flag that indicates whether the trigger path fastEdit info
   * (see getTriggerFastEditInfo()) is available to \if_dotnet delegates \else callbacks \endif\.
   * @M_SINCE 9.2.3
   */
  SbBool getTriggerFastEditInfoFlag() const
  { return doTrigFastEditInfo; }

  /**
   * If this is a priority 0 data sensor, returns the type of change that occurred.
   * Returns UNSPECIFIED if the sensor was not triggered by a group children change
   * or a multi-value field change or if this sensor is not a priority 0 sensor.
   * A GROUP_* return value indicates that getTriggerChild and getTriggerChildIndex
   * will return valid data.  A FIELD_* return value indicates that the getTriggerMField*
   * methods will return valid data.
   */
  ChangeType getTriggerType() const;

  /**
   * If this is a priority 0 data sensor, and a change to a group node's
   * children caused this sensor to be triggered (getTriggerType returns
   * GROUP_ADD_CHILD, GROUP_INSERT_CHILD, or GROUP_REPLACE_CHILD), returns
   * the node that was added to the group, 
   * and NULL in all other cases.
   */
  SoNode* getTriggerChild() const;

  /**
   * If this is a priority 0 data sensor, and a change to a group node's
   * children caused this sensor to be triggered (getTriggerType returns
   * GROUP_ADD_CHILD, GROUP_INSERT_CHILD, or GROUP_REPLACE_CHILD), returns
   * the index of the node that was added or removed,
   * and -1 in all other cases.
   */
  int getTriggerChildIndex() const;

  /**
   * If this is a priority 0 data sensor, and a change in the data values 
   * of a multiple field (e.g., SoMFVec3f) caused this sensor to be 
   * triggered, returns the first index of the range of the potentially changed values. 
   * Otherwise, returns -1
   */
  int getTriggerMFieldStartIndex() const;

  /**
   * If this is a priority 0 data sensor, and a change in the data values 
   * of a multiple field (e.g., SoMFVec3f) caused this sensor to be 
   * triggered, returns the size of the range of the potentially changed values. 
   * Otherwise, returns -1
   */
  int getTriggerMFieldNumValues() const;

  /**
   * Returns true if the triggered changes come from a field or node that was
   * below a Separator with a fastEditPolicy field with a value different than OFF.
   * @M_SINCE 9.2.3
   */
  int getTriggerFastEditInfo() const;

  // Override unschedule() to reset trigNode and trigPath.
  virtual void unschedule();

 SoINTERNAL public:
  // Override trigger to reset trigNode and trigPath, if
  // necessary.
  virtual void trigger();

  // Propagates modification notification through an instance. By
  // default, this schedules the sensor to be triggered and saves
  // some information from the notification list for use by the
  // callback function.  Called by SoBase.
  virtual void        notify(SoNotList *list);

  // This is called when the base (path, field, node, whatever) is
  // deleted. All subclasses must implement this to do the right
  // thing.
  virtual void        dyingReference() = 0;

  // delete the list of dataSensor in the pending list
  static void clearPendingDeleteList();

  // add an SoDataSensor to the pending delete list
  static void addToPendingDeleteList(SoDataSensor* dataSensor);

 protected:
  // Invokes the delete callback
  void                invokeDeleteCallback();

 private:
   SoSensorCB* deleteFunc;    // CB and data for when attached SoBase
   void* deleteData;    // is about to be deleted

   class SoDataSensorData
   {
   public:
     SoDataSensorData()
     {
       reset();
     }
     void reset()
     {
       trigNode   = NULL;
       trigField  = NULL;
       trigPath   = NULL;

       trigChildIndex = -1;
       trigChild = NULL;
       trigMFieldStartIndex = -1;
       trigMFieldNumValues = -1;
       m_flag.trigChangeType = UNSPECIFIED;
       m_flag.trigFromFastEdit = 0;
     }
     // usefull for debug
     bool isEmpty() const
     {
       return (trigNode==NULL) && (trigField==NULL) && (trigPath==NULL) && (m_flag.trigChangeType==UNSPECIFIED);
     }

     void setTriggerType(ChangeType changeType)
     {
       //assert(changeType< (1<<31));
       m_flag.trigChangeType = changeType;
     }

     ChangeType getTriggerType() const
     { return (ChangeType)(m_flag.trigChangeType); }

     void setTriggerFromFastEdit(bool flag)
     { m_flag.trigFromFastEdit = flag; }

     bool isTriggerFromFastEdit() const
     { return m_flag.trigFromFastEdit; }

   public:
     // members
     SoNode* trigNode;      // Node that triggered sensor
     SoField* trigField;    // Field that triggered sensor
     SoPath* trigPath;      // Path to trigNode
     SoNode* trigChild;        // changed child (if parent is SoGroup)
     int trigChildIndex;       // index of changed child in group (if parent is SoGroup)
     int trigMFieldStartIndex; // index of the first value which change in the MField
     int trigMFieldNumValues;  // num of values which change in the MField
  private:
     struct {
        /** ChangeType (if parent is SoGroup) */
        unsigned int trigChangeType   : 31;
        /** 1 if the notification come from a field/node that was below a Separator with field FastEditPolicy!=OFF */
        unsigned int trigFromFastEdit : 1;
     } m_flag;
   };

   // Store the current notification info (common to all threads)
   // Doing multiple notification before trigger is managed will result in only the last 
   // notification to be triggered
   // any access read/write to this member should be lock/unlock protected.
   // as we cannot lock/unlock the trigger() method which use this member in readMode,
   // we copy m_notifyData in m_triggerNotifyData (with a specific lock) in order to keep
   // trigger call safe.
   SoDataSensorData m_notifyData;

   SoDataSensorData m_triggerNotifyData;
   SbThreadSpinlock m_triggerNotifyDataMutex;

  bool doTrigPath;          // Whether to compute trigger path
  bool doTrigFastEditInfo;  // Whether to compute fastEditInfo

  // manage pending delete list
  static SbThreadSpinlock s_pendingDeleteListMutex;
  static std::list<SoDataSensor*> s_pendingDeleteList;

};

#ifdef _WIN32
#pragma warning(pop)
#endif

#endif  /* _SO_DATA_SENSOR_ */

