/*=======================================================================
** VSG_COPYRIGHT_TAG
**=======================================================================*/

#ifndef SOTIMERINTERFACE_H
#define SOTIMERINTERFACE_H

#include <Inventor/SbBase.h>
#include <Inventor/misc/SoRef.h>

/**
 * @VSGEXT Interface used by SoSystemTimer to execute a task
 *
 * @ingroup ViewerComponents
 *
 * @DESCRIPTION
 * This object is used by SoSystemTimer to execute 
 * a custom action when the timer is triggered
 *
 * @SEE_ALSO
 * SoSensorHandler, SoSystemTimer
 * [OIV-WRAPPER-CLASS MODULE{INVENTOR}]
 */
class SoSystemTimerTask : public SoRefCounter
{
public:
  /**
  * This method is executed by SoSystemTimer when it is triggered
  * [OIVJAVA-WRAPPER HELPER_BEGIN{onRun()}]
  */
  virtual void run() = 0;
};

/**
 * @VSGEXT Common interface to expose system dependent timers
 *
 * @ingroup ViewerComponents
 *
 * @DESCRIPTION
 *  The timer implementation usually depends on the window toolkit used (Qt, Awt, Swt...).
 *  The purpose of this class is to expose a common interface for system dependent timers.
 *  When the timer is triggered, it must call the run() method of its SoSystemTimerTask (given
 *  by the setTask method).
 *
 * @SEE_ALSO
 *    SoSensorHandler
 *
 * [OIV-WRAPPER-CLASS DERIVABLE]
 */
class INVENTOR_API SoSystemTimer : public SoRefCounter
{
public:
  /** Start the timer. If it is already started, stop it and restart */
  virtual void start() = 0;

  /** Stop the timer */
  virtual void stop() = 0;

  /** Set the timer's timeout in milliseconds */
  virtual void setDelay(int time) = 0;

  /** Return true if the timer is running */
  virtual bool isPending() const = 0;
  
  /** 
   * If set to true, the timer will be triggered periodically at the rate 
   * defined by setDelay 
   */
  virtual void setRepeat(bool flag) = 0;

  /**
   * Sets the task executed when the timer is triggered. When the timer is 
   * is triggered, it calls the run() method of task.
   */
  virtual void setTask(SoSystemTimerTask* task) = 0;  
};

#endif
