/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Nicolas DAGUISE (Oct 2008)
**=======================================================================*/


#ifndef _SB_THREAD_LOCAL_STORAGE_H_
#define _SB_THREAD_LOCAL_STORAGE_H_

#include <Inventor/SbBase.h>

/** @memberof SbThreadLocalStorage */
typedef void SoInitTLSClassCB();
/** @memberof SbThreadLocalStorage */
typedef void SoExitTLSClassCB();

/**
 * Requests allocation of thread local storage.
 *
 * @relates SbThreadLocalStorage
 *
 * This macro should be called within initClass(). The structName
 * argument should be the structure (the type name) defining what size
 * of memory should be allocated for each thread, for this class
 */
#define SB_THREAD_INIT_CLASS(_className_,_structName_) \
  SbThreadLocalStorage::createStorage(_className_::MT_Id, sizeof(struct _className_::_structName_), _className_::initTLSClass, _className_::exitTLSClass, OIV_FUNCTION );

/**
 * Returns true if a TLS storage has been allocated for the given classThreadId.
 *
 * @relates SbThreadLocalStorage
 *
 * This MACRO is used in exitTLSClass() implementation to avoid calling getStorage()
 * which will create the TLS if not found, which is useless as the purpose of exitTLSClass()
 * is to delete associated resources.
 */
#define SB_THREAD_IS_INITIALIZED(_className_) \
  SbThreadLocalStorage::isInitialized(_className_::MT_Id)

/**
 * Requests deallocation of thread local storage.
 *
 * @relates SbThreadLocalStorage
 *
 * This macro should be called within exitClass(). The structName
 * argument should be the structure (the type name) defining what size
 * of memory should be allocated for each thread, for this class
 */
#define SB_THREAD_EXIT_CLASS(_className_) \
  SbThreadLocalStorage::deleteStorage( _className_::MT_Id, OIV_FUNCTION );

/**
 * Gets direct access to current thread storage inside variable.
 *
 * @relates SbThreadLocalStorage
 */
#define GET_THREAD_LOCAL_VAR(_className_, _structName_, _varName_) \
  ((static_cast<struct _className_::_structName_ *> (SbThreadLocalStorage::getStorage(_className_::MT_Id)))->_varName_)

/**
 * Returns pointer to current thread storage for defined class.
 *
 * @relates SbThreadLocalStorage
 */
#define GET_THREAD_LOCAL_STORAGE(_className_) \
  (static_cast<void *> (SbThreadLocalStorage::getStorage(_className_::MT_Id)))

/**
 * Defines headers for required member variables in thread storage management.
 *
 * @relates SbThreadLocalStorage
 */
#define SB_THREAD_TLS_HEADER() \
SoINTERNAL public: \
  /** TLS storage unique id */ \
  static size_t MT_Id; \
  /** TLS storage initialization method */ \
  static void initTLSClass(); \
  /** TLS storage cleanup method */ \
  static void exitTLSClass()


/**
 * Defines source for required member variables in thread storage management.
 *
 * @relates SbThreadLocalStorage
 */
#define SB_THREAD_TLS_SOURCE( _className_ ) \
  /** Unique Id to reference access TLS data associated to this class through getStorage method */ \
  size_t _className_::MT_Id = -1

/**
 * Compatiblity define SoNodeTLS to SbThreadLocalStorage.
 *
 * @relates SbThreadLocalStorage
 */
#ifndef SoNodeTLS
#define SoNodeTLS SbThreadLocalStorage
#endif

/**
 * @VSGEXT Thread Local Storage (TLS) class.
 *
 * @ingroup Threads
 *
 * @DESCRIPTION
 *   This class allows to create thread local data storage.
 *   This allow to manage per thread static global variables
 *   in multithread mode.
 *
 * Example on how to use the Thread Local Storage (TLS):
 *
 * In this example we move static variables s_myStaticVar1 and 
 * s_myStaticVar2 of the class MyClass to Thread Local Storage,
 *
 * - Original myClass implementation : @BR
 *   \code
 *       class myClass
 *       {
 *          
 *          static void initClass();
 *          static void exitClass();
 *          ... 
 *          static int    s_myStaticVar1;
 *          static float* s_myStaticVar2;
 *          ...
 *       }
 *
 *       void myClass::initClass()
 *       {
 *         s_myStaticVar1 = 32;
 *         s_myStaticVar2 = new float[32];
 *       }
 *
 *       void myClass::exitClass()
 *       {
 *         s_myStaticVar1 = 0;
 *         delete [] s_myStaticVar2;
 *       }
 *   \endcode
 * 
 * - Modified myClass implementation to support use TLS : @BR
 *   \code
 *       class myClass
 *       {
 *          
 *          static void initClass();
 *          static void exitClass();
 *
 *          // Declare necessary members and methods for TLS usage
 *          SB_THREAD_TLS_HEADER();
 *          ...
 *          // structure to handle per thread static variables
 *          struct MTstruct {
 *            int    s_myStaticVar1;
 *            float* s_myStaticVar2;
 *          }
 *          ...
 *       }
 *
 *       // Properly initilize members and methods for TLS usage
 *       SB_THREAD_TLS_SOURCE( myClass );
 *
 *       void myClass::initClass()
 *       {
 *         // register and reserve space for TLS
 *         SB_THREAD_INIT_CLASS(myClass, MTstruct);
 *       }
 *
 *       void myClass::exitClass()
 *       {
 *       }
 *
 *       void myClass::initTLSClass()
 *       {
 *         // get access to TLS and init/allocate variables once per thread
 *         struct myClass::MTstruct * mtstruct = (struct myClass::MTstruct *)GET_THREAD_LOCAL_STORAGE(myClass) ;
 *         mtstruct->s_myStaticVar1 = 32;
 *         mtstruct->s_myStaticVar2 = new float[32];
 *       }
 *
 *       void myClass::exitTLSClass()
 *       {
 *         if (SbThreadLocaStorage::isInitialized(myClass::MT_id))
 *         {
 *           // get access to TLS and deallocate variables once per thread
 *           struct myClass::MTstruct * mtstruct = (struct myClass::MTstruct *)GET_THREAD_LOCAL_STORAGE(myClass) ;
 *           delete [] mtstruct->s_myStaticVar2;
 *         }
 *       }
 *   \endcode
 *
 * - Each time a variable in Thread Local Storage is used, the structure
 *   containing this variable needs to be fetched using the following macro :
 *   \code
 *       struct myClass::MTstruct * mtstruct = (struct myClass::MTstruct *) GET_THREAD_LOCAL_STORAGE(myClass) ;
 *   \endcode
 *   The variables can then be used as, for example: mtstruct->variable1. @BR
 *   Another macro exists to bring back one single variable of the structure : @BR
 *   \code
 *       GET_THREAD_LOCAL_VAR(myClass, MTstruct, variable1)
 *   \endcode
 *
 * @SEE_ALSO
 *    SbThread,
 *    SbThreadAutoLock,
 *    SbThreadAutoReadLock,
 *    SbThreadAutoWriteLock,
 *    SbThreadBarrier,
 *    SbThreadRWMutex,
 *    SbThreadSignal
 *
 * [OIV-WRAPPER-CLASS NO_WRAP]
 */
class INVENTORBASE_API SbThreadLocalStorage
{
 public:
  /**
   * Creates or updates the local storage for the current thread.
   * Requests that 'size' bytes of storage should be allocated
   * for all threads for the calling class.
   * Returns a unique classThreadId to be used by getStorage method to access
   * the storage.
   * Used by SB_THREAD_INIT_CLASS
   */
  static void createStorage(size_t& classThreadId, const size_t byteSize, SoInitTLSClassCB* initFunc=NULL, SoExitTLSClassCB* exitFunc=NULL, const char* funcName=NULL);

  /**
   * Deletes the local storage for all threads.
   * Used by SB_THREAD_EXIT_CLASS macro.
   */
  static void deleteStorage( size_t& classThreadId, const char* funcName=NULL );

  /**
   * Returns true if a TLS storage has already been allocated for the given classThreadId.
   * The method is used in exitTLSClass() implementation to avoid calling getStorage()
   * which will create the TLS if not found, which is useless as the purpose of exitTLSClass()
   * is to delete associated resources.
   * Used by SB_THREAD_IS_INITIALIZED macro.
   */
  static bool isInitialized(const size_t classThreadId);

  /**
   * Returns a pointer on the storage for the given classThreadId.
   * This storage is guaranteed to be local to the current thread.
   * The size of the storage is defined by what was requested
   * by createStorage, unless the pointer returned is NULL.
   *
   * If the storage has not been created, then it will create it if possible.
   */
  static void *getStorage(const size_t classThreadId);

SoEXTENDER public:
  /**
   * Creates or updates the local storage for the current thread. @BR
   * This is different from creating the thread key, since here we
   * access the local storage that is attached to that key for the
   * current thread. This method can be called several times for the
   * same thread. It should be called after initClass and after an
   * OIV class has asked for storage creation (see createStorage).
   * createStorage can be called several times before calling
   * initThreadLocalStorage, but the storage shouldn't be accessed
   * (via getStorage) after createStorage until a call to
   * initThreadLocalStorage.
   *
   * Note that it is calls internally when needed (on first usage).
   * Application should not have to call it unless it wants to force tls storage creation earlier than on first use.
   * 
   */
  static void initThreadLocalStorage();

  /**
   * Frees all memory for the local storage associated with the
   * current thread. initThreadLocalStorage can be called after
   * closeThreadLocalStorage to reinitialize the storage.
   *
   * Note that it is called internally when needed (on thread exit).
   * Application should not have to call it unless it wants to force tls storage deletion earlier than on thread exit.
   */
  static void closeThreadLocalStorage();

SoINTERNAL public:
  /**
   * Initializes the SbThreadLocalStorage class.
   * Creates the thread key that will be used by all threads
   * accessing the thread local storages
   */
  static void initClass();

  /**
   * Finalizes the SbThreadLocalStorage class.
   * Deletes the thread key associated with the storage.
   */
  static void exitClass();

  /**
   * Return the TLS struct StructName of the class ClassName
   * eg: getTlsStruct<MTStruct, MyClass>(this);
   */
  template<typename StructName, typename ClassName>
  static StructName*
  getStruct()
  {
    return static_cast<StructName*>(SbThreadLocalStorage::getStorage(ClassName::MT_Id));
  }

  /**
   * Return the TLS struct StructName of the class ClassName
   * eg: getTlsStruct<MTStruct>(this);
   */
  template<typename StructName, typename ClassName>
  static StructName*
  getStruct(const ClassName*)
  {
    return getStruct<StructName, ClassName>();
  }

  /**
   * Return TRUE if ThreadStorage is initialized for the current Thread.
   */
  static bool isInitialized();

};

#endif //_SB_THREAD_LOCAL_STORAGE_H_

/**/


