/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (Feb 2008)
**=======================================================================*/

#ifndef _SB_THREAD_SPINLOCK_H_
#define _SB_THREAD_SPINLOCK_H_

#include <Inventor/SbBase.h>
#include <Inventor/threads/SbThread.h>
#include <Inventor/threads/SbThreadMutex.h>

/**
 * @VSGEXT Portable spinlock class.
 * 
 * @ingroup Threads
 * 
 * @DESCRIPTION
 *   This class provides a portable "spinlock". It is implemented using the pthreads
 *   API on UNIX/Linux and the Win32 API on Microsoft Windows.
 *   
 *   A spinlock can be used just like a mutex to protect access to critical data 
 *   or code that is used by multiple threads.  In some cases a spinlock may be
 *   more efficient than a mutex.  But note that this class may be implemented using
 *   a mutex if there is no native support, which is currently the case on UNIX/Linux.
 *   
 *   It is not necessary to use this class in order to use multiple threads with Open
 *   Inventor. It is provided only as a convenience. (However, note that you should
 *   use pthreads on UNIX/Linux and Win32 on Windows to be certain of compatibility
 *   with Open Inventor.)
 * 
 * @SEE_ALSO
 *    SbThread,
 *    SbThreadAutoLock,
 *    SbThreadBarrier,
 *    SbThreadMutex
 * 
 * [OIV-WRAPPER-CLASS NO_WRAP]
 */
class INVENTORBASE_API SbThreadSpinlock
{
 public:
  /**
   * Create a spinlock.
   */
  SbThreadSpinlock() ;
  /**
   * Destructor.
   */
  ~SbThreadSpinlock() ;

  /**
   * Request exclusive access for this spinlock.
   */
  void lock() ;

  /**
   * Release exclusive access.
   */
  void unlock() ;

private:
#if defined(_WIN32)
  LONG volatile m_spinLock;
  SbThreadId_t m_curThread;
  int m_enterCount;
#else
  SbThreadMutex m_spinLock;
#endif
};

/**
 * @VSGEXT Provides reliable unlocking of a spinlock.
 *
 * @ingroup Threads
 *
 * @DESCRIPTION
 *   This class provides reliable unlocking of a spinlock.
 *
 *   An SbThreadMutex object must be created separately. The address of the spinlock
 *   object is passed to the constructor, which locks the spinlock and saves a reference
 *   to the spinlock object. The autolock object should be created on the stack, so each
 *   thread will have its own autolock. The destructor automatically unlocks the
 *   spinlock.
 *
 *   Because the autolock object is created on the stack, it avoids the need to
 *   explicitly unlock the spinlock at every point of exit from the function (a common
 *   source of errors). The spinlock will be unlocked when the autolock goes out of
 *   scope.
 *
 *   Example:
 *   \code
 *      // myClass contains a Mutex object named m_mutex
 *      void myClass::someFunction()
 *      {
 *         // m_spinlock is a ptr to an existing SbThreadSpinlock object
 *         SbThreadAutoSpinlock dummyName( m_spinlock );
 *         ... other code ...
 *      } // spinlock automatically unlocked
 *   \endcode
 *
 * @SEE_ALSO
 *    SbThread,
 *    SbThreadSpinlock
 *
 * [OIV-WRAPPER-CLASS NO_WRAP]
 */
class INVENTORBASE_API SbThreadAutoSpinlock {
 public:
  /**
   * Creates a spinlock "autolock" object and automatically calls the mutex lock method.
   * The spinlock will be automatically unlocked when the autolock object is destroyed.
   */
  SbThreadAutoSpinlock(SbThreadSpinlock *spinlock)
  {
    m_spinlock = spinlock;
    if ( m_spinlock )
      m_spinlock->lock();
  }

  SbThreadAutoSpinlock( SbThreadSpinlock &spinlock )
  {
    m_spinlock = &spinlock;
    m_spinlock->lock();
  }

#ifndef HIDDEN_FROM_DOC
  ~SbThreadAutoSpinlock()
  {
    if ( m_spinlock )
      m_spinlock->unlock();
  }
#endif // HIDDEN_FROM_DOC

 private:
  // Default constructor not accessible
  SbThreadAutoSpinlock()
  :m_spinlock(NULL)
  {}

  // Copy constructor not accessible
  SbThreadAutoSpinlock(const SbThreadAutoSpinlock&)
  :m_spinlock(NULL)
  {}

  SbThreadSpinlock *m_spinlock;
};

#endif // _SB_THREAD_SPINLOCK_H_


