/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : ... (MMM yyyy)
** Modified by : ... (MMM yyyy)
**=======================================================================*/

#ifndef  _SO_TOUCH_MANAGER
#define  _SO_TOUCH_MANAGER

#include <Inventor/gestures/recognizers/SoGestureRecognizer.h>
#include <Inventor/touch/events/SoTouchEvent.h>
#include <Inventor/SoSceneManager.h>

class SoTouchEvent;
class SoScaleGestureRecognizer;
class SoRotateGestureRecognizer;
class SoDoubleTapGestureRecognizer;
class SoLongTapGestureRecognizer;

/** 
 *
 * Class for managing events from touch screen input devices.
 * 
 * @ingroup touch
 * 
 * @DESCRIPTION
 *   This is the base class for managing events from touch screen input devices. 
 *   A touch manager has a SoTouchEvent table and a finger ids table which are updated each time a finger moves.
 *   A touch manager has a list of gesture recognizers that generate events when the gesture is recognized.
 *   By default, the list is empty and recognizers must be added to generate the corresponding events. 
 *   The addDefaultRecognizers() method is provided for convenience.
 *
 *   Note that all registered gesture recognizers are active simultaneously. During a gesture event,
 *   events may generated for a different gesture. The application is responsible for enforcing a
 *   "one gesture at a time" policy if appropriate.
 *
 *   @B Note@b: Unlike mouse and keyboard events, the application must explicitly register a touch
 *   screen device (e.g. SoWinTouchScreen) in order to get touch events.
 *
 * @EXAMPLE
 *   Get the touch manager from the touch device to register gesture recognizers.
 *   \if_cpp
 *     \code
 *       // Create recognizers
 *       SoScaleGestureRecognizer  scaleRecognizer;
 *       SoRotateGestureRecognizer rotateRecognizer;
 *
 *       // Register recognizers
 *       SoWinTouchScreen touchScreenDevice( myWindow );
 *       touchScreenDevice.getTouchManager()->addRecognizer( &scaleRecognizer );
 *       touchScreenDevice.getTouchManager()->addRecognizer( &rotateRecognizer );
 *     \endcode
 *   \endif
 *   \if_dotnet
 *     \code
 *       // Create recognizers
 *       SoScaleGestureRecognizer  scaleRecognizer  = new SoScaleGestureRecognizer();
 *       SoRotateGestureRecognizer rotateRecognizer = new SoRotateGestureRecognizer();
 *
 *       // Register recognizers
 *       SoWinTouchScreen touchScreenDevice = new SoWinTouchScreen(this);
 *       touchScreenDevice.GetTouchManager().AddRecognizer(scaleRecognizer);
 *       touchScreenDevice.GetTouchManager().AddRecognizer(rotateRecognizer);
 *     \endcode
 *   \endif
 *   \if_java
 *     Not available
 *   \endif
 *
 * @SEE_ALSO
 *    SoQtTouchScreen,
 *    SoWinTouchScreen,
 *    SoTouchEvent,
 *    SoGestureEvent,
 *    SoGestureRecognizer
 */
class INVENTOR_API SoTouchManager 
{

public :

  SoTouchManager();

  virtual ~SoTouchManager();

  /**
   *  Get the corresponding event for the finger id, NULL if id is incorrect.
   */
  const SoTouchEvent* getEventById(unsigned long id) const; 

  /**
   *  Get the corresponding event for the index, NULL if index is incorrect. 
   *  Use getFingerNumber() to know the number of fingers touching the screen.
   */
  const SoTouchEvent* getEventbyIndex(int index) const;

  /**
   *  Get the table with ids of all fingers touching the screen.
   */
  const std::vector<unsigned long>& getAllFingerIds() const;

  /**
   *  Get the current number of fingers touching the screen.
   */
  int getFingerNumber() const;

  /**
   *  Get the maximum number of simultaneous fingers supported by the device.
   */
  int getMaxSimultaneousFingers() const;

  /**
   *  Set the maximum number of simultaneous fingers supported by the device.
   */
  void setMaxSimultaneousFingers(int fingers);

  /**
   *  Set the detection threshold (0.0001 pixels by default).
   *  This is the minimum difference between previous and current event positions
   *  to consider that the finger has moved. 
   */
   void setThreshold(float t);

  /**
   *  Get the detection threshold (in pixels).
   */
  float getThreshold() const; 

  /**
   *  Returns true if the touch screen device is available, false otherwise.
   */
  bool isTouchScreenAvailable();

  /**
   *  Set the touch screen availability.
   */
  void setTouchScreenAvailability(bool available);

  /**
   *  Remove a gesture recognizer from the recognizer list.
   * [OIV-WRAPPER CUSTOM_CODE]
   */
  void removeRecognizer(SoGestureRecognizer* gesture);

  /**
   *  Add a gesture recognizer to the recognizer list.
   * [OIV-WRAPPER CUSTOM_CODE]
   */
  void addRecognizer(SoGestureRecognizer* recognizer);

  /**
   *  Add a default list of recognizers to the recognizer list.
   *  The following recognizers are added:
   *    - SoDoubleTapGestureRecognizer
   *    - SoLongTapGestureRecognizer
   *    - SoScaleGestureRecognizer
   *    - SoRotateGestureRecognizer
   */
  void addDefaultRecognizers();

  /**
   *  Remove the default recognizers from the recognizer list.
   *  It removes all the recognizers previously added by addDefaultRecognizers().
   */
  void removeDefaultRecognizers();

  /**
   * Returns the gesture recognizers list.
   */
  const std::vector<SoGestureRecognizer*>& getRecognizerList() const;

  /**
   * Translate the given touch down event into a list of SoEvent. 
   * This list may contain some recognized SoGestureEvent.
   * @return this list.
   * @param fingerId id of the finger used.
   * @param fingerPos position of the finger in pixels.
   * @param time timestamp of the touch event in seconds.
   */
  const std::vector<const SoEvent*>& getTouchDownEvent(unsigned long fingerId, const SbVec2f& fingerPos, const SbTime& time);

  /**
   * Translate the given touch up event into a list of SoEvent. 
   * This list may contain some recognized SoGestureEvent.
   * @return this list.
   * @param fingerId id of the finger used.
   * @param fingerPos position of the finger in pixels.
   * @param time timestamp of the touch event in seconds.
   */
  const std::vector<const SoEvent*>& getTouchUpEvent(unsigned long fingerId, const SbVec2f& fingerPos, const SbTime& time);

  /**
   * Translate the given touch move event into a list of SoEvent. 
   * This list may contain some recognized SoGestureEvent.
   * @return this list.
   * @param fingerId id of the finger used.
   * @param fingerPos position of the finger in pixels.
   * @param time timestamp of the touch event in seconds.
   */
  const std::vector<const SoEvent*>& getTouchMoveEvent(unsigned long fingerId, const SbVec2f& fingerPos, const SbTime& time);

#if SoDEPRECATED_BEGIN(9500)
  // The scene manager argument is useless. The process event should be now done by using the 
  // class SoRenderAreaCore.
  SoDEPRECATED_METHOD(9500,"Scene manager argument is useless.")
  SoTouchManager(SoSceneManager* sM);

  /**
   * Update finger's id of the event identified by id and process it with the scene manager. 
   * Events are also analyzed by every recognizer in the recognizers list.
   * Position is in pixels and time is in seconds.
   */  
  SoDEPRECATED_METHOD(9500,"Use getTouchDownEvent, getTouchUpEvent or getTouchMoveEvent instead. Process the events by using SoRenderAreaCore instead.")
  SoTouchEvent* updateAndProcessTouchEvent(unsigned long id, SoTouchEvent::State state, SbVec2f position, SbTime time);

#endif /** @DEPRECATED_END */

private :
  void init();

  //Replace event indexed by i with j
  void replaceEvent(int i, int j);

  const std::vector<const SoEvent*>& getTouchEvent(unsigned long fingerId, SoTouchEvent::State state, const SbVec2f& fingerPos, const SbTime& time);

  // Events table
  std::vector<SoTouchEvent> m_touchEvent;

  // Recognizers list
  std::vector<SoGestureRecognizer*> m_recognizerList;

  // default recognizers
  SoScaleGestureRecognizer* m_scaleRecognizer;
  SoRotateGestureRecognizer* m_rotateRecognizer;
  SoDoubleTapGestureRecognizer* m_doubleTapRecognizer;
  SoLongTapGestureRecognizer* m_longTapRecognizer;

  bool m_touchScreenAvailable;

  unsigned int m_supportedFingersNumber;
  float m_threshold;
  // Number of fingers touching the screen
  int m_fingerNumber;
  // ID of all fingers touching the screen
  std::vector<unsigned long> m_allFingers;
  // Notify if the last event state was UP
  bool m_fingerWasUp;
  // Index of the event
  int m_eventIndex;
  
  int m_maxSimultaneousFingers;

  std::vector<const SoEvent*> m_soeventlist;

  // Deprecated member: process event should be done by using SoRenderAreaCore
  // Kept only for compat as it is used by the deprecated method updateAndProcessTouchEvent
  SoSceneManager* m_sceneManager; 
};

#endif /* _SO_TOUCH_MANAGER */


