/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : ... (MMM yyyy)
** Modified by : ... (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/

#ifndef  _SO_TOUCH_EVENT_
#define  _SO_TOUCH_EVENT_

#include <Inventor/SbBasic.h>
#include <Inventor/events/SoSubEvent.h>
#include <Inventor/SbTime.h>

class SoTouchManager;

/** 
 *
 * Base class for touch events.
 * 
 * @ingroup touch
 * 
 * @DESCRIPTION
 *   SoTouchEvent is the base class for touch events in the Open Inventor event model.
 *   A touch event typically represents a user finger action on a touch screen input device, 
 *   such as putting a finger down on the screen. SoTouchEvent is an SoEvent with touch information
 *   including the  displacement and speed of the finger and the state (UP, DOWN or MOVE).
 *
 *   The SoTouchEvent::getTouchManager() method can be use to retrieve the corresponding touch device
 *   currently used and information about other fingers touching the screen at the same time.
 *
 *   Open Inventor also supports gesture events (see SoGestureEvent) such as pinch and rotate.
 *   Gesture events are generated by gesture recognizers (see SoGestureRecognizer) which analyze
 *   a series of touch events.
 *
 *   @B Note@b: Unlike mouse and keyboard events, the application must explicitly register a touch
 *   screen device (SoWinTouchScreen, SoQtTouchScreen, etc) in order to get touch events.
 *
 * @SEE_ALSO
 *    SoEvent,
 *    SoButtonEvent,
 *    SoMouseButtonEvent,
 *    SoKeyboardEvent,
 *    SoLocation2Event,
 *    SoMotion3Event,
 *    SoSpaceballButtonEvent,
 *    SoTouchManager
 * 
 */
class INVENTOR_API SoTouchEvent : public SoEvent 
{

  SO_EVENT_HEADER();
    
public:

  /** The different states a touch event can have */
  enum State {
    /**
     *  Unknown state
     */
    UNKNOWN,
    /**
     *  Up state : The finger is off the screen
     */
    UP, 
    /**
     *  Down state : The finger is on the screen
     */
    DOWN,
    /**
     *  Move state : The finger is moving on the screen
     */
    MOVE
  };

  /**
   *  Constructor.
   */
  SoTouchEvent();

  #ifndef HIDDEN_FROM_DOC
  virtual ~SoTouchEvent();
  #endif // HIDDEN_FROM_DOC

  /**
   *  Gets the touch event state (Up, Down, Move or Unknown).
   *  @useenum{State}.
   */
  State getState() const;

  /**
   *  Gets the id of the finger triggering this event.
   */
  unsigned long getFingerId() const;
	
  /**
   * Gets the elapsed time since previous event, triggered by the same finger, in seconds. 
   * If the current event is a DOWN, the function returns 0.
   */
  SbTime getElapsedTimeSincePreviousEvent() const;
	
  /**
   *  Gets the time of first contact in seconds of the finger identified by getFingerId().
   *  This date is set when the DOWN event occurs.
   */
  SbTime getFirstContactTime() const;

  /**
   *  Gets the elapsed time between the first contact and previous event, for the finger identified by getFingerId(), in seconds.
   */
  SbTime getElapsedTimeUntilPreviousEvent() const;

  /**
   *  Gets the time of previous event of the finger identified by getFingerId(). 
   *  If current event is a DOWN, the function returns the current time.
   */
  SbTime getPreviousEventTime() const;

  /**
   *  Gets the speed (in pixels per second) between 
   *  the previous event and the current one.
   */
  SbVec2f getSpeed() const;
	
  /**
   *  Gets the previous position of the same finger (in pixels). If current event is a DOWN, the 
   *  function returns the current position.
   */
  SbVec2f getPreviousPosition() const;
	
  /**
   *  Gets the deplacement vector (in pixels)
   *  between the previous event and the current one.
   */
  SbVec2f getDisplacement() const;

  /**
   *  Gets the acceleration vector (in pixels per second squared) 
   *  between the previous event and the current one.
   */
  SbVec2f getAcceleration() const;

  /**
   *  Gets the previous speed (in pixels per second).
   */
  SbVec2f getPreviousSpeed() const;

  /**
   *  Gets the corresponding TouchManager for this event.
   */
  SoTouchManager* getTouchManager() const;

  /**
   *  Sets the TouchManager for this event.
   */
  void setTouchManager(SoTouchManager* manager);

  /**
   *  Gets the first position (in pixels). This position is set when the DOWN event occurs.
   */
  SbVec2f getFirstPosition() const;

  /**
   * Sets the state of the touch event.
   */
  void setState(SoTouchEvent::State b);

  /**
   *  Sets the id of the finger triggering this event.
   */
  void setFingerId(unsigned long idFinger);

  /**
   *  Sets the current position (in pixels) 
   *  and the displacement vector.
   */
  virtual void setPosition(const SbVec2f &p);
  virtual void setPosition(const SbVec2s &p);

  /**
   *  Sets the time at which the event occurred.  
   */
  virtual void setTime(SbTime t);

  /**
   *  Sets the previous speed in pixels per second.
   */
  void setPreviousSpeed();

  /**
   *  Sets the first position (in pixels).
   */
  void setFirstPosition(SbVec2f);

SoINTERNAL public:
  static void initClass();
  static void exitClass();
  
private:
  // Id of the finger launching this event
  unsigned long m_fingerId; 
  // Up, Down, Move or Unknown.
  State m_state; 
  // The difference between the event position and the previous event position.
  SbVec2f m_displacement;
  // The time derivative of instant displacement 
  SbVec2f m_previousSpeed;
  // The time elapsed since the first date contact and the previous event
  SbTime m_elapsedTime;
  // Date of first contact in seconds
  SbTime m_firstContactTime; 
  // First position
  SbVec2f m_firstPosition;

  SoTouchManager* m_touchManager;
};

#endif /* _SO_TOUCH_EVENT_ */


